/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2010 University of Bonn. All rights reserved.
 * Please see the LICENSE file or "Copyright notice" in builder.cpp for details.
 */

/*
 * WorkerPoolUnitTest.cpp
 *
 *  Created on: Feb 28, 2012
 *      Author: heber
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <cppunit/CompilerOutputter.h>
#include <cppunit/extensions/TestFactoryRegistry.h>
#include <cppunit/ui/text/TestRunner.h>

#include "WorkerPoolUnitTest.hpp"

#include <vector>

#include "CodePatterns/Assert.hpp"
#include "CodePatterns/Observer/Channels.hpp"
#include "Pool/WorkerPool.hpp"
#include "WorkerAddress.hpp"


#ifdef HAVE_TESTRUNNER
#include "UnitTestMain.hpp"
#endif /*HAVE_TESTRUNNER*/

/********************************************** Test classes **************************************/

#include "stubs/FragmentJobStub.hpp"
#include "unittests/stubs/ObserverStub.hpp"

// Registers the fixture into the 'registry'
CPPUNIT_TEST_SUITE_REGISTRATION( WorkerPoolTest );


void WorkerPoolTest::setUp()
{
  // Throw assertions
  ASSERT_DO(Assert::Throw);

  pool = new WorkerPool();
  addobserver = new NotificationObserver(pool->getChannel(WorkerPool::WorkerAdded));
  removeobserver = new NotificationObserver(pool->getChannel(WorkerPool::WorkerRemoved));
  idleobserver = new NotificationObserver(pool->getChannel(WorkerPool::WorkerIdle));

  // and sign on
  pool->signOn(addobserver, WorkerPool::WorkerAdded);
  pool->signOn(removeobserver, WorkerPool::WorkerRemoved);
  pool->signOn(idleobserver, WorkerPool::WorkerIdle);
}

void WorkerPoolTest::tearDown()
{
  pool->signOff(addobserver, WorkerPool::WorkerAdded);
  pool->signOff(removeobserver, WorkerPool::WorkerRemoved);
  pool->signOff(idleobserver, WorkerPool::WorkerIdle);

  delete removeobserver;
  delete addobserver;
  delete idleobserver;
  delete pool;
}

/** UnitTest for working add/removeWorker
 */
void WorkerPoolTest::WorkerTest()
{
  WorkerAddress address("service", "port");
  WorkerAddress otheraddress("otherservice", "otherport");

  // add worker
  CPPUNIT_ASSERT( pool->addWorker(address) );
  CPPUNIT_ASSERT_EQUAL( (size_t)1, pool->idle_queue.size() );
  CPPUNIT_ASSERT_EQUAL( (size_t)0, pool->busy_queue.size() );
  CPPUNIT_ASSERT_EQUAL( (size_t)1, pool->pool.size() );
  CPPUNIT_ASSERT( addobserver->wasNotified );
  CPPUNIT_ASSERT( !removeobserver->wasNotified );
  CPPUNIT_ASSERT( idleobserver->wasNotified );
  addobserver->wasNotified = false;
  idleobserver->wasNotified = false;

  // remove non-present worker
  std::cout << "The following error message is intended and does not indicate a failure." << std::endl;
  CPPUNIT_ASSERT( !pool->removeWorker(otheraddress) );
  CPPUNIT_ASSERT( !addobserver->wasNotified );
  CPPUNIT_ASSERT( !removeobserver->wasNotified );
  CPPUNIT_ASSERT( !idleobserver->wasNotified );

  // check for presence
  CPPUNIT_ASSERT( pool->presentInPool(address) );
  CPPUNIT_ASSERT( !pool->presentInPool(otheraddress) );

  // remove worker
  CPPUNIT_ASSERT( pool->removeWorker(address) );
  CPPUNIT_ASSERT_EQUAL( (size_t)0, pool->idle_queue.size() );
  CPPUNIT_ASSERT_EQUAL( (size_t)0, pool->busy_queue.size() );
  CPPUNIT_ASSERT_EQUAL( (size_t)0, pool->pool.size() );
  CPPUNIT_ASSERT( !addobserver->wasNotified );
  CPPUNIT_ASSERT( removeobserver->wasNotified );
  CPPUNIT_ASSERT( !idleobserver->wasNotified );
  removeobserver->wasNotified = false;
}

/** UnitTest for working (un)markWorkerBusy
 */
void WorkerPoolTest::markBusyTest()
{
  WorkerAddress address("service", "port");
  CPPUNIT_ASSERT( pool->addWorker(address) );

  // check that not busy
  CPPUNIT_ASSERT( !pool->isWorkerBusy(address) );
  CPPUNIT_ASSERT_EQUAL( (size_t)1, pool->idle_queue.size() );
  CPPUNIT_ASSERT_EQUAL( (size_t)0, pool->busy_queue.size() );
  CPPUNIT_ASSERT_EQUAL( (size_t)1, pool->pool.size() );
  CPPUNIT_ASSERT( addobserver->wasNotified );
  CPPUNIT_ASSERT( !removeobserver->wasNotified );
  CPPUNIT_ASSERT( idleobserver->wasNotified );
  addobserver->wasNotified = false;
  idleobserver->wasNotified = false;

  // unmark and check
  pool->unmarkWorkerBusy(address);
  CPPUNIT_ASSERT( !pool->isWorkerBusy(address) );
  CPPUNIT_ASSERT_EQUAL( (size_t)1, pool->idle_queue.size() );
  CPPUNIT_ASSERT_EQUAL( (size_t)0, pool->busy_queue.size() );

  // mark as busy and check
  const WorkerAddress otheraddress = pool->getNextIdleWorker();
  CPPUNIT_ASSERT( pool->isWorkerBusy(address) );
  CPPUNIT_ASSERT_EQUAL( (size_t)0, pool->idle_queue.size() );
  CPPUNIT_ASSERT_EQUAL( (size_t)1, pool->busy_queue.size() );

  // get one more
  CPPUNIT_ASSERT( !pool->presentIdleWorkers() );
#ifndef NDEBUG
  std::cout << "The following assertion is intended and does not indicate a failure." << std::endl;
  CPPUNIT_ASSERT_THROW( pool->getNextIdleWorker(), Assert::AssertionFailure );
#else
  const WorkerAddress emptyAddress = pool->getNextIdleWorker();
#endif
  CPPUNIT_ASSERT( pool->isWorkerBusy(address) );
  CPPUNIT_ASSERT_EQUAL( (size_t)0, pool->idle_queue.size() );
  CPPUNIT_ASSERT_EQUAL( (size_t)1, pool->busy_queue.size() );
  CPPUNIT_ASSERT( !addobserver->wasNotified );
  CPPUNIT_ASSERT( !removeobserver->wasNotified );
  CPPUNIT_ASSERT( !idleobserver->wasNotified );

  // unmark and check
  pool->unmarkWorkerBusy(address);
  CPPUNIT_ASSERT( !pool->isWorkerBusy(address) );
  CPPUNIT_ASSERT_EQUAL( (size_t)1, pool->idle_queue.size() );
  CPPUNIT_ASSERT_EQUAL( (size_t)0, pool->busy_queue.size() );
  CPPUNIT_ASSERT( !addobserver->wasNotified );
  CPPUNIT_ASSERT( !removeobserver->wasNotified );
  CPPUNIT_ASSERT( idleobserver->wasNotified );
}
