/*
 * PlaneUnittest.cpp
 *
 *  Created on: Apr 30, 2010
 *      Author: crueger
 */

#include "PlaneUnittest.hpp"

#include <cppunit/CompilerOutputter.h>
#include <cppunit/extensions/TestFactoryRegistry.h>
#include <cppunit/ui/text/TestRunner.h>

#ifdef HAVE_TESTRUNNER
#include "UnitTestMain.hpp"
#endif /*HAVE_TESTRUNNER*/

#include "vector.hpp"

CPPUNIT_TEST_SUITE_REGISTRATION( PlaneUnittest );

void PlaneUnittest::setUp(){
  p1 = new Plane(e1,e2,e3);
  p2 = new Plane(e1,e2,zeroVec);
  p3 = new Plane(e1,zeroVec,e3);
  p4 = new Plane(zeroVec,e2,e3);
}

void PlaneUnittest::tearDown(){
  delete p1;
  delete p2;
  delete p3;
  delete p4;
}

void PlaneUnittest::constructionErrorTest(){
  // try several method of construction..
  // see if error checking works

  // three points
  CPPUNIT_ASSERT_NO_THROW(Plane(e1,e2,e3));
  // when only two points are differnt this gives an error
  CPPUNIT_ASSERT_THROW(Plane(e1,e2,e2),LinearDependenceException);
  // same with only one point
  CPPUNIT_ASSERT_THROW(Plane(e1,e1,e1),LinearDependenceException);

  // use two vector giving two directions
  CPPUNIT_ASSERT_NO_THROW(Plane(e1,e2,0));
  // and again this is actually only one vector
  CPPUNIT_ASSERT_THROW(Plane(e1,e1,0),LinearDependenceException);
  // Zero vector does not give a good direction
  CPPUNIT_ASSERT_THROW(Plane(e1,zeroVec,0),ZeroVectorException);

  // use a normalvector and an scalar offset
  CPPUNIT_ASSERT_NO_THROW(Plane(e1,0));
  // The zero vector is no good as a normalvector
  CPPUNIT_ASSERT_THROW(Plane(zeroVec,0),ZeroVectorException);

  // use a normalvector and an offset vector
  CPPUNIT_ASSERT_NO_THROW(Plane(e1,zeroVec));
  // and the bad zeroVector again
  CPPUNIT_ASSERT_THROW(Plane(zeroVec,zeroVec),ZeroVectorException);
}


// we need to test normals independent of the direction
bool testNormal(const Vector &normal1, const Vector &normal2){
  return (normal1==normal2) || (normal1==-1*normal2);
}

void PlaneUnittest::constructionResultTest(){
  {
    // construct with three points on plane
    Plane p1(e1,e2,zeroVec);
    CPPUNIT_ASSERT(testNormal(e3,p1.getNormal()));
    CPPUNIT_ASSERT_EQUAL(0.,p1.getOffset());

    Plane p2(e1,e3,zeroVec);
    CPPUNIT_ASSERT(testNormal(e2,p2.getNormal()));
    CPPUNIT_ASSERT_EQUAL(0.,p2.getOffset());

    Plane p3(e2,e3,zeroVec);
    CPPUNIT_ASSERT(testNormal(e1,p3.getNormal()));
    CPPUNIT_ASSERT_EQUAL(0.,p3.getOffset());
  }
  {
    // construct with two directions + offset
    Plane p1(e1,e2,0);
    CPPUNIT_ASSERT(testNormal(e3,p1.getNormal()));
    CPPUNIT_ASSERT_EQUAL(0.,p1.getOffset());

    Plane p2(e1,e3,0);
    CPPUNIT_ASSERT(testNormal(e2,p2.getNormal()));
    CPPUNIT_ASSERT_EQUAL(0.,p2.getOffset());

    Plane p3(e2,e3,0);
    CPPUNIT_ASSERT(testNormal(e1,p3.getNormal()));
    CPPUNIT_ASSERT_EQUAL(0.,p3.getOffset());
  }
}

void PlaneUnittest::pointsTest(){
  std::vector<Vector> points1 = p1->getPointsOnPlane();
  CPPUNIT_ASSERT(p1->isContained(points1[0]));
  CPPUNIT_ASSERT(p1->isContained(points1[1]));
  CPPUNIT_ASSERT(p1->isContained(points1[2]));
  // check that the three points differ
  CPPUNIT_ASSERT(points1[0]!=points1[1]);
  CPPUNIT_ASSERT(points1[0]!=points1[2]);
  CPPUNIT_ASSERT(points1[1]!=points1[2]);


  std::vector<Vector> points2 = p2->getPointsOnPlane();
  CPPUNIT_ASSERT(p2->isContained(points2[0]));
  CPPUNIT_ASSERT(p2->isContained(points2[1]));
  CPPUNIT_ASSERT(p2->isContained(points2[2]));
  // check that the three points differ
  CPPUNIT_ASSERT(points2[0]!=points2[1]);
  CPPUNIT_ASSERT(points2[0]!=points2[2]);
  CPPUNIT_ASSERT(points2[1]!=points2[2]);

  std::vector<Vector> points3 = p3->getPointsOnPlane();
  CPPUNIT_ASSERT(p3->isContained(points3[0]));
  CPPUNIT_ASSERT(p3->isContained(points3[1]));
  CPPUNIT_ASSERT(p3->isContained(points3[2]));
  // check that the three points differ
  CPPUNIT_ASSERT(points3[0]!=points3[1]);
  CPPUNIT_ASSERT(points3[0]!=points3[2]);
  CPPUNIT_ASSERT(points3[1]!=points3[2]);

  std::vector<Vector> points4 = p4->getPointsOnPlane();
  CPPUNIT_ASSERT(p4->isContained(points4[0]));
  CPPUNIT_ASSERT(p4->isContained(points4[1]));
  CPPUNIT_ASSERT(p4->isContained(points4[2]));
  // check that the three points differ
  CPPUNIT_ASSERT(points4[0]!=points4[1]);
  CPPUNIT_ASSERT(points4[0]!=points4[2]);
  CPPUNIT_ASSERT(points4[1]!=points4[2]);
}


void PlaneUnittest::operationsTest(){
  {
    Vector t = (1./3.)*(e1+e2+e3);
    CPPUNIT_ASSERT(fabs(p1->distance(zeroVec)-t.Norm()) < MYEPSILON);
    CPPUNIT_ASSERT_EQUAL(t,p1->getClosestPoint(zeroVec));
  }

  CPPUNIT_ASSERT(fabs(p2->distance(e3)-1) < MYEPSILON);
  CPPUNIT_ASSERT_EQUAL(zeroVec,p2->getClosestPoint(e3));
  CPPUNIT_ASSERT(fabs(p3->distance(e2)-1) < MYEPSILON);
  CPPUNIT_ASSERT_EQUAL(zeroVec,p3->getClosestPoint(e2));
  CPPUNIT_ASSERT(fabs(p4->distance(e1)-1) < MYEPSILON);
  CPPUNIT_ASSERT_EQUAL(zeroVec,p4->getClosestPoint(e1));


}
