/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2012 University of Bonn. All rights reserved.
 * Copyright (C)  2013 Frederik Heber. All rights reserved.
 * Please see the COPYING file or "Copyright notice" in builder.cpp for details.
 * 
 *
 *   This file is part of MoleCuilder.
 *
 *    MoleCuilder is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    MoleCuilder is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with MoleCuilder.  If not, see <http://www.gnu.org/licenses/>. 
 */

/*
 * ThreeBodyPotential_Angle.cpp
 *
 *  Created on: Oct 11, 2012
 *      Author: heber
 */


// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "CodePatterns/MemDebug.hpp"

#include "ThreeBodyPotential_Angle.hpp"

#include <boost/assign/list_of.hpp> // for 'map_list_of()'
#include <boost/bind.hpp>
#include <boost/lambda/lambda.hpp>
#include <string>

#include "CodePatterns/Assert.hpp"

#include "FunctionApproximation/Extractors.hpp"
#include "FunctionApproximation/TrainingData.hpp"
#include "Potentials/helpers.hpp"
#include "Potentials/InternalCoordinates/ThreeBody_Angle.hpp"
#include "Potentials/ParticleTypeCheckers.hpp"
#include "RandomNumbers/RandomNumberGeneratorFactory.hpp"
#include "RandomNumbers/RandomNumberGenerator.hpp"

class Fragment;

// static definitions
const ThreeBodyPotential_Angle::ParameterNames_t
ThreeBodyPotential_Angle::ParameterNames =
      boost::assign::list_of<std::string>
      ("spring_constant")
      ("equilibrium_distance")
    ;
const std::string ThreeBodyPotential_Angle::potential_token("harmonic_angle");
Coordinator::ptr ThreeBodyPotential_Angle::coordinator(Memory::ignore(new ThreeBody_Angle()));

static BindingModel generateBindingModel(const EmpiricalPotential::ParticleTypes_t &_ParticleTypes)
{
  // fill nodes
  BindingModel::vector_nodes_t nodes;
  {
    ASSERT( _ParticleTypes.size() == (size_t)3,
        "generateBindingModel() - ThreeBodyPotential_Angle needs three types.");
    nodes.push_back( FragmentNode(_ParticleTypes[0], 1) );
    nodes.push_back( FragmentNode(_ParticleTypes[1], 2) );
    nodes.push_back( FragmentNode(_ParticleTypes[2], 1) );
  }

  // there are no edges
  HomologyGraph::edges_t edges;
  {
    std::pair<HomologyGraph::edges_t::iterator, bool > inserter;
    inserter = edges.insert( std::make_pair( FragmentEdge(_ParticleTypes[0], _ParticleTypes[1]), 1) );
    if (!inserter.second)
      ++(inserter.first->second);
    inserter = edges.insert( std::make_pair( FragmentEdge(_ParticleTypes[1], _ParticleTypes[2]), 1) );
    if (!inserter.second)
      ++(inserter.first->second);
  }

  return BindingModel(nodes, edges);
}

ThreeBodyPotential_Angle::ThreeBodyPotential_Angle() :
  EmpiricalPotential(),
  params(parameters_t(MAXPARAMS, 0.)),
  bindingmodel(BindingModel())
{
  // have some decent defaults for parameter_derivative checking
  params[spring_constant] = 1.;
  params[equilibrium_distance] = 0.1;
}

ThreeBodyPotential_Angle::ThreeBodyPotential_Angle(
    const ParticleTypes_t &_ParticleTypes
    ) :
  EmpiricalPotential(_ParticleTypes),
  params(parameters_t(MAXPARAMS, 0.)),
  bindingmodel(generateBindingModel(_ParticleTypes))
{
  // have some decent defaults for parameter_derivative checking
  params[spring_constant] = 1.;
  params[equilibrium_distance] = 0.1;
}

ThreeBodyPotential_Angle::ThreeBodyPotential_Angle(
    const ParticleTypes_t &_ParticleTypes,
    const double _spring_constant,
    const double _equilibrium_distance) :
  EmpiricalPotential(_ParticleTypes),
  params(parameters_t(MAXPARAMS, 0.)),
  bindingmodel(generateBindingModel(_ParticleTypes))
{
  params[spring_constant] = _spring_constant;
  params[equilibrium_distance] = _equilibrium_distance;
}

void ThreeBodyPotential_Angle::setParameters(const parameters_t &_params)
{
  const size_t paramsDim = _params.size();
  ASSERT( paramsDim <= getParameterDimension(),
      "ThreeBodyPotential_Angle::setParameters() - we need not more than "
      +toString(getParameterDimension())+" parameters.");
  for(size_t i=0;i<paramsDim;++i)
    params[i] = _params[i];

#ifndef NDEBUG
  parameters_t check_params(getParameters());
  check_params.resize(paramsDim); // truncate to same size
  ASSERT( check_params == _params,
      "ThreeBodyPotential_Angle::setParameters() - failed, mismatch in to be set "
      +toString(_params)+" and set "+toString(check_params)+" params.");
#endif
}

ThreeBodyPotential_Angle::result_t
ThreeBodyPotential_Angle::function_theta(
    const double &r_ij,
    const double &r_jk,
    const double &r_ik
  ) const
{
//  Info info(__func__);
  const double angle = Helpers::pow(r_ij,2) + Helpers::pow(r_jk,2) - Helpers::pow(r_ik,2);
  const double divisor = 2.* r_ij * r_jk;

//  LOG(2, "DEBUG: cos(theta)= " << angle/divisor);
  if (divisor == 0.)
    return 0.;
  else
    return angle/divisor;
}

ThreeBodyPotential_Angle::results_t
ThreeBodyPotential_Angle::operator()(
    const list_of_arguments_t &listarguments
    ) const
{
  result_t result = 0.;
  for(list_of_arguments_t::const_iterator iter = listarguments.begin();
      iter != listarguments.end(); ++iter) {
    const arguments_t &arguments = *iter;
    ASSERT( arguments.size() == 3,
        "ThreeBodyPotential_Angle::operator() - requires exactly three arguments.");
    ASSERT( ParticleTypeChecker::checkArgumentsAgainstParticleTypes(
        arguments, getParticleTypes()),
        "ThreeBodyPotential_Angle::operator() - types don't match with ones in arguments.");
    const argument_t &r_ij = arguments[0]; // 01
    const argument_t &r_jk = arguments[2]; // 12
    const argument_t &r_ik = arguments[1]; // 02
    result +=
        params[spring_constant]
               * Helpers::pow( function_theta(r_ij.distance, r_jk.distance, r_ik.distance)
                   - params[equilibrium_distance], 2 );
  }
  return results_t(1, result);
}

ThreeBodyPotential_Angle::derivative_components_t
ThreeBodyPotential_Angle::derivative(
    const list_of_arguments_t &listarguments
    ) const
{
  result_t result = 0.;
  for(list_of_arguments_t::const_iterator iter = listarguments.begin();
      iter != listarguments.end(); ++iter) {
    const arguments_t &arguments = *iter;
    ASSERT( arguments.size() == 3,
        "ThreeBodyPotential_Angle::operator() - requires exactly three arguments.");
    ASSERT( ParticleTypeChecker::checkArgumentsAgainstParticleTypes(
        arguments, getParticleTypes()),
        "ThreeBodyPotential_Angle::operator() - types don't match with ones in arguments.");
    const argument_t &r_ij = arguments[0]; //01
    const argument_t &r_jk = arguments[2]; //12
    const argument_t &r_ik = arguments[1]; //02
    result +=
        2. * params[spring_constant] *
        ( function_theta(r_ij.distance, r_jk.distance, r_ik.distance)
            - params[equilibrium_distance]);
  }
  return derivative_components_t(1, result);
}

ThreeBodyPotential_Angle::results_t
ThreeBodyPotential_Angle::parameter_derivative(
    const list_of_arguments_t &listarguments,
    const size_t index
    ) const
{
  result_t result = 0.;
  for(list_of_arguments_t::const_iterator iter = listarguments.begin();
      iter != listarguments.end(); ++iter) {
    const arguments_t &arguments = *iter;
    ASSERT( arguments.size() == 3,
        "ThreeBodyPotential_Angle::parameter_derivative() - requires exactly three arguments.");
    ASSERT( ParticleTypeChecker::checkArgumentsAgainstParticleTypes(
        arguments, getParticleTypes()),
        "ThreeBodyPotential_Angle::operator() - types don't match with ones in arguments.");
    const argument_t &r_ij = arguments[0]; //01
    const argument_t &r_jk = arguments[2]; //12
    const argument_t &r_ik = arguments[1]; //02
    switch (index) {
      case spring_constant:
      {
        result +=
            Helpers::pow( function_theta(r_ij.distance, r_jk.distance, r_ik.distance) - params[equilibrium_distance], 2 );
        break;
      }
      case equilibrium_distance:
      {
        result +=
            -2. * params[spring_constant]
                   * ( function_theta(r_ij.distance, r_jk.distance, r_ik.distance) - params[equilibrium_distance]);
        break;
      }
      default:
        ASSERT(0, "ThreeBodyPotential_Angle::parameter_derivative() - derivative to unknown parameter desired.");
        break;
    }
  }
  return results_t(1, result);
}

FunctionModel::filter_t ThreeBodyPotential_Angle::getSpecificFilter() const
{
  FunctionModel::filter_t returnfunction =
      boost::bind(&Extractors::filterArgumentsByBindingModel,
        _2, _1,
        boost::cref(getParticleTypes()), boost::cref(getBindingModel())
      );
  return returnfunction;
}

void
ThreeBodyPotential_Angle::setParametersToRandomInitialValues(
    const TrainingData &data)
{
  RandomNumberGenerator &random = RandomNumberGeneratorFactory::getInstance().makeRandomNumberGenerator();
  const double rng_min = random.min();
  const double rng_max = random.max();
  params[ThreeBodyPotential_Angle::spring_constant] = 1e+0*(random()/(rng_max-rng_min));// 0.2;
  params[ThreeBodyPotential_Angle::equilibrium_distance] = -0.3;//2e+0*(random()/(rng_max-rng_min)) - 1.;// 1.;
}

