/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2013 Frederik Heber. All rights reserved.
 * Please see the COPYING file or "Copyright notice" in builder.cpp for details.
 * 
 *
 *   This file is part of MoleCuilder.
 *
 *    MoleCuilder is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    MoleCuilder is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with MoleCuilder.  If not, see <http://www.gnu.org/licenses/>. 
 */

/*
 * FourBodyPotential_Torsion.cpp
 *
 *  Created on: Jul 08, 2013
 *      Author: heber
 */


// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "CodePatterns/MemDebug.hpp"

#include "FourBodyPotential_Torsion.hpp"

#include <boost/assign/list_of.hpp> // for 'map_list_of()'
#include <boost/bind.hpp>
#include <boost/lambda/lambda.hpp>
#include <string>

#include "CodePatterns/Assert.hpp"

#include "FunctionApproximation/Extractors.hpp"
#include "FunctionApproximation/TrainingData.hpp"
#include "Potentials/helpers.hpp"
#include "Potentials/InternalCoordinates/FourBody_TorsionAngle.hpp"
#include "Potentials/ParticleTypeCheckers.hpp"
#include "RandomNumbers/RandomNumberGeneratorFactory.hpp"
#include "RandomNumbers/RandomNumberGenerator.hpp"

class Fragment;

// static definitions1
const FourBodyPotential_Torsion::ParameterNames_t
FourBodyPotential_Torsion::ParameterNames =
      boost::assign::list_of<std::string>
      ("spring_constant")
      ("equilibrium_distance")
    ;
const std::string FourBodyPotential_Torsion::potential_token("torsion");
Coordinator::ptr FourBodyPotential_Torsion::coordinator(Memory::ignore(new FourBody_TorsionAngle()));

static BindingModel generateBindingModel(const EmpiricalPotential::ParticleTypes_t &_ParticleTypes)
{
  // fill nodes
  BindingModel::vector_nodes_t nodes;
  {
    ASSERT( _ParticleTypes.size() == (size_t)4,
        "generateBindingModel() - FourBodyPotential_Torsion needs four types.");
    nodes.push_back( FragmentNode(_ParticleTypes[0], 1) );
    nodes.push_back( FragmentNode(_ParticleTypes[1], 2) );
    nodes.push_back( FragmentNode(_ParticleTypes[2], 2) );
    nodes.push_back( FragmentNode(_ParticleTypes[3], 1) );
  }

  // there are no edges
  HomologyGraph::edges_t edges;
  {
    std::pair<HomologyGraph::edges_t::iterator, bool > inserter;
    inserter = edges.insert( std::make_pair( FragmentEdge(_ParticleTypes[0], _ParticleTypes[1]), 1) );
    if (!inserter.second)
      ++(inserter.first->second);
    inserter = edges.insert( std::make_pair( FragmentEdge(_ParticleTypes[1], _ParticleTypes[2]), 1) );
    if (!inserter.second)
      ++(inserter.first->second);
    inserter = edges.insert( std::make_pair( FragmentEdge(_ParticleTypes[2], _ParticleTypes[3]), 1) );
    if (!inserter.second)
      ++(inserter.first->second);
  }

  return BindingModel(nodes, edges);
}

FourBodyPotential_Torsion::FourBodyPotential_Torsion() :
  EmpiricalPotential(),
  params(parameters_t(MAXPARAMS, 0.)),
  bindingmodel(BindingModel())
{
  // have some decent defaults for parameter_derivative checking
  params[spring_constant] = 1.;
  params[equilibrium_distance] = 0.1;
}

FourBodyPotential_Torsion::FourBodyPotential_Torsion(
    const ParticleTypes_t &_ParticleTypes
    ) :
  EmpiricalPotential(_ParticleTypes),
  params(parameters_t(MAXPARAMS, 0.)),
  bindingmodel(generateBindingModel(_ParticleTypes))
{
  // have some decent defaults for parameter_derivative checking
  params[spring_constant] = 1.;
  params[equilibrium_distance] = 0.1;
}

FourBodyPotential_Torsion::FourBodyPotential_Torsion(
    const ParticleTypes_t &_ParticleTypes,
    const double _spring_constant,
    const double _equilibrium_distance) :
  EmpiricalPotential(_ParticleTypes),
  params(parameters_t(MAXPARAMS, 0.)),
  bindingmodel(generateBindingModel(_ParticleTypes))
{
  params[spring_constant] = _spring_constant;
  params[equilibrium_distance] = _equilibrium_distance;
}

void FourBodyPotential_Torsion::setParameters(const parameters_t &_params)
{
  const size_t paramsDim = _params.size();
  ASSERT( paramsDim <= getParameterDimension(),
      "FourBodyPotential_Torsion::setParameters() - we need not more than "
      +toString(getParameterDimension())+" parameters.");
  for(size_t i=0;i<paramsDim;++i)
    params[i] = _params[i];

#ifndef NDEBUG
  parameters_t check_params(getParameters());
  check_params.resize(paramsDim); // truncate to same size
  ASSERT( check_params == _params,
      "FourBodyPotential_Torsion::setParameters() - failed, mismatch in to be set "
      +toString(_params)+" and set "+toString(check_params)+" params.");
#endif
}

FourBodyPotential_Torsion::result_t
FourBodyPotential_Torsion::function_theta(
    const double &r_ij,
    const double &r_ik,
    const double &r_il,
    const double &r_jk,
    const double &r_jl,
    const double &r_kl
  ) const
{
//  Info info(__func__);
  const double h_1 = .5*sqrt(2.*(Helpers::pow(r_ij,2)+Helpers::pow(r_ik,2))-Helpers::pow(r_jk,2));
  const double h_2 = .5*sqrt(2.*(Helpers::pow(r_jl,2)+Helpers::pow(r_kl,2))-Helpers::pow(r_jk,2));
  const double angle = Helpers::pow(h_1,2) + Helpers::pow(h_2,2) - Helpers::pow(r_il,2);
  const double divisor = 2.* h_1 * h_2;

//  LOG(2, "DEBUG: cos(theta)= " << angle/divisor);
  if (divisor == 0.)
    return 0.;
  else
    return angle/divisor;
}

FourBodyPotential_Torsion::results_t
FourBodyPotential_Torsion::operator()(
    const list_of_arguments_t &listarguments
    ) const
{
  result_t result = 0.;
  for(list_of_arguments_t::const_iterator iter = listarguments.begin();
      iter != listarguments.end(); ++iter) {
    const arguments_t &arguments = *iter;
    ASSERT( arguments.size() == getSpecificArgumentCount(),
        "FourBodyPotential_Torsion::operator() - requires exactly three arguments.");
    ASSERT( ParticleTypeChecker::checkArgumentsAgainstParticleTypes(
        arguments, getParticleTypes()),
        "FourBodyPotential_Torsion::operator() - types don't match with ones in arguments.");
    const argument_t &r_ij = arguments[0]; // 01
    const argument_t &r_ik = arguments[1]; // 02
    const argument_t &r_il = arguments[2]; // 03
    const argument_t &r_jk = arguments[3]; // 12
    const argument_t &r_jl = arguments[4]; // 13
    const argument_t &r_kl = arguments[5]; // 23
    result +=
        params[spring_constant]
               * Helpers::pow( function_theta(r_ij.distance, r_ik.distance, r_il.distance, r_jk.distance, r_jl.distance, r_kl.distance)
                   - params[equilibrium_distance], 2 );
  }
  return results_t(1, result);
}

FourBodyPotential_Torsion::derivative_components_t
FourBodyPotential_Torsion::derivative(
    const list_of_arguments_t &listarguments
    ) const
{
  result_t result = 0.;
  for(list_of_arguments_t::const_iterator iter = listarguments.begin();
      iter != listarguments.end(); ++iter) {
    const arguments_t &arguments = *iter;
    ASSERT( arguments.size() == getSpecificArgumentCount(),
        "FourBodyPotential_Torsion::operator() - requires exactly three arguments.");
    ASSERT( ParticleTypeChecker::checkArgumentsAgainstParticleTypes(
        arguments, getParticleTypes()),
        "FourBodyPotential_Torsion::operator() - types don't match with ones in arguments.");
    const argument_t &r_ij = arguments[0]; // 01
    const argument_t &r_ik = arguments[1]; // 02
    const argument_t &r_il = arguments[2]; // 03
    const argument_t &r_jk = arguments[3]; // 12
    const argument_t &r_jl = arguments[4]; // 13
    const argument_t &r_kl = arguments[5]; // 23
    result +=
        2. * params[spring_constant] *
        ( function_theta(r_ij.distance, r_ik.distance, r_il.distance, r_jk.distance, r_jl.distance, r_kl.distance)
            - params[equilibrium_distance]);
  }
  return derivative_components_t(1, result);
}

FourBodyPotential_Torsion::results_t
FourBodyPotential_Torsion::parameter_derivative(
    const list_of_arguments_t &listarguments,
    const size_t index
    ) const
{
  result_t result = 0.;
  for(list_of_arguments_t::const_iterator iter = listarguments.begin();
      iter != listarguments.end(); ++iter) {
    const arguments_t &arguments = *iter;
    ASSERT( arguments.size() == getSpecificArgumentCount(),
        "FourBodyPotential_Torsion::parameter_derivative() - requires exactly three arguments.");
    ASSERT( ParticleTypeChecker::checkArgumentsAgainstParticleTypes(
        arguments, getParticleTypes()),
        "FourBodyPotential_Torsion::operator() - types don't match with ones in arguments.");
    const argument_t &r_ij = arguments[0]; // 01
    const argument_t &r_ik = arguments[1]; // 02
    const argument_t &r_il = arguments[2]; // 03
    const argument_t &r_jk = arguments[3]; // 12
    const argument_t &r_jl = arguments[4]; // 13
    const argument_t &r_kl = arguments[5]; // 23
    switch (index) {
      case spring_constant:
      {
        result +=
            Helpers::pow( function_theta(r_ij.distance, r_ik.distance, r_il.distance, r_jk.distance, r_jl.distance, r_kl.distance) - params[equilibrium_distance], 2 );
        break;
      }
      case equilibrium_distance:
      {
        result +=
            -2. * params[spring_constant]
                   * ( function_theta(r_ij.distance, r_ik.distance, r_il.distance, r_jk.distance, r_jl.distance, r_kl.distance) - params[equilibrium_distance]);
        break;
      }
      default:
        ASSERT(0, "FourBodyPotential_Torsion::parameter_derivative() - derivative to unknown parameter desired.");
        break;
    }
  }
  return results_t(1, result);
}

FunctionModel::filter_t
FourBodyPotential_Torsion::getSpecificFilter() const
{
  FunctionModel::filter_t returnfunction =
      boost::bind(&Extractors::filterArgumentsByBindingModel,
        _2, _1,
        boost::cref(getParticleTypes()), boost::cref(getBindingModel())
      );
  return returnfunction;
}

void
FourBodyPotential_Torsion::setParametersToRandomInitialValues(
    const TrainingData &data)
{
  RandomNumberGenerator &random = RandomNumberGeneratorFactory::getInstance().makeRandomNumberGenerator();
  const double rng_min = random.min();
  const double rng_max = random.max();
  params[FourBodyPotential_Torsion::spring_constant] = 2.*(random()/(rng_max-rng_min));
  params[FourBodyPotential_Torsion::equilibrium_distance] = 2.*(random()/(rng_max-rng_min));
}

