/*
 * CountBondsUnitTest.cpp
 *
 *  Created on: Mar 30, 2010
 *      Author: heber
 */


using namespace std;

#include <cppunit/CompilerOutputter.h>
#include <cppunit/extensions/TestFactoryRegistry.h>
#include <cppunit/ui/text/TestRunner.h>

#include <iostream>
#include <stdio.h>
#include <cstring>

#include "analysis_bonds.hpp"
#include "atom.hpp"
#include "bond.hpp"
#include "bondgraph.hpp"
#include "element.hpp"
#include "molecule.hpp"
#include "periodentafel.hpp"
#include "CountBondsUnitTest.hpp"

/********************************************** Test classes **************************************/

// Registers the fixture into the 'registry'
CPPUNIT_TEST_SUITE_REGISTRATION( CountBondsTest );


void CountBondsTest::setUp()
{
  atom *Walker = NULL;

  // init private all pointers to zero
  molecules = NULL;
  TestMolecule1 = NULL;
  TestMolecule2 = NULL;
  hydrogen = NULL;
  oxygen = NULL;
  tafel = NULL;

  // construct element
  hydrogen = new element;
  hydrogen->Z = 1;
  hydrogen->CovalentRadius = 0.23;
  strcpy(hydrogen->name, "hydrogen");
  strcpy(hydrogen->symbol, "H");
  oxygen = new element;
  oxygen->Z = 8;
  oxygen->CovalentRadius = 0.68;
  strcpy(oxygen->name, "oxygen");
  strcpy(oxygen->symbol, "O");

  // construct periodentafel
  tafel = new periodentafel;
  tafel->AddElement(hydrogen);
  tafel->AddElement(oxygen);

  // construct molecule (water molecule)
  molecules = new MoleculeListClass;
  TestMolecule1 = new molecule(tafel);
  Walker = new atom();
  Walker->type = hydrogen;
  Walker->node->Init(-0.2418, 0.9350, 0. );
  TestMolecule1->AddAtom(Walker);
  Walker = new atom();
  Walker->type = hydrogen;
  Walker->node->Init(0.9658, 0., 0. );
  TestMolecule1->AddAtom(Walker);
  Walker = new atom();
  Walker->type = oxygen;
  Walker->node->Init(0., 0., 0. );
  TestMolecule1->AddAtom(Walker);
  molecules->insert(TestMolecule1);

  TestMolecule2 = new molecule(tafel);
  Walker = new atom();
  Walker->type = hydrogen;
  Walker->node->Init(-0.2418, 0.9350, 0. );
  TestMolecule2->AddAtom(Walker);
  Walker = new atom();
  Walker->type = hydrogen;
  Walker->node->Init(0.9658, 0., 0. );
  TestMolecule2->AddAtom(Walker);
  Walker = new atom();
  Walker->type = oxygen;
  Walker->node->Init(0., 0., 0. );
  TestMolecule2->AddAtom(Walker);
  molecules->insert(TestMolecule2);

  // check that TestMolecule was correctly constructed
  CPPUNIT_ASSERT_EQUAL( TestMolecule1->AtomCount, 3 );
  Walker = TestMolecule1->start->next;
  CPPUNIT_ASSERT( TestMolecule1->end != Walker );
  CPPUNIT_ASSERT_EQUAL( TestMolecule2->AtomCount, 3 );
  Walker = TestMolecule2->start->next;
  CPPUNIT_ASSERT( TestMolecule2->end != Walker );

  // create a small file with table
  BG = new BondGraph(true);

  // construct bond graphs
  CPPUNIT_ASSERT_EQUAL( true , BG->ConstructBondGraph(TestMolecule1) );
  CPPUNIT_ASSERT_EQUAL( true , BG->ConstructBondGraph(TestMolecule2) );
//  TestMolecule1->Output((ofstream *)&cout);
//  TestMolecule1->OutputBondsList();
};


void CountBondsTest::tearDown()
{
  // remove the file
  delete(BG);

  // remove molecule
  delete(molecules);
  // note that all the atoms are cleaned by TestMolecule
  delete(tafel);
  // note that element is cleaned by periodentafel
};

/** UnitTest for CountBondsTest::BondsOfTwoTest().
 */
void CountBondsTest::BondsOfTwoTest()
{
  CPPUNIT_ASSERT_EQUAL( 4 , CountBondsOfTwo(molecules, hydrogen, oxygen) );
  CPPUNIT_ASSERT_EQUAL( 0 , CountBondsOfTwo(molecules, hydrogen, hydrogen) );
  CPPUNIT_ASSERT_EQUAL( 0 , CountBondsOfTwo(molecules, oxygen, oxygen) );
};

/** UnitTest for CountBondsTest::BondsOfThreeTest().
 */
void CountBondsTest::BondsOfThreeTest()
{
  CPPUNIT_ASSERT_EQUAL( 2 , CountBondsOfThree(molecules, hydrogen, oxygen, hydrogen) );
  CPPUNIT_ASSERT_EQUAL( 0 , CountBondsOfThree(molecules, oxygen, hydrogen, oxygen) );
};

void OutputTestMolecule(molecule *mol, const char *name)
{
  ofstream output(name);
  mol->OutputXYZ(&output);
  output.close();
}

/** UnitTest for CountBondsTest::HydrogenBridgeBondsTest().
 */
void CountBondsTest::HydrogenBridgeBondsTest()
{
  double *mirror = new double[3];
  for (int i=0;i<3;i++)
    mirror[i] = -1.;
  Vector Translator;

  //OutputTestMolecule(TestMolecule1, "testmolecule1.xyz");

  cout << "Case 1: offset of (3,0,0), hence angles are (104.5, 0, 75.5, 180) < 30." << endl;
  Translator.Init(3,0,0);
  TestMolecule2->Translate(&Translator);
  CPPUNIT_ASSERT_EQUAL( 1 , CountHydrogenBridgeBonds(molecules, NULL) );
  CPPUNIT_ASSERT_EQUAL( 0 , CountHydrogenBridgeBonds(molecules, oxygen) );
  //OutputTestMolecule(TestMolecule2, "testmolecule2-1.xyz");
  Translator.Init(-3,0,0);
  TestMolecule2->Translate(&Translator);

  cout << "Case 2: offset of (0,3,0), hence angle are (14.5, 165.5, 90) < 30 (only three, because other 90 is missing due to first H01 only fulfilling H-bond criteria)." << endl;
  Translator.Init(0,3,0);
  TestMolecule2->Translate(&Translator);
  CPPUNIT_ASSERT_EQUAL( 1 , CountHydrogenBridgeBonds(molecules, NULL) );
  //OutputTestMolecule(TestMolecule2, "testmolecule2-2.xyz");
  Translator.Init(0,-3,0);
  TestMolecule2->Translate(&Translator);

  cout << "Case 3: offset of (0,-3,0) and mirror, hence angle are (165.5, 90, 165.5, 90) > 30." << endl;
  Translator.Init(0,-3,0);
  TestMolecule2->Scale((const double ** const)&mirror);
  TestMolecule2->Translate(&Translator);
  CPPUNIT_ASSERT_EQUAL( 0 , CountHydrogenBridgeBonds(molecules, NULL) );
  //OutputTestMolecule(TestMolecule2, "testmolecule2-3.xyz");
  Translator.Init(0,3,0);
  TestMolecule2->Translate(&Translator);
  TestMolecule2->Scale((const double ** const)&mirror);

  cout << "Case 4: offset of (2,1,0), hence angle are (78, 26.6, 102, 153.4) < 30." << endl;
  Translator.Init(2,1,0);
  TestMolecule2->Translate(&Translator);
  CPPUNIT_ASSERT_EQUAL( 1 , CountHydrogenBridgeBonds(molecules, NULL) );
  //OutputTestMolecule(TestMolecule2, "testmolecule2-4.xyz");
  Translator.Init(-2,-1,0);
  TestMolecule2->Translate(&Translator);

  cout << "Case 5: offset of (0,0,3), hence angle are (90, 90, 90, 90) > 30." << endl;
  Translator.Init(0,0,3);
  TestMolecule2->Translate(&Translator);
  CPPUNIT_ASSERT_EQUAL( 0 , CountHydrogenBridgeBonds(molecules, NULL) );
  //OutputTestMolecule(TestMolecule2, "testmolecule2-5.xyz");
  Translator.Init(0,0,-3);
  TestMolecule2->Translate(&Translator);

  cout << "Case 6: offset of (-3,0,0) and mirror, hence angle are (75.5, 180, 104.5, 180) > 30." << endl;
  Translator.Init(-3,0,0);
  TestMolecule2->Scale((const double ** const)&mirror);
  TestMolecule2->Translate(&Translator);
  CPPUNIT_ASSERT_EQUAL( 0 , CountHydrogenBridgeBonds(molecules, NULL) );
  //OutputTestMolecule(TestMolecule2, "testmolecule2-6.xyz");
  Translator.Init(3,0,0);
  TestMolecule2->Translate(&Translator);
  TestMolecule2->Scale((const double ** const)&mirror);

  cout << "Case 7: offset of (3,0,0) and mirror, hence angles are (104.5, 0, 104.5, 0) < 30, but interfering hydrogens." << endl;
  Translator.Init(3,0,0);
  TestMolecule2->Scale((const double ** const)&mirror);
  TestMolecule2->Translate(&Translator);
  CPPUNIT_ASSERT_EQUAL( 0 , CountHydrogenBridgeBonds(molecules, NULL) );
  //OutputTestMolecule(TestMolecule2, "testmolecule2-7.xyz");
  Translator.Init(-3,0,0);
  TestMolecule2->Translate(&Translator);
  TestMolecule2->Scale((const double ** const)&mirror);

  cout << "Case 8: offset of (0,3,0), hence angle are (14.5, 90, 14.5, 90) < 30, but interfering hydrogens." << endl;
  Translator.Init(0,3,0);
  TestMolecule2->Scale((const double ** const)&mirror);
  TestMolecule2->Translate(&Translator);
  //OutputTestMolecule(TestMolecule2, "testmolecule2-8.xyz");
  CPPUNIT_ASSERT_EQUAL( 0 , CountHydrogenBridgeBonds(molecules, NULL) );
  Translator.Init(0,-3,0);
  TestMolecule2->Translate(&Translator);
  TestMolecule2->Scale((const double ** const)&mirror);

  delete(mirror);
};


/********************************************** Main routine **************************************/

int main(int argc, char **argv)
{
  // Get the top level suite from the registry
  CppUnit::Test *suite = CppUnit::TestFactoryRegistry::getRegistry().makeTest();

  // Adds the test to the list of test to run
  CppUnit::TextUi::TestRunner runner;
  runner.addTest( suite );

  // Change the default outputter to a compiler error format outputter
  runner.setOutputter( new CppUnit::CompilerOutputter( &runner.result(),
                                                       std::cerr ) );
  // Run the tests.
  bool wasSucessful = runner.run();

  // Return error code 1 if the one of test failed.
  return wasSucessful ? 0 : 1;
};
