/** \file linkedcell.cpp
 *
 * Function implementations for the class LinkedCell.
 *
 */


#include "linkedcell.hpp"
#include "molecules.hpp"
#include "tesselation.hpp"

// ========================================================= class LinkedCell ===========================================


/** Constructor for class LinkedCell.
 */
LinkedCell::LinkedCell()
{
  LC = NULL;
  for(int i=0;i<NDIM;i++)
    N[i] = 0;
  index = -1;
  RADIUS = 0.;
  max.Zero();
  min.Zero();
};

/** Puts all atoms in \a *mol into a linked cell list with cell's lengths of \a RADIUS
 * \param *set LCNodeSet class with all LCNode's
 * \param RADIUS edge length of cells
 */
LinkedCell::LinkedCell(PointCloud *set, double radius)
{
  TesselPoint *Walker = NULL;

  RADIUS = radius;
  LC = NULL;
  for(int i=0;i<NDIM;i++)
    N[i] = 0;
  index = -1;
  max.Zero();
  min.Zero();
  cout << Verbose(1) << "Begin of LinkedCell" << endl;
  if (set->IsEmpty()) {
    cerr << "ERROR: set contains no linked cell nodes!" << endl;
    return;
  }
  // 1. find max and min per axis of atoms
  set->GoToFirst();
  Walker = set->GetPoint();
  for (int i=0;i<NDIM;i++) {
    max.x[i] = Walker->node->x[i];
    min.x[i] = Walker->node->x[i];
  }
  set->GoToFirst();
  while (!set->IsLast()) {
    Walker = set->GetPoint();
    for (int i=0;i<NDIM;i++) {
      if (max.x[i] < Walker->node->x[i])
        max.x[i] = Walker->node->x[i];
      if (min.x[i] > Walker->node->x[i])
        min.x[i] = Walker->node->x[i];
    }
    set->GoToNext();
  }
  cout << Verbose(2) << "Bounding box is " << min << " and " << max << "." << endl;

  // 2. find then number of cells per axis
  for (int i=0;i<NDIM;i++) {
    N[i] = (int)floor((max.x[i] - min.x[i])/RADIUS)+1;
  }
  cout << Verbose(2) << "Number of cells per axis are " << N[0] << ", " << N[1] << " and " << N[2] << "." << endl;

  // 3. allocate the lists
  cout << Verbose(2) << "Allocating cells ... ";
  if (LC != NULL) {
    cout << Verbose(1) << "ERROR: Linked Cell list is already allocated, I do nothing." << endl;
    return;
  }
  LC = new LinkedNodes[N[0]*N[1]*N[2]];
  for (index=0;index<N[0]*N[1]*N[2];index++) {
    LC [index].clear();
  }
  cout << "done."  << endl;

  // 4. put each atom into its respective cell
  cout << Verbose(2) << "Filling cells ... ";
  set->GoToFirst();
  while (!set->IsLast()) {
    Walker = set->GetPoint();
    for (int i=0;i<NDIM;i++) {
      n[i] = (int)floor((Walker->node->x[i] - min.x[i])/RADIUS);
    }
    index = n[0] * N[1] * N[2] + n[1] * N[2] + n[2];
    LC[index].push_back(Walker);
    //cout << Verbose(2) << *Walker << " goes into cell " << n[0] << ", " << n[1] << ", " << n[2] << " with No. " << index << "." << endl;
    set->GoToNext();
  }
  cout << "done."  << endl;
  cout << Verbose(1) << "End of LinkedCell" << endl;
};

/** Destructor for class LinkedCell.
 */
LinkedCell::~LinkedCell()
{
  if (LC != NULL)
  for (index=0;index<N[0]*N[1]*N[2];index++)
    LC[index].clear();
  delete[](LC);
  for(int i=0;i<NDIM;i++)
    N[i] = 0;
  index = -1;
  max.Zero();
  min.Zero();
};

/** Checks whether LinkedCell::n[] is each within [0,N[]].
 * \return if all in intervals - true, else -false
 */
bool LinkedCell::CheckBounds()
{
  bool status = true;
  for(int i=0;i<NDIM;i++)
    status = status && ((n[i] >=0) && (n[i] < N[i]));
  if (!status)
  cerr << "ERROR: indices are out of bounds!" << endl;
  return status;
};


/** Returns a pointer to the current cell.
 * \return LinkedAtoms pointer to current cell, NULL if LinkedCell::n[] are out of bounds.
 */
LinkedNodes* LinkedCell::GetCurrentCell()
{
  if (CheckBounds()) {
    index = n[0] * N[1] * N[2] + n[1] * N[2] + n[2];
    return (&(LC[index]));
  } else {
    return NULL;
  }
};

/** Calculates the index for a given LCNode *Walker.
 * \param *Walker LCNode to set index tos
 * \return if the atom is also found in this cell - true, else - false
 */
bool LinkedCell::SetIndexToNode(TesselPoint *Walker)
{
  bool status = false;
  for (int i=0;i<NDIM;i++) {
    n[i] = (int)floor((Walker->node->x[i] - min.x[i])/RADIUS);
  }
  index = n[0] * N[1] * N[2] + n[1] * N[2] + n[2];
  if (CheckBounds()) {
    for (LinkedNodes::iterator Runner = LC[index].begin(); Runner != LC[index].end(); Runner++)
      status = status || ((*Runner) == Walker);
    return status;
  } else {
    cerr << Verbose(1) << "ERROR: Node at " << *Walker << " is out of bounds." << endl;
    return false;
  }
};

/** Calculates the index for a given Vector *x.
 * \param *x Vector with coordinates
 * \return Vector is inside bounding box - true, else - false
 */
bool LinkedCell::SetIndexToVector(Vector *x)
{
  bool status = true;
  for (int i=0;i<NDIM;i++) {
    n[i] = (int)floor((x->x[i] - min.x[i])/RADIUS);
    if (max.x[i] < x->x[i])
      status = false;
    if (min.x[i] > x->x[i])
      status = false;
  }
  return status;
};

