/*
 * PythonScripting.hpp
 *
 *  Created on: Apr 13, 2012
 *      Author: heber
 */

#ifndef PYTHONSCRIPTING_IMPL_HPP_
#define PYTHONSCRIPTING_IMPL_HPP_


// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif


#include <boost/python.hpp>
#include <boost/python/module.hpp>
#include <boost/python/args.hpp>
#include <boost/python/numpy.hpp>

#include "CodePatterns/toString.hpp"

// all "getter" functions
#include "modules.hpp"

//!> define all present actions
#include "GlobalListOfActions.hpp"

//!> python wrapping for all of these actions
#include "AllActionPython.hpp"

namespace MoleCuilder {

namespace PythonTypes {

inline void IndexError(){
    PyErr_SetString(PyExc_IndexError, "Index out of range");
    boost::python::throw_error_already_set();
}

template<class T>
struct vec_item{
    typedef typename T::value_type V;
    static V& get(T& x, int i){
        static V nothing;
        if(i < 0) i += x.size();
        if(i >= 0 && i < int(x.size())) return x[i];
        IndexError();
        return nothing;
    }
    static void set(T& x, int i, V const& v){
        if(i < 0) i += x.size();
        if(i >= 0 && i < int(x.size())) x[i] = v;
        else IndexError();
    }
    static void del(T& x, int i){
        if(i < 0) i += x.size();
        if(i >= 0 && i < int(x.size())) x.erase(x.begin() + i);
        else IndexError();
    }
    static void add(T& x, V const& v){
        x.push_back(v);
    }
    static std::string toStringRepr(T& x){
        return toString(x);
    }
};


} /* namespace PythonTypes */

} /* namespace MoleCuilder */

void export_numpy();

BOOST_PYTHON_MODULE(pyMoleCuilder)
{
  // from this moment on, we need to be sure to deeinitialize in the correct or
  // this is handled by the cleanup function
  atexit(MoleCuilder::detail::module_exit);

  // initialize numpy C-API
  boost::python::numpy::initialize();

  // set the docstring of the current module scope
  boost::python::scope().attr("__doc__") = "pyMolecuilder are the python bindings to all Actions of the program suite MoleCuilder.\n\nMoleCuilder is a program to build molecular (dynamics) worlds, allowing you indefinite manipulation, control and analysis over the atoms and molecules within a simulation domain.";

  export_numpy();

  boost::python::def(
      "reinit",
      MoleCuilder::detail::module_reinit,
      "Reinitializes the internal state of the python module as if it had been freshly imported,saves all input files beforehand."
  );
  boost::python::def< bool() >(
      "wait",
      MoleCuilder::detail::module_wait,
      "Waits until all currently queued actions have been processed."
  );
  boost::python::def< std::string() >(
      "getGraph6String",
      MoleCuilder::detail::module_getGraph6String,
      "returns chemical graph of the current selected set of atoms as graph6 representation."
  );
  boost::python::def< std::string() >(
      "getElementListAsString",
      MoleCuilder::detail::module_getElementListAsString,
      "returns list of elements over the nodes of the graph of the current selected set of atoms."
  );
  boost::python::def< MoleCuilder::detail::stringVec() >(
      "getAllGraph6Strings",
      MoleCuilder::detail::module_getAllGraph6Strings,
      "returns chemical graphs of the current selected set of atoms as graph6 representation with all non-hydrogen atom permutations."
  );
  boost::python::def< MoleCuilder::detail::stringVec() >(
      "getAllElementListAsStrings",
      MoleCuilder::detail::module_getAllElementListAsStrings,
      "returns lists of elements over the nodes of the graph of the current selected set of atoms with all non-hydrogen atom permutations."
  );
  boost::python::def< MoleCuilder::detail::doubleVec() >(
      "getBoundingBox",
      MoleCuilder::detail::module_getBoundingBox,
      "returns the cuboid bounding box of the current domain."
  );
  boost::python::def< double() >(
      "getDomainVolume",
      MoleCuilder::detail::module_getDomainVolume,
      "returns the volume of the simulation domain."
  );
  boost::python::def< MoleCuilder::detail::elementVec() >(
      "getSelectedAtomElements",
      MoleCuilder::detail::module_getSelectedAtomElements,
      "returns the element numbers of all currently selected atoms."
  );
  boost::python::def< MoleCuilder::detail::atomPositionsVec() >(
      "getSelectedAtomPositions",
      MoleCuilder::detail::module_getSelectedAtomPositions,
      "returns the positions of all currently selected atoms."
  );
  boost::python::def< int() >(
      "getSelectedAtomCount",
      MoleCuilder::detail::module_getSelectedAtomCount,
      "returns the number of all currently selected atoms."
  );
  boost::python::def< MoleCuilder::detail::atomIdVec() >(
      "getSelectedAtomIds",
      MoleCuilder::detail::module_getSelectedAtomIds,
      "returns the ids of all currently selected atoms."
  );
  boost::python::def< double() >(
      "getSelectedMolarMass",
      MoleCuilder::detail::module_getSelectedMolarMass,
      "returns the molar mass of all selected atoms."
  );

  // STL Vectors:
  // unsignedIntVec
  boost::python::class_< std::vector< atomId_t > >("PythonType_unsignedIntVec")
      .def("__len__", &std::vector< unsigned int >::size)
      .def("clear", &std::vector< unsigned int >::clear)
      .def("append", &MoleCuilder::PythonTypes::vec_item< std::vector< unsigned int > >::add,
            boost::python::with_custodian_and_ward<1, 2>()) // let container keep value
      .def("__getitem__", &MoleCuilder::PythonTypes::vec_item< std::vector< unsigned int > >::get,
           boost::python::return_value_policy<boost::python::copy_non_const_reference>())
      .def("__setitem__", &MoleCuilder::PythonTypes::vec_item< std::vector< unsigned int > >::set,
           boost::python::with_custodian_and_ward<1,2>()) // to let container keep value
      .def("__delitem__", &MoleCuilder::PythonTypes::vec_item< std::vector< unsigned int > >::del)
      .def("__iter__", boost::python::iterator< std::vector< unsigned int > >())
      .def("__repr__", &MoleCuilder::PythonTypes::vec_item< std::vector< unsigned int > >::toStringRepr)
  ;
  // doubleVec
  boost::python::class_< std::vector< double > >("PythonType_doubleVec")
      .def("__len__", &std::vector< double >::size)
      .def("clear", &std::vector< double >::clear)
      .def("append", &MoleCuilder::PythonTypes::vec_item< std::vector< double > >::add,
            boost::python::with_custodian_and_ward<1, 2>()) // let container keep value
      .def("__getitem__", &MoleCuilder::PythonTypes::vec_item< std::vector< double > >::get,
           boost::python::return_value_policy<boost::python::copy_non_const_reference>())
      .def("__setitem__", &MoleCuilder::PythonTypes::vec_item< std::vector< double > >::set,
           boost::python::with_custodian_and_ward<1,2>()) // to let container keep value
      .def("__delitem__", &MoleCuilder::PythonTypes::vec_item< std::vector< double > >::del)
      .def("__iter__", boost::python::iterator< std::vector< double > >())
      .def("__repr__", &MoleCuilder::PythonTypes::vec_item< std::vector< double > >::toStringRepr)
  ;
  // positions
  boost::python::class_< std::vector< std::vector< double > > >("PythonType_positions")
      .def("__len__", &std::vector< std::vector< double > >::size)
      .def("clear", &std::vector< std::vector< double > >::clear)
      .def("append", &MoleCuilder::PythonTypes::vec_item< std::vector< std::vector< double > > >::add,
            boost::python::with_custodian_and_ward<1, 2>()) // let container keep value
      .def("__getitem__", &MoleCuilder::PythonTypes::vec_item< std::vector< std::vector< double > > >::get,
           boost::python::return_value_policy<boost::python::copy_non_const_reference>())
      .def("__setitem__", &MoleCuilder::PythonTypes::vec_item< std::vector< std::vector< double > > >::set,
           boost::python::with_custodian_and_ward<1,2>()) // to let container keep value
      .def("__delitem__", &MoleCuilder::PythonTypes::vec_item< std::vector< std::vector< double > > >::del)
      .def("__iter__", boost::python::iterator< std::vector< std::vector< double > > >())
      .def("__repr__", &MoleCuilder::PythonTypes::vec_item< std::vector< std::vector< double > > >::toStringRepr)
  ;
  // positions
  boost::python::class_< std::vector< std::string > >("PythonType_stringVec")
      .def("__len__", &std::vector< std::string >::size)
      .def("clear", &std::vector< std::string >::clear)
      .def("append", &MoleCuilder::PythonTypes::vec_item< std::vector< std::string > >::add,
            boost::python::with_custodian_and_ward<1, 2>()) // let container keep value
      .def("__getitem__", &MoleCuilder::PythonTypes::vec_item< std::vector< std::string > >::get,
           boost::python::return_value_policy<boost::python::copy_non_const_reference>())
      .def("__setitem__", &MoleCuilder::PythonTypes::vec_item< std::vector< std::string > >::set,
           boost::python::with_custodian_and_ward<1,2>()) // to let container keep value
      .def("__delitem__", &MoleCuilder::PythonTypes::vec_item< std::vector< std::string > >::del)
      .def("__iter__", boost::python::iterator< std::vector< std::string > >())
      .def("__repr__", &MoleCuilder::PythonTypes::vec_item< std::vector< std::string > >::toStringRepr)
  ;

  // access to all Actions
#define export_print(z,n,list) \
  BOOST_PP_CAT(export_, BOOST_PP_SEQ_ELEM(n, list))();
#define BOOST_PP_LOCAL_MACRO(n) export_print(~, n, GLOBALLISTOFPYTHONACTIONS)
#define BOOST_PP_LOCAL_LIMITS  (0, BOOST_PP_DEC(BOOST_PP_SEQ_SIZE(GLOBALLISTOFPYTHONACTIONS)))
#include BOOST_PP_LOCAL_ITERATE()
#undef instance_print
}


#endif /* PYTHONSCRIPTING_IMPL_HPP_ */
