/*
 * atom.hpp
 *
 *  Created on: Aug 3, 2009
 *      Author: heber
 */

#ifndef ATOM_HPP_
#define ATOM_HPP_

using namespace std;

/*********************************************** includes ***********************************/

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <iosfwd>
#include <list>
#include <vector>

#include "Helpers/helpers.hpp"
#include "atom_atominfo.hpp"
#include "atom_bondedparticle.hpp"
#include "atom_graphnode.hpp"
#include "atom_particleinfo.hpp"
#include "TesselPoint.hpp"
#include "types.hpp"

#include "CodePatterns/enumeration.hpp"

/****************************************** forward declarations *****************************/

class Vector;
class World;
class molecule;
class Shape;

/********************************************** declarations *******************************/

/** Single atom.
 * Class incorporates position, type
 */
class atom : public GraphNode, public BondedParticle, public TesselPoint {
  friend atom* NewAtom(atomId_t);
  friend void  DeleteAtom(atom*);
  public:
    atom *father;   //!< In many-body bond order fragmentations points to originating atom
    int *sort;      //!< sort criteria

  virtual atom *clone();

  /** Pushes back another step in all trajectory vectors.
   *
   * This allows to extend all trajectories contained in different classes
   * consistently. This is implemented by the topmost class which calls the
   * real functions, \sa AppendTrajectoryStep(), by all necessary subclasses.
   */
  virtual void UpdateSteps();

  bool OutputIndexed(ofstream * const out, const int ElementNo, const int AtomNo, const char *comment = NULL) const;
  bool OutputArrayIndexed(ostream * const out,const enumeration<const element*>&, int *AtomNo, const char *comment = NULL) const;
  bool OutputXYZLine(ofstream *out) const;
  bool OutputTrajectory(ofstream * const out, const enumeration<const element*>&, int *AtomNo, const int step) const;
  bool OutputTrajectoryXYZ(ofstream * const out, const int step) const;
  void OutputMPQCLine(ostream * const out, const Vector *center) const;

  void InitComponentNr();
  void resetGraphNr();

  void EqualsFather ( const atom *ptr, const atom **res ) const;
  bool isFather(const atom *ptr);
  void CorrectFather();
  atom *GetTrueFather();
  bool Compare(const atom &ptr) const;

  double DistanceToVector(const Vector &origin) const;
  double DistanceSquaredToVector(const Vector &origin) const;
  bool IsInShape(const Shape&) const;

  // getter and setter

  /**
   * returns the World that contains this atom.
   * Use this if you need to get the world without locking
   * the singleton for example.
   *
   */
  World *getWorld();
  void setWorld(World*);

  virtual atomId_t getId() const;
  virtual bool changeId(atomId_t newId);

  /**
   * this function sets the Id without notifying the world. Only use it, if the world has already
   * gotten an ID for this Atom.
   */
   virtual void setId(atomId_t);

   void setMolecule(molecule*);
   molecule* getMolecule() const;
   void removeFromMolecule();

   int getNr() const;

   // Output operator
   std::ostream & operator << (std::ostream &ost) const;

  protected:

    /**
     * Protected constructor to ensure construction of atoms through the world.
     * see World::createAtom()
     */
    atom();

    /**
     * Protected copy-constructor to ensure construction of atoms by cloning.
     * see atom::clone()
     */
    atom(class atom *pointer);

    /**
     * Protected destructor to ensure destruction of atoms through the world.
     * see World::destroyAtom()
     */
    virtual ~atom();
  private:
    molecule *mol; // !< the molecule this atom belongs to
    World* world;
    atomId_t id;
};

/**
 * Global output operator for class atom.
 */
std::ostream & operator << (std::ostream &ost, const atom &_atom);

/**
 * internal method used by the world. Do not use if you don't know what you are doing.
 * You might get burned...
 * Use World::createAtom() instead.
 */
atom* NewAtom(atomId_t _id);

/**
* internal method used by the world. Do not use if you don't know what you are doing.
 * You might get burned...
 * Use World::destroyAtom() instead.
 */
void  DeleteAtom(atom*);

/**
 * Simple function to compare atoms by their elements to allow sorting of atoms by this criteria
 */
bool compareAtomElements(atom* atom1,atom* atom2);


#endif /* ATOM_HPP_ */
