/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2010-2011 University of Bonn. All rights reserved.
 * Please see the LICENSE file or "Copyright notice" in builder.cpp for details.
 */

/*
 * BoundaryPointSet.cpp
 *
 *  Created on: Jul 29, 2010
 *      Author: heber
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "CodePatterns/MemDebug.hpp"

#include "BoundaryPointSet.hpp"

#include <iostream>

#include "BoundaryLineSet.hpp"
#include "BoundaryTriangleSet.hpp"
#include "Atom/TesselPoint.hpp"

#include "CodePatterns/Assert.hpp"
#include "CodePatterns/Info.hpp"
#include "CodePatterns/Log.hpp"
#include "LinearAlgebra/Vector.hpp"
#include "CodePatterns/Verbose.hpp"

using namespace std;

// ======================================== Points on Boundary =================================

/** Constructor of BoundaryPointSet.
 */
BoundaryPointSet::BoundaryPointSet() :
  LinesCount(0),
  value(0.),
  Nr(-1)
{
  Info FunctionInfo(__func__);
  LOG(1, "Adding noname.");
}
;

/** Constructor of BoundaryPointSet with Tesselpoint.
 * \param *Walker TesselPoint this boundary point represents
 */
BoundaryPointSet::BoundaryPointSet(TesselPoint * const Walker) :
  LinesCount(0),
  node(Walker),
  value(0.),
  Nr(Walker->getNr())
{
  Info FunctionInfo(__func__);
  LOG(1, "Adding Node " << *Walker);
}
;

/** Destructor of BoundaryPointSet.
 * Sets node to NULL to avoid removing the original, represented TesselPoint.
 * \note When removing point from a class Tesselation, use RemoveTesselationPoint()
 */
BoundaryPointSet::~BoundaryPointSet()
{
  Info FunctionInfo(__func__);
  //LOG(0, "Erasing point Nr. " << Nr << ".");
  if (!lines.empty())
    ELOG(2, "Memory Leak! I " << *this << " am still connected to some lines.");
  node = NULL;
}
;

/** Add a line to the LineMap of this point.
 * \param *line line to add
 */
void BoundaryPointSet::AddLine(BoundaryLineSet * const line)
{
  Info FunctionInfo(__func__);
  LOG(1, "Adding " << *this << " to line " << *line << ".");
  if (line->endpoints[0] == this) {
    lines.insert(LinePair(line->endpoints[1]->Nr, line));
  } else {
    lines.insert(LinePair(line->endpoints[0]->Nr, line));
  }
  LinesCount++;
}

const std::string& BoundaryPointSet::getName() const
{
  ASSERT(node != NULL, "BoundaryPointSet::getName() - internal node is NULL.");
  return node->getName();
}

const Vector& BoundaryPointSet::getPosition() const
{
  ASSERT(node != NULL, "BoundaryPointSet::getPosition() - internal node is NULL.");
  return node->getPosition();
}

const int& BoundaryPointSet::getNr() const
{
  ASSERT(node != NULL, "BoundaryPointSet::getPosition() - internal node is NULL.");
  return node->getNr();
}

TesselPoint *BoundaryPointSet::getTesselPoint()
{
  return node;
}

ostream & operator <<(ostream &ost, const BoundaryPointSet &a)
{
  ost << "[" << a.Nr << "|" << a.getName() << " at " << a.getPosition() << "]";
  return ost;
}

