/*
 * Dialog.hpp
 *
 *  Created on: Jan 5, 2010
 *      Author: crueger
 */

#ifndef DIALOG_HPP_
#define DIALOG_HPP_

#include<string>
#include<list>

class MoleculeListClass;
class molecule;
class Vector;
class element;


/** Dialog is one of the two main classes of the UIFactory base class.
 *
 * The Dialog is meant for asking the user for information needed to perform actions he
 * desires, such as asking for a position in space or a length.
 *
 * For this purpose there is the base class Query and numerous specializations for each
 * of the types to be asked. There are primitives integer, doubles and string, but also
 * advanced types such as element, molecule or Vector. There is also an empty query for
 * displaying text.
 */
class Dialog
{
public:
  Dialog();
  virtual ~Dialog();

  virtual void queryEmpty(const char *, std::string = "")=0;
  virtual void queryInt(const char *, int *, std::string = "")=0;
  virtual void queryDouble(const char*,double *, std::string = "")=0;
  virtual void queryString(const char*, std::string *, std::string = "")=0;
  virtual void queryMolecule(const char*,molecule**,MoleculeListClass*, std::string = "")=0;
  virtual void queryVector(const char*,Vector *,const double *const,bool, std::string = "")=0;
  virtual void queryElement(const char*,const element **, std::string = "")=0;

  virtual bool display();

protected:
  // methodology for handling queries
  // all queries are stored and then performed at appropriate times

  //these queries can be handled by this dialog

  //TODO: Find a way to reduce complexity...
  //needs O(N*M) query classes, where N is the number of query types and M is the number of GUIs
  //usual approach for reducing inheritance complexity (strategy pattern) does not work,
  //due to lack of common code for query types as well as GUI-Types (all subtypes differ a lot)

  //base class for all queries
  class Query {
  public:
    Query(std::string _title, std::string _description = "");
    virtual ~Query();
    virtual bool handle()=0;
    virtual void setResult()=0;
  protected:
    const std::string getTitle() const;
    const std::string getDescription() const;
  private:
    std::string title;  //!< short title of the query
    std::string description; //!< longer description for tooltips or for help
  };

  // Empty Query is just meant for showing text, such as version, help, initial message or alike
  class EmptyQuery : public Query {
  public:
    EmptyQuery(std::string title, std::string _description = "");
    virtual ~EmptyQuery();
    virtual bool handle()=0;
    virtual void setResult();
  };

  //Specialized classes for certain types. GUI-Types are not specialized at this time
  class IntQuery : public Query {
  public:
    IntQuery(std::string title,int *_target, std::string _description = "");
    virtual ~IntQuery();
    virtual bool handle()=0;
    virtual void setResult();
  protected:
    int tmp;
  private:
    int *target;
  };

  class DoubleQuery : public Query {
  public:
    DoubleQuery(std::string title,double *_target, std::string _description = "");
    virtual ~DoubleQuery();
    virtual bool handle()=0;
    virtual void setResult();
  protected:
    double tmp;
  private:
    double *target;
  };

  class StringQuery : public Query {
  public:
    StringQuery(std::string title,std::string *_target, std::string _description = "");
    virtual ~StringQuery();
    virtual bool handle()=0;
    virtual void setResult();
  protected:
    std::string tmp;
  private:
    std::string *target;
  };


  class MoleculeQuery : public Query {
  public:
    MoleculeQuery(std::string title, molecule **_target, MoleculeListClass *_molecules, std::string _description = "");
    virtual ~MoleculeQuery();
    virtual bool handle()=0;
    virtual void setResult();
  protected:
    molecule *tmp;
    MoleculeListClass *molecules;
  private:
    molecule **target;
  };

  class VectorQuery : public Query {
  public:
      VectorQuery(std::string title,Vector *_target,const double *const _cellSize,bool _check, std::string _description = "");
      virtual ~VectorQuery();
      virtual bool handle()=0;
      virtual void setResult();
    protected:
      Vector *tmp;
      const double *const cellSize;
      bool check;
    private:
      Vector *target;
  };

  class ElementQuery : public Query {
  public:
    ElementQuery(std::string title, const element**_target, std::string _description = "");
    virtual ~ElementQuery();
    virtual bool handle()=0;
    virtual void setResult();
  protected:
    const element *tmp;
  private:
    const element **target;
  };

void registerQuery(Query* query);

private:
  std::list<Query*> queries;

};


#endif /* DIALOG_HPP_ */
