/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2010-2012 University of Bonn. All rights reserved.
 * 
 *
 *   This file is part of MoleCuilder.
 *
 *    MoleCuilder is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    MoleCuilder is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with MoleCuilder.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * FragmentationAction.cpp
 *
 *  Created on: May 9, 2010
 *      Author: heber
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "CodePatterns/MemDebug.hpp"

#include "Atom/atom.hpp"
#include "CodePatterns/Log.hpp"
#include "Fragmentation/Exporters/ExportGraph_ToFiles.hpp"
#include "Fragmentation/Fragmentation.hpp"
#include "Fragmentation/HydrogenSaturation_enum.hpp"
#include "Graph/CheckAgainstAdjacencyFile.hpp"
#include "Graph/DepthFirstSearchAnalysis.hpp"
#include "Helpers/defs.hpp"
#include "molecule.hpp"
#include "World.hpp"

#include <iostream>
#include <map>
#include <string>
#include <vector>

#include "Actions/FragmentationAction/FragmentationAction.hpp"

using namespace MoleCuilder;

// and construct the stuff
#include "FragmentationAction.def"
#include "Action_impl_pre.hpp"
/** =========== define the function ====================== */
Action::state_ptr FragmentationFragmentationAction::performCall() {
  clock_t start,end;
  int ExitFlag = -1;
  World &world = World::getInstance();

  // inform about used parameters
  LOG(0, "STATUS: Fragmenting molecular system with current connection matrix maximum bond distance "
      << params.distance.get() << " up to "
      << params.order.get() << " order. Fragment files begin with "
      << params.prefix.get() << " and are stored as: "
      << params.types.get() << "." << std::endl);

  // check for selected atoms
  if (world.beginAtomSelection() == world.endAtomSelection()) {
    ELOG(1, "There are not atoms selected for fragmentation.");
    return Action::failure;
  }

  // go through all atoms, note down their molecules and group them
  typedef std::multimap<molecule *, atom *> clusters_t;
  clusters_t clusters;
  for (World::AtomSelectionConstIterator iter = world.beginAtomSelection();
      iter != world.endAtomSelection(); ++iter) {
    clusters.insert( std::make_pair(iter->second->getMolecule(), iter->second) );
  }
  LOG(1, "INFO: There are " << clusters.size()  << " molecules to consider.");

  // parse in Adjacency file
  std::ifstream File;
  std::string filename;
  filename = params.prefix.get() + ADJACENCYFILE;
  File.open(filename.c_str(), ios::out);
  CheckAgainstAdjacencyFile FileChecker(File);

  DepthFirstSearchAnalysis DFS;
  start = clock();
  // go through all keys (i.e. all molecules)
  clusters_t::const_iterator advanceiter;
  for (clusters_t::const_iterator iter = clusters.begin();
      iter != clusters.end();
      iter = advanceiter) {
    // get iterator to past last atom in this molecule
    molecule * mol = iter->first;
    advanceiter = clusters.upper_bound(mol);

    // copy molecule's atoms' ids as parameters to Fragmentation's AtomMask
    std::vector<atomId_t> mols_atomids;
    std::transform(iter, advanceiter, std::back_inserter(mols_atomids),
        boost::bind( &atom::getNr,
            boost::bind( &clusters_t::value_type::second, _1 ))
    );
    LOG(2, "INFO: Fragmenting in molecule " << mol->getName() << " in " << clusters.count(mol) 
        << " atoms, out of " << mol->getAtomCount() << ".");
    const enum HydrogenSaturation saturation =  params.DoSaturation.get() ? DoSaturate : DontSaturate;
    Fragmentation Fragmenter(mol, FileChecker, saturation);

    // perform fragmentation
    LOG(0, std::endl << " ========== Fragmentation of molecule " << mol->getName() << " ========================= ");
    {
      const int tempFlag = Fragmenter.FragmentMolecule(mols_atomids, params.order.get(), params.prefix.get(), DFS);
      if ((ExitFlag == 2) && (tempFlag != 2))
        ExitFlag = tempFlag; // if there is one molecule that needs further fragmentation, it overrides others
      if (ExitFlag == -1)
        ExitFlag = tempFlag; // if we are the first, we set the standard
    }

    // store molecule's fragment to file
    ExportGraph_ToFiles exporter(Fragmenter.getGraph(), mol, saturation);
    exporter.setPrefix(params.prefix.get());
    exporter.setOutputTypes(params.types.get());
    exporter();
  }
  World::getInstance().setExitFlag(ExitFlag);
  end = clock();
  LOG(0, "STATUS: Clocks for this operation: " << (end-start) << ", time: " << ((double)(end-start)/CLOCKS_PER_SEC) << "s.");


  return Action::success;
}

Action::state_ptr FragmentationFragmentationAction::performUndo(Action::state_ptr _state) {
  return Action::success;
}

Action::state_ptr FragmentationFragmentationAction::performRedo(Action::state_ptr _state){
  return Action::success;
}

bool FragmentationFragmentationAction::canUndo() {
  return true;
}

bool FragmentationFragmentationAction::shouldUndo() {
  return true;
}
/** =========== end of function ====================== */
