/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2010-2012 University of Bonn. All rights reserved.
 *
 *
 *   This file is part of MoleCuilder.
 *
 *    MoleCuilder is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    MoleCuilder is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with MoleCuilder.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * CombineShapesAction.cpp
 *
 *  Created on: Oct 1, 2012
 *      Author: ankele
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "CodePatterns/MemDebug.hpp"

#include "Shapes/BaseShapes.hpp"
#include "Shapes/ShapeFactory.hpp"
#include "Shapes/ShapeRegistry.hpp"
#include "CodePatterns/Log.hpp"
#include "CodePatterns/Verbose.hpp"

#include <iostream>
#include <string>

#include "CombineShapesAction.hpp"

using namespace MoleCuilder;

// and construct the stuff
#include "CombineShapesAction.def"
#include "Action_impl_pre.hpp"
/** =========== define the function ====================== */
ActionState::ptr ShapeCombineShapesAction::performCall() {
  LOG(1, "Combining shapes in ShapeRegistry.");

  std::vector<Shape*> selectedShapes = ShapeRegistry::getInstance().getSelectedShapes();
  Shape s = Sphere();

  std::string op = params.shape_op.get();
  if (op == "AND"){
    if (selectedShapes.size() != 2){
      STATUS("Operation AND requires exactly 2 selected shapes.");
      return Action::failure;
    }
    s = (*selectedShapes[0]) && (*selectedShapes[1]);
  }else if (op == "OR"){
    if (selectedShapes.size() != 2){
      STATUS("Operation OR requires exactly 2 selected shapes.");
      return Action::failure;
    }
    s = (*selectedShapes[0]) || (*selectedShapes[1]);
  }else if (op == "NOT"){
    if (selectedShapes.size() != 1){
      STATUS("Operation NOT requires exactly 1 selected shape.");
      return Action::failure;
    }
    s = ! (*selectedShapes[0]);
  }else{
    ASSERT(0, "ShapeCombineShapesAction::performCall() - unhandled shape operation");
  }

  if (params.shape_name.get() != "")
    s.setName(params.shape_name.get());
  else {
    const std::string baseName = 
        ShapeFactory::getInstance().getShapeName(s.getType());
    s.setName(
        ShapeRegistry::getInstance().getDefaultNameForShape(baseName)
    );
  }
  std::string old_name = s.getName();
  std::cout << s.getName() << "\n";
  ShapeRegistry::getInstance().addShape(s);

  return ActionState::ptr(new ShapeCombineShapesState(old_name, params));
}

ActionState::ptr ShapeCombineShapesAction::performUndo(ActionState::ptr _state) {
  ShapeCombineShapesState *state = assert_cast<ShapeCombineShapesState*>(_state.get());

  LOG(1, "INFO: Removing '" << state->old_name << "' from ShapeRegistry.");

  std::cout << state->old_name << "\n";
  ShapeRegistry::getInstance().removeShape(state->old_name);

  return ActionState::ptr(_state);
}

ActionState::ptr ShapeCombineShapesAction::performRedo(ActionState::ptr _state){
  ShapeCombineShapesState *state = assert_cast<ShapeCombineShapesState*>(_state.get());

  std::vector<Shape*> selectedShapes = ShapeRegistry::getInstance().getSelectedShapes();
  Shape s = Sphere();

  std::string op = params.shape_op.get();
  if (op == "AND"){
    s = (*selectedShapes[0]) && (*selectedShapes[1]);
  }else if (op == "OR"){
    s = (*selectedShapes[0]) || (*selectedShapes[1]);
  }else if (op == "NOT"){
    s = ! (*selectedShapes[0]);
  }else{
    ASSERT(0, "ShapeCombineShapesAction::performCall() - unhandled shape operation");
  }

  s.setName(state->old_name);
  ShapeRegistry::getInstance().addShape(s);

  return ActionState::ptr(_state);
}

bool ShapeCombineShapesAction::canUndo() {
  return true;
}

bool ShapeCombineShapesAction::shouldUndo() {
  return true;
}
/** =========== end of function ====================== */
