/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2010-2012 University of Bonn. All rights reserved.
 * 
 *
 *   This file is part of MoleCuilder.
 *
 *    MoleCuilder is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    MoleCuilder is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with MoleCuilder.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * NotAtomByIdAction.cpp
 *
 *  Created on: May 12, 2010
 *      Author: heber
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "CodePatterns/MemDebug.hpp"

#include "Atom/atom.hpp"
#include "CodePatterns/Log.hpp"
#include "CodePatterns/Verbose.hpp"
#include "Descriptors/AtomIdDescriptor.hpp"
#include "World.hpp"

#include <iostream>
#include <string>

#include "NotAtomByIdAction.hpp"

using namespace MoleCuilder;

// and construct the stuff
#include "NotAtomByIdAction.def"
#include "Action_impl_pre.hpp"
/** =========== define the function ====================== */
ActionState::ptr SelectionNotAtomByIdAction::performCall()
{
  enum Sucess {
    NoStatus,
    AllAtomsSelected,
    AtomsUnselected,
    AtomMissing
  } status = NoStatus;

  const atomids_t atomids = params.atomids.get();
  atomids_t undoatomids;
  undoatomids.reserve(atomids.size());
  for (atomids_t::const_iterator iter = atomids.begin(); iter != atomids.end(); ++iter) {
    const atom *Walker = World::getInstance().getAtom(AtomById(*iter));
    if (Walker != NULL) {
      if (World::getInstance().isSelected(Walker)) {
        LOG(1, "Unselecting atom " << *Walker);
        World::getInstance().unselectAtom(Walker);
        undoatomids.push_back(*iter);
        if (status < AtomMissing)
          status = AtomsUnselected;
      } else {
        if (status == NoStatus)
          status = AllAtomsSelected;
      }
    } else {
      status = AtomMissing;
    }
  }
  LOG(0, World::getInstance().countSelectedAtoms() << " atoms remain selected.");

  switch (status) {
    case AtomMissing:
      STATUS("Cannot find all atoms by given ids.");
      return Action::failure;
      break;
    case AllAtomsSelected:
    case AtomsUnselected:
      return ActionState::ptr(new SelectionNotAtomByIdState(undoatomids, params));
      break;
    default:
      ASSERT(0, "SelectionAtomByIdAction::performCall() - this must not happen.");
      return Action::failure;
      break;
  }
  return Action::failure;
}

ActionState::ptr SelectionNotAtomByIdAction::performUndo(ActionState::ptr _state) {
  SelectionNotAtomByIdState *state = assert_cast<SelectionNotAtomByIdState*>(_state.get());

  for (atomids_t::const_iterator iter = state->undoatomids.begin();
      iter != state->undoatomids.end(); ++iter) {
    const atom *Walker = World::getInstance().getAtom(AtomById(*iter));
    World::getInstance().selectAtom(Walker);
  }

  return ActionState::ptr(_state);
}

ActionState::ptr SelectionNotAtomByIdAction::performRedo(ActionState::ptr _state){
  SelectionNotAtomByIdState *state = assert_cast<SelectionNotAtomByIdState*>(_state.get());

  for (atomids_t::const_iterator iter = state->undoatomids.begin();
      iter != state->undoatomids.end(); ++iter) {
    const atom *Walker = World::getInstance().getAtom(AtomById(*iter));
    World::getInstance().unselectAtom(Walker);
  }

  return ActionState::ptr(_state);
}

bool SelectionNotAtomByIdAction::canUndo() {
  return true;
}

bool SelectionNotAtomByIdAction::shouldUndo() {
  return true;
}
/** =========== end of function ====================== */
