/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2010-2012 University of Bonn. All rights reserved.
 * 
 *
 *   This file is part of MoleCuilder.
 *
 *    MoleCuilder is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    MoleCuilder is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with MoleCuilder.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * NotAllAtomsInsideVolumeAction.cpp
 *
 *  Created on: Aug 9, 2010
 *      Author: heber
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "CodePatterns/MemDebug.hpp"

#include "Descriptors/AtomSelectionDescriptor.hpp"
#include "Descriptors/AtomShapeDescriptor.hpp"
#include "Atom/atom.hpp"
#include "CodePatterns/Log.hpp"
#include "CodePatterns/Verbose.hpp"
#include "LinearAlgebra/Vector.hpp"
#include "Shapes/ShapeRegistry.hpp"
#include "World.hpp"

#include <boost/foreach.hpp>
#include <iostream>
#include <string>

#include "NotAllAtomsInsideVolumeAction.hpp"

using namespace MoleCuilder;

// and construct the stuff
#include "NotAllAtomsInsideVolumeAction.def"
#include "Action_impl_pre.hpp"
/** =========== define the function ====================== */
ActionState::ptr SelectionNotAllAtomsInsideVolumeAction::performCall() {
  LOG(1, "Unselecting all atoms inside a volume.");
  std::vector<Shape*> selectedShapes = ShapeRegistry::getInstance().getSelectedShapes();
  if (selectedShapes.size() != 1){
    STATUS("There has to be exactly 1 selected shape.");
    return Action::failure;
  }
  std::vector<atom *> unselectedAtoms = World::getInstance().getAllAtoms((!AtomsBySelection()) && AtomsByShape(*selectedShapes[0]));
  World::getInstance().unselectAllAtoms(AtomsByShape(*selectedShapes[0]));
  LOG(0, World::getInstance().countSelectedAtoms() << " atoms remain selected.");
  return ActionState::ptr(new SelectionNotAllAtomsInsideVolumeState(unselectedAtoms, params));
}

ActionState::ptr SelectionNotAllAtomsInsideVolumeAction::performUndo(ActionState::ptr _state) {
  SelectionNotAllAtomsInsideVolumeState *state = assert_cast<SelectionNotAllAtomsInsideVolumeState*>(_state.get());

  std::vector<Shape*> selectedShapes = ShapeRegistry::getInstance().getSelectedShapes();
  World::getInstance().selectAllAtoms(AtomsByShape(*selectedShapes[0]));
  BOOST_FOREACH(atom *_atom, state->unselectedAtoms)
    World::getInstance().unselectAtom(_atom);

  return ActionState::ptr(_state);
}

ActionState::ptr SelectionNotAllAtomsInsideVolumeAction::performRedo(ActionState::ptr _state){
  //SelectionNotAllAtomsInsideVolumeState *state = assert_cast<SelectionNotAllAtomsInsideVolumeState*>(_state.get());

  std::vector<Shape*> selectedShapes = ShapeRegistry::getInstance().getSelectedShapes();
  World::getInstance().unselectAllAtoms(AtomsByShape(*selectedShapes[0]));

  return ActionState::ptr(_state);
}

bool SelectionNotAllAtomsInsideVolumeAction::canUndo() {
  return true;
}

bool SelectionNotAllAtomsInsideVolumeAction::shouldUndo() {
  return true;
}
/** =========== end of function ====================== */
