/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2010-2012 University of Bonn. All rights reserved.
 * 
 *
 *   This file is part of MoleCuilder.
 *
 *    MoleCuilder is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    MoleCuilder is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with MoleCuilder.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * AtomByIdAction.cpp
 *
 *  Created on: May 12, 2010
 *      Author: heber
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "CodePatterns/MemDebug.hpp"

#include "Atom/atom.hpp"
#include "CodePatterns/Log.hpp"
#include "CodePatterns/Verbose.hpp"
#include "World.hpp"
#include "Descriptors/AtomIdDescriptor.hpp"

#include <iostream>
#include <string>

#include "AtomByIdAction.hpp"

using namespace MoleCuilder;

// and construct the stuff
#include "AtomByIdAction.def"
#include "Action_impl_pre.hpp"
/** =========== define the function ====================== */
ActionState::ptr SelectionAtomByIdAction::performCall()
{
  enum Sucess {
    NoStatus,
    AllAtomsUnselected,
    AtomsSelected,
    AtomMissing
  } status = NoStatus;

  const atomids_t atomids = params.atomids.get();
  atomids_t undoatomids;
  undoatomids.reserve(atomids.size());
  for (atomids_t::const_iterator iter = atomids.begin(); iter != atomids.end(); ++iter) {
    const atom *Walker = World::getInstance().getAtom(AtomById(*iter));
    if (Walker != NULL) {
      if (!World::getInstance().isSelected(Walker)) {
        LOG(1, "Selecting atom " << *Walker);
        World::getInstance().selectAtom(Walker);
        undoatomids.push_back(*iter);
        if (status < AtomMissing)
          status = AtomsSelected;
      } else {
        if (status == NoStatus)
          status = AllAtomsUnselected;
      }
    } else {
      status = AtomMissing;
    }
  }
  LOG(0, World::getInstance().countSelectedAtoms() << " atoms selected.");

  switch (status) {
    case AtomMissing:
      STATUS("Cannot find all atoms with given ids.");
      return Action::failure;
      break;
    case AllAtomsUnselected:
    case AtomsSelected:
      return ActionState::ptr(new SelectionAtomByIdState(undoatomids, params));
      break;
    default:
      ASSERT(0, "SelectionAtomByIdAction::performCall() - this must not happen.");
      return Action::failure;
      break;
  }
  return Action::failure;
}

ActionState::ptr SelectionAtomByIdAction::performUndo(ActionState::ptr _state) {
  SelectionAtomByIdState *state = assert_cast<SelectionAtomByIdState*>(_state.get());

  for (atomids_t::const_iterator iter = state->undoatomids.begin();
      iter != state->undoatomids.end(); ++iter) {
    const atom *Walker = World::getInstance().getAtom(AtomById(*iter));
    World::getInstance().unselectAtom(Walker);
  }

  return ActionState::ptr(_state);
}

ActionState::ptr SelectionAtomByIdAction::performRedo(ActionState::ptr _state){
  SelectionAtomByIdState *state = assert_cast<SelectionAtomByIdState*>(_state.get());

  for (atomids_t::const_iterator iter = state->undoatomids.begin();
      iter != state->undoatomids.end(); ++iter) {
    const atom *Walker = World::getInstance().getAtom(AtomById(*iter));
    World::getInstance().selectAtom(Walker);
  }

  return ActionState::ptr(_state);
}

bool SelectionAtomByIdAction::canUndo() {
  return true;
}

bool SelectionAtomByIdAction::shouldUndo() {
  return true;
}
/** =========== end of function ====================== */
