/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2010-2012 University of Bonn. All rights reserved.
 * 
 *
 *   This file is part of MoleCuilder.
 *
 *    MoleCuilder is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    MoleCuilder is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with MoleCuilder.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * RotateAroundSelfByAngleAction.cpp
 *
 *  Created on: Aug 06, 2010
 *      Author: heber
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "CodePatterns/MemDebug.hpp"

#include "CodePatterns/Log.hpp"
#include "CodePatterns/Verbose.hpp"
#include "LinearAlgebra/Line.hpp"
#include "LinearAlgebra/Vector.hpp"
#include "Atom/atom.hpp"
#include "molecule.hpp"

#include <iostream>
#include <fstream>
#include <string>

#include "Actions/MoleculeAction/RotateAroundSelfByAngleAction.hpp"

using namespace MoleCuilder;

// and construct the stuff
#include "RotateAroundSelfByAngleAction.def"
#include "Action_impl_pre.hpp"
/** =========== define the function ====================== */
ActionState::ptr MoleculeRotateAroundSelfByAngleAction::performCall() {
  // check whether a molecule is selected
  std::vector<molecule *> selectedMolecules = World::getInstance().getSelectedMolecules();
  if (selectedMolecules.size() == 0) {
    STATUS("There are need to be some molecules selected.");
    return Action::failure;
  }

  // go through all selected molecules
  BOOST_FOREACH(molecule *mol, selectedMolecules) {
    // check whether Axis is valid
    if (params.Axis.get().IsZero()) {
      STATUS("Rotation Axis must not be zero.");
      return Action::failure;
    }

    // convert from degrees to radian
    params.angle.set(params.angle.get() * M_PI/180.);

    // Creation Line that is the rotation axis
    Vector *CenterOfGravity = mol->DetermineCenterOfGravity();
    LOG(0,  "Center of gravity is " << *CenterOfGravity << ".");
    Line RotationAxis(*CenterOfGravity, params.Axis.get());
    delete(CenterOfGravity);
    LOG(0, "Rotate " << mol->getName() << " around self by " << params.angle.get() << " radian around axis " << RotationAxis << ".");

    for (molecule::iterator iter = mol->begin(); iter != mol->end(); ++iter) {
      (*iter)->setPosition(RotationAxis.rotateVector((*iter)->getPosition(), params.angle.get()));
    }
    LOG(0, "done.");
  }

  return ActionState::ptr(new MoleculeRotateAroundSelfByAngleState(selectedMolecules, params));
}

ActionState::ptr MoleculeRotateAroundSelfByAngleAction::performUndo(ActionState::ptr _state) {
  MoleculeRotateAroundSelfByAngleState *state = assert_cast<MoleculeRotateAroundSelfByAngleState*>(_state.get());

  BOOST_FOREACH(molecule *mol, state->selectedMolecules) {
    Vector *CenterOfGravity = mol->DetermineCenterOfGravity();
    LOG(0,  "Center of gravity is " << *CenterOfGravity << ".");
    Line RotationAxis(*CenterOfGravity, state->params.Axis.get());
    delete(CenterOfGravity);
    LOG(0, "Rotate " << mol->getName() << " around self by " << -state->params.angle.get() << " radian around axis " << RotationAxis << ".");

    for (molecule::iterator iter = mol->begin(); iter != mol->end(); ++iter) {
      (*iter)->setPosition(RotationAxis.rotateVector((*iter)->getPosition(), -state->params.angle.get()));
    }
  }

  return ActionState::ptr(_state);
}

ActionState::ptr MoleculeRotateAroundSelfByAngleAction::performRedo(ActionState::ptr _state){
  MoleculeRotateAroundSelfByAngleState *state = assert_cast<MoleculeRotateAroundSelfByAngleState*>(_state.get());

  BOOST_FOREACH(molecule *mol, state->selectedMolecules) {
    Vector *CenterOfGravity = mol->DetermineCenterOfGravity();
    LOG(0,  "Center of gravity is " << *CenterOfGravity << ".");
    Line RotationAxis(*CenterOfGravity, state->params.Axis.get());
    delete(CenterOfGravity);
    LOG(0, "Rotate " << mol->getName() << " around self by " << state->params.angle.get() << " radian around axis " << RotationAxis << ".");

    for (molecule::iterator iter = mol->begin(); iter != mol->end(); ++iter) {
      (*iter)->setPosition(RotationAxis.rotateVector((*iter)->getPosition(), state->params.angle.get()));
    }
  }

  return ActionState::ptr(_state);
}

bool MoleculeRotateAroundSelfByAngleAction::canUndo() {
  return true;
}

bool MoleculeRotateAroundSelfByAngleAction::shouldUndo() {
  return true;
}
/** =========== end of function ====================== */
