/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2010-2012 University of Bonn. All rights reserved.
 * 
 *
 *   This file is part of MoleCuilder.
 *
 *    MoleCuilder is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    MoleCuilder is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with MoleCuilder.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * LoadAction.cpp
 *
 *  Created on: May 8, 2010
 *      Author: heber
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "CodePatterns/MemDebug.hpp"

#include "CodePatterns/Log.hpp"
#include "CodePatterns/Verbose.hpp"
#include "Descriptors/MoleculeIdDescriptor.hpp"
#include "Parser/FormatParserInterface.hpp"
#include "Parser/FormatParserStorage.hpp"
#include "Parser/FormatParser_Parameters.hpp"
#include "molecule.hpp"
#include "MoleculeListClass.hpp"
#include "World.hpp"

#include <iostream>

#include <boost/filesystem/fstream.hpp>

#include "LoadAction.hpp"

using namespace MoleCuilder;

// and construct the stuff
#include "LoadAction.def"
#include "Action_impl_pre.hpp"
/** =========== define the function ====================== */
ActionState::ptr MoleculeLoadAction::performCall() {
  // parsing file if present
  if (!boost::filesystem::exists(params.filename.get())) {
    STATUS("Specified input file "+params.filename.get().string()+" not found.");
    return Action::failure;
  } else {
    LOG(1, "Specified input file found, parsing ... ");

    // extract suffix
    std::string FilenameSuffix;
    std::string FilenamePrefix;
    if (params.filename.get().has_filename()) {
      // get suffix
      FilenameSuffix = params.filename.get().extension().string().substr(1); // remove the prefixed "."
      FilenamePrefix = params.filename.get().stem().string();
    } else {
      STATUS("Input file does not have a suffix, cannot recognize format.");
      return Action::failure;
    }

    // get undo state for parser
    enum ParserTypes type = FormatParserStorage::getInstance().getTypeFromSuffix(FilenameSuffix);
    ASSERT(type != ParserTypes_end,
        "MoleculeLoadAction::performCall() - unknown file suffix "+FilenameSuffix+".");
    FormatParser_Parameters *ParserParams = FormatParserStorage::getInstance().get(type).parameters;
    if (ParserParams != NULL)
      ParserParams = ParserParams->clone();
    else
      ParserParams = NULL;

    // parse the file
    boost::filesystem::ifstream input;
    input.open(params.filename.get());
    FormatParserStorage::getInstance().load(input, FilenameSuffix);
    input.close();

    // set file name of last molecule
    MoleculeList::const_reverse_iterator iter = World::getInstance().getMolecules()->ListOfMolecules.rbegin();
    (*iter)->SetNameFromFilename(FilenamePrefix.c_str());
    LOG(0, "Chemical formula is " << (*iter)->getFormula());

    return ActionState::ptr(
        new MoleculeLoadState((*iter)->getId(),FilenamePrefix,FilenameSuffix,boost::shared_ptr<FormatParser_Parameters>(ParserParams),params)
    );
  }
}

ActionState::ptr MoleculeLoadAction::performUndo(ActionState::ptr _state) {
  MoleculeLoadState *state = assert_cast<MoleculeLoadState*>(_state.get());

  // remove loaded molecule and its atoms
  molecule *mol = World::getInstance().getMolecule(MoleculeById(state->molId));
  for(molecule::iterator iter = mol->begin(); !mol->empty(); iter = mol->begin())
    World::getInstance().destroyAtom(*iter);
  //World::getInstance().destroyMolecule(mol);

  // undo changes to FormatParser
  enum ParserTypes type = FormatParserStorage::getInstance().getTypeFromSuffix(state->FilenameSuffix);
  FormatParser_Parameters *ParserParams = FormatParserStorage::getInstance().get(type).parameters;
  if (ParserParams != NULL)
    ParserParams->makeClone(*state->ParserParameters);

  return ActionState::ptr(_state);
}

ActionState::ptr MoleculeLoadAction::performRedo(ActionState::ptr _state){
  MoleculeLoadState *state = assert_cast<MoleculeLoadState*>(_state.get());

  // get undo state for parser
  enum ParserTypes type = FormatParserStorage::getInstance().getTypeFromSuffix(state->FilenameSuffix);
  FormatParser_Parameters *ParserParams = FormatParserStorage::getInstance().get(type).parameters;
  if (ParserParams != NULL)
    ParserParams = ParserParams->clone();
  else
    ParserParams = NULL;

  // parse the file
  boost::filesystem::ifstream input;
  input.open(state->params.filename.get());
  FormatParserStorage::getInstance().load(input, state->FilenameSuffix);
  input.close();

  // set file name of last molecule
  MoleculeList::const_reverse_iterator iter = World::getInstance().getMolecules()->ListOfMolecules.rbegin();
  (*iter)->SetNameFromFilename(state->FilenamePrefix.c_str());
  (*iter)->setId(state->molId);
  LOG(0, "Chemical formula is " << (*iter)->getFormula());

  return ActionState::ptr(
      new MoleculeLoadState((*iter)->getId(),state->FilenamePrefix,state->FilenameSuffix,boost::shared_ptr<FormatParser_Parameters>(ParserParams),params)
  );
}

bool MoleculeLoadAction::canUndo() {
  return true;
}

bool MoleculeLoadAction::shouldUndo() {
  return true;
}
/** =========== end of function ====================== */
