/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2010-2012 University of Bonn. All rights reserved.
 * Copyright (C)  2013 Frederik Heber. All rights reserved.
 * 
 *
 *   This file is part of MoleCuilder.
 *
 *    MoleCuilder is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    MoleCuilder is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with MoleCuilder.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * UpdateMoleculesAction.cpp
 *
 *  Created on: May 9, 2010
 *      Author: heber
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "CodePatterns/MemDebug.hpp"

#include "Descriptors/AtomIdDescriptor.hpp"
#include "Descriptors/MoleculeDescriptor.hpp"

#include "Atom/atom.hpp"
#include "Bond/bond.hpp"
#include "CodePatterns/Log.hpp"
#include "CodePatterns/Verbose.hpp"
#include "Graph/BondGraph.hpp"
#include "Graph/DepthFirstSearchAnalysis.hpp"
#include "molecule.hpp"
#include "MoleculeListClass.hpp"
#include "World.hpp"

#include <iostream>
#include <string>

typedef std::map< moleculeId_t, std::vector<atomId_t> > MolAtomList;
typedef std::map< atomId_t, atomId_t > AtomAtomList;

#include "Actions/GraphAction/UpdateMoleculesAction.hpp"

using namespace MoleCuilder;

// and construct the stuff
#include "UpdateMoleculesAction.def"
#include "Action_impl_pre.hpp"
/** =========== define the function ====================== */
ActionState::ptr GraphUpdateMoleculesAction::performCall() {
  // first create stuff for undo state
  LOG(0, "STATUS: Creating undo state.");
  MolAtomList moleculelist;
  vector<molecule *> allmolecules = World::getInstance().getAllMolecules();
  for (vector<molecule *>::const_iterator moliter = allmolecules.begin(); moliter != allmolecules.end(); ++moliter) {
    std::vector<atomId_t> atomlist;
    atomlist.resize((*moliter)->size());
    for (molecule::const_iterator atomiter = (*moliter)->begin(); atomiter != (*moliter)->end(); ++atomiter) {
      atomlist.push_back((*atomiter)->getId());
    }
    moleculelist.insert( std::pair< moleculeId_t, std::vector<atomId_t> > ((*moliter)->getId(), atomlist));
  }
  GraphUpdateMoleculesState *UndoState = new GraphUpdateMoleculesState(moleculelist, params);

  // 0a. remove all present molecules
  LOG(0, "STATUS: Removing all present molecules.");
  MoleculeListClass *molecules = World::getInstance().getMolecules();
  for (vector<molecule *>::iterator MolRunner = allmolecules.begin(); MolRunner != allmolecules.end(); ++MolRunner) {
    molecules->erase(*MolRunner);
    World::getInstance().destroyMolecule(*MolRunner);
  }

  // 2. scan for connected subgraphs
  DepthFirstSearchAnalysis DFS;
  DFS();
  DFS.UpdateMoleculeStructure();
  if (World::getInstance().numMolecules() == 0) {
    //World::getInstance().destroyMolecule(mol);
    STATUS("There are no molecules.");
    return Action::failure;
  }

  LOG(1, "I scanned " << World::getInstance().numMolecules() << " molecules.");

  return ActionState::ptr(UndoState);
}

ActionState::ptr GraphUpdateMoleculesAction::performUndo(ActionState::ptr _state) {
  GraphUpdateMoleculesState *state = assert_cast<GraphUpdateMoleculesState*>(_state.get());

  {
    // remove all present molecules
    MoleculeListClass *molecules = World::getInstance().getMolecules();
    vector<molecule *> allmolecules = World::getInstance().getAllMolecules();
    for (vector<molecule *>::iterator MolRunner = allmolecules.begin(); MolRunner != allmolecules.end(); ++MolRunner) {
      molecules->erase(*MolRunner);
      World::getInstance().destroyMolecule(*MolRunner);
    }
  }

  {
    // construct the old state
    MoleculeListClass *molecules = World::getInstance().getMolecules();
    molecule *mol = NULL;
    for (MolAtomList::const_iterator iter = state->moleculelist.begin(); iter != state->moleculelist.end(); ++iter) {
      mol = World::getInstance().createMolecule();
      if (mol->getId() != (*iter).first)
        World::getInstance().changeMoleculeId(mol->getId(), (*iter).first);
      for (std::vector<atomId_t>::const_iterator atomiter = (*iter).second.begin(); atomiter != (*iter).second.end(); ++atomiter) {
        atom *Walker = World::getInstance().getAtom(AtomById(*atomiter));
        mol->AddAtom(Walker);
      }
      molecules->insert(mol);
    }
  }

  return ActionState::ptr(_state);
}

ActionState::ptr GraphUpdateMoleculesAction::performRedo(ActionState::ptr _state){
  return performCall();
}

bool GraphUpdateMoleculesAction::canUndo() {
  return true;
}

bool GraphUpdateMoleculesAction::shouldUndo() {
  return true;
}
/** =========== end of function ====================== */
