/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2010-2012 University of Bonn. All rights reserved.
 * 
 *
 *   This file is part of MoleCuilder.
 *
 *    MoleCuilder is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    MoleCuilder is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with MoleCuilder.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * FillRegularGridAction.cpp
 *
 *  Created on: Jan 12, 2012
 *      Author: heber
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "CodePatterns/MemDebug.hpp"

#include "Actions/UndoRedoHelpers.hpp"
#include "Atom/atom.hpp"
#include "Atom/AtomicInfo.hpp"
#include "Atom/CopyAtoms/CopyAtoms_withBonds.hpp"
#include "Bond/BondInfo.hpp"
#include "CodePatterns/Log.hpp"
#include "Descriptors/MoleculeOrderDescriptor.hpp"
#include "Filling/Cluster.hpp"
#include "Filling/Filler.hpp"
#include "Filling/Inserter/Inserter.hpp"
#include "Filling/Inserter/RandomInserter.hpp"
#include "Filling/Mesh/CubeMesh.hpp"
#include "Filling/Predicates/IsInsideSurface_FillPredicate.hpp"
#include "Filling/Predicates/IsVoidNode_FillPredicate.hpp"
#include "Filling/Predicates/Ops_FillPredicate.hpp"
#include "LinkedCell/linkedcell.hpp"
#include "LinkedCell/PointCloudAdaptor.hpp"
#include "molecule.hpp"
#include "MoleculeListClass.hpp"
#include "Parser/FormatParserInterface.hpp"
#include "Parser/FormatParserStorage.hpp"
#include "Shapes/BaseShapes.hpp"
#include "Tesselation/tesselation.hpp"
#include "Tesselation/BoundaryLineSet.hpp"
#include "Tesselation/BoundaryTriangleSet.hpp"
#include "Tesselation/CandidateForTesselation.hpp"
#include "World.hpp"


#include <algorithm>
#include <iostream>
#include <string>
#include <vector>

#include "Actions/FillAction/FillRegularGridAction.hpp"

using namespace MoleCuilder;

// and construct the stuff
#include "FillRegularGridAction.def"
#include "Action_impl_pre.hpp"
/** =========== define the function ====================== */
ActionState::ptr FillRegularGridAction::performCall() {
  typedef std::vector<atom*> AtomVector;

  // get the filler molecule
  std::vector<AtomicInfo> movedatoms;
  const std::vector< molecule *> molecules = World::getInstance().getSelectedMolecules();
  if (molecules.size() != 1) {
    STATUS("No exactly one molecule selected, aborting,");
    return Action::failure;
  }
  molecule *filler = *(molecules.begin());
  for(molecule::const_iterator iter = filler->begin(); iter != filler->end(); ++iter)
    movedatoms.push_back( AtomicInfo(*(*iter)) );
  LOG(1, "INFO: Chosen molecule has " << filler->size() << " atoms.");

  // check for selected molecules and create surfaces from them
  std::vector<atom *> atoms(World::getInstance().getSelectedAtoms());
  FillPredicate * surface_predicate = NULL;
  LinkedCell_deprecated * LC = NULL;
  Tesselation * TesselStruct = NULL;
  if (params.SphereRadius.get() != 0.) {
    if ( atoms.size() == 0) {
      STATUS("You have given a sphere radius "+toString(params.SphereRadius.get())
          +" != 0, but have not select any atoms.");
      return Action::failure;
    }
    // create adaptor for the selected atoms
    PointCloudAdaptor< std::vector<atom *> > cloud(&atoms, std::string("Selected atoms"));

    // create tesselation
    LC = new LinkedCell_deprecated(cloud, 2.*params.SphereRadius.get());
    TesselStruct = new Tesselation;
    (*TesselStruct)(cloud, params.SphereRadius.get());

    // and create predicate
    surface_predicate = new FillPredicate( IsInsideSurface_FillPredicate( *TesselStruct, *LC ) );
  }

  // create predicate, mesh, and filler
  FillRegularGridState *UndoState = NULL;
  bool successflag = false;
  {
    FillPredicate *voidnode_predicate = new FillPredicate(
        IsVoidNode_FillPredicate(
            Sphere(zeroVec, params.mindistance.get())
            )
        );
    FillPredicate Andpredicate = (*voidnode_predicate);
    if (surface_predicate != NULL)
      Andpredicate = (Andpredicate) && !(*surface_predicate);
    Mesh *mesh = new CubeMesh(params.counts.get(), params.offset.get(), World::getInstance().getDomain().getM());
    Inserter *inserter = new Inserter(
        Inserter::impl_ptr(
            new RandomInserter(
                params.RandAtomDisplacement.get(),
                params.RandMoleculeDisplacement.get(),
                params.DoRotate.get())
            )
        );

    // fill
    {
      Filler *fillerFunction = new Filler(*mesh, Andpredicate, *inserter);
      // TODO: When molecule::getBoundingSphere() does not use a sphere anymore,
      // we need to check whether we rotate the molecule randomly. For this to
      // work we need a sphere!
      const Shape s = filler->getBoundingSphere(params.RandAtomDisplacement.get());
      ClusterInterface::Cluster_impl cluster( new Cluster(filler->getAtomIds(), s) );
      CopyAtoms_withBonds copyMethod;
      Filler::ClusterVector_t ClonedClusters;
      successflag = (*fillerFunction)(copyMethod, cluster, ClonedClusters);
      delete fillerFunction;

      // append each cluster's atoms to clonedatoms (however not selected ones)
      std::vector<const atom *> clonedatoms;
      std::vector<AtomicInfo> clonedatominfos;
      for (Filler::ClusterVector_t::const_iterator iter = ClonedClusters.begin();
          iter != ClonedClusters.end(); ++iter) {
        const AtomIdSet &atoms = (*iter)->getAtomIds();
        clonedatoms.reserve(clonedatoms.size()+atoms.size());
        for (AtomIdSet::const_iterator atomiter = atoms.begin(); atomiter != atoms.end(); ++atomiter)
          if (!filler->containsAtom(*atomiter)) {
            clonedatoms.push_back( *atomiter );
            clonedatominfos.push_back( AtomicInfo(*(*atomiter)) );
          }
      }
      std::vector< BondInfo > clonedbonds;
      StoreBondInformationFromAtoms(clonedatoms, clonedbonds);
      LOG(2, "DEBUG: There are " << clonedatominfos.size() << " newly created atoms.");

      if (!successflag) {
        STATUS("Insertion failed, removing inserted clusters, translating original one back");
        RemoveAtomsFromAtomicInfo(clonedatominfos);
        clonedatoms.clear();
        SetAtomsFromAtomicInfo(movedatoms);
      } else {
        std::vector<Vector> MovedToVector(filler->size(), zeroVec);
        std::transform(filler->begin(), filler->end(), MovedToVector.begin(),
            boost::bind(&AtomInfo::getPosition, _1) );
        UndoState = new FillRegularGridState(clonedatominfos,clonedbonds,movedatoms,MovedToVector,params);
      }
    }

    // remove
    delete mesh;
    delete inserter;
    delete voidnode_predicate;
    delete surface_predicate;
    delete LC;
    delete TesselStruct;
  }

  if (successflag)
    return ActionState::ptr(UndoState);
  else {
    return Action::failure;
  }
}

ActionState::ptr FillRegularGridAction::performUndo(ActionState::ptr _state) {
  FillRegularGridState *state = assert_cast<FillRegularGridState*>(_state.get());

  // remove all created atoms
  RemoveAtomsFromAtomicInfo(state->clonedatoms);
  // add the original cluster
  SetAtomsFromAtomicInfo(state->movedatoms);

  return ActionState::ptr(_state);
}

ActionState::ptr FillRegularGridAction::performRedo(ActionState::ptr _state){
  FillRegularGridState *state = assert_cast<FillRegularGridState*>(_state.get());

  // place filler cluster again at new spot
  ResetAtomPosition(state->movedatoms, state->MovedToVector);

  // re-create all clusters
  bool statusflag = AddAtomsFromAtomicInfo(state->clonedatoms);

  // re-create the bonds
  if (statusflag)
    AddBondsFromBondInfo(state->clonedbonds);
  if (statusflag)
    return ActionState::ptr(_state);
  else {
    STATUS("Failed re-adding filled in atoms.");
    return Action::failure;
  }
}

bool FillRegularGridAction::canUndo() {
  return true;
}

bool FillRegularGridAction::shouldUndo() {
  return true;
}
/** =========== end of function ====================== */
