/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2010-2012 University of Bonn. All rights reserved.
 * 
 *
 *   This file is part of MoleCuilder.
 *
 *    MoleCuilder is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    MoleCuilder is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with MoleCuilder.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * BondLengthTableAction.cpp
 *
 *  Created on: May 9, 2010
 *      Author: heber
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

// include headers that implement a archive in simple text format
#include <boost/archive/text_oarchive.hpp>
#include <boost/archive/text_iarchive.hpp>

#include "CodePatterns/MemDebug.hpp"

#include "CodePatterns/Log.hpp"
#include "CodePatterns/Verbose.hpp"
#include "Graph/BondGraph.hpp"
#include "World.hpp"

#include <iostream>
#include <string>

#include "Actions/CommandAction/BondLengthTableAction.hpp"

using namespace MoleCuilder;

// and construct the stuff
#include "BondLengthTableAction.def"
#include "Action_impl_pre.hpp"
/** =========== define the function ====================== */
ActionState::ptr CommandBondLengthTableAction::performCall() {
  ostringstream usage;

  LOG(0, "Using " << params.BondGraphFileName.get() << " as bond length table.");
  BondGraph *&BG = World::getInstance().getBondGraph();

  // create undo state
  std::stringstream undostream;
  boost::archive::text_oarchive oa(undostream);
  oa << BG;
  CommandBondLengthTableState *UndoState =
      new CommandBondLengthTableState(
          undostream.str(),
          params
      );

  BG->CleanupBondLengthTable();
  if ((!params.BondGraphFileName.get().empty())
      && boost::filesystem::exists(params.BondGraphFileName.get())) {
    std::ifstream input(params.BondGraphFileName.get().string().c_str());
    if ((input.good()) && (BG->LoadBondLengthTable(input))) {
      LOG(1, "Bond length table parsed successfully.");
      input.close();
      return ActionState::ptr(UndoState);
    } else {
      STATUS("Bond length table "+params.BondGraphFileName.get().string()
          +" parsing failed.");
      input.close();
    }
  } else {
    STATUS("Bond length table "+params.BondGraphFileName.get().string()
        +" loading failed.");
  }
  // recover bond graph
  boost::archive::text_iarchive ia(undostream);
  delete BG;
  ia >> BG;
  delete UndoState;
  return Action::failure;
}

ActionState::ptr CommandBondLengthTableAction::performUndo(ActionState::ptr _state) {
  CommandBondLengthTableState *state = assert_cast<CommandBondLengthTableState*>(_state.get());

  BondGraph *BG;
  std::stringstream undostream(state->undostring);
  boost::archive::text_iarchive ia(undostream);
  ia >> BG;
  World::getInstance().setBondGraph(BG);

  return ActionState::ptr(_state);
}

ActionState::ptr CommandBondLengthTableAction::performRedo(ActionState::ptr _state){
  CommandBondLengthTableState *state = assert_cast<CommandBondLengthTableState*>(_state.get());

  BondGraph *&BG = World::getInstance().getBondGraph();
  BG->CleanupBondLengthTable();
  std::ifstream input(state->params.BondGraphFileName.get().string().c_str());
  if ((input.good()) && (BG->LoadBondLengthTable(input))) {
    LOG(0, "Bond length table parsed successfully.");
    input.close();
  }

  return ActionState::ptr(_state);
}

bool CommandBondLengthTableAction::canUndo() {
  return true;
}

bool CommandBondLengthTableAction::shouldUndo() {
  return true;
}
/** =========== end of function ====================== */
