/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2010-2012 University of Bonn. All rights reserved.
 * 
 *
 *   This file is part of MoleCuilder.
 *
 *    MoleCuilder is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    MoleCuilder is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with MoleCuilder.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * RotateAroundOriginByAngleAction.cpp
 *
 *  Created on: Aug 06, 2010
 *      Author: heber
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "CodePatterns/MemDebug.hpp"

#include "CodePatterns/Log.hpp"
#include "CodePatterns/Verbose.hpp"
#include "LinearAlgebra/Line.hpp"
#include "LinearAlgebra/Vector.hpp"
#include "molecule.hpp"


#include <cmath>
#include <iostream>
#include <fstream>
#include <string>

#include "Actions/AtomAction/RotateAroundOriginByAngleAction.hpp"

using namespace MoleCuilder;

// and construct the stuff
#include "RotateAroundOriginByAngleAction.def"
#include "Action_impl_pre.hpp"
/** =========== define the function ====================== */
ActionState::ptr AtomRotateAroundOriginByAngleAction::performCall() {
  std::vector<atom *> selectedAtoms = World::getInstance().getSelectedAtoms();

  // check whether Axis is valid
  if (params.Axis.get().IsZero()) {
    STATUS("Specified Rotation axis is zero.");
    return Action::failure;
  }

  // convert from degrees to radian
  double radian = params.angle.get() * M_PI/180.;

  // Creation Line that is the rotation axis
  Line RotationAxis(Vector(0.,0.,0.), params.Axis.get());

  LOG(0, "Rotate around origin by " << radian << " radian, axis from origin to " << params.Axis.get() << ".");
  // TODO: use AtomSet::rotate?
  for (std::vector<atom *>::iterator iter = selectedAtoms.begin(); iter != selectedAtoms.end(); ++iter) {
    (*iter)->setPosition(RotationAxis.rotateVector((*iter)->getPosition(), radian));
  }
  LOG(0, "done.");
  return ActionState::ptr(new AtomRotateAroundOriginByAngleState(World::getInstance().getSelectedAtoms(), params));
}

ActionState::ptr AtomRotateAroundOriginByAngleAction::performUndo(ActionState::ptr _state) {
  AtomRotateAroundOriginByAngleState *state = assert_cast<AtomRotateAroundOriginByAngleState*>(_state.get());

  // convert from degrees to radian
  double radian = params.angle.get() * M_PI/180.;

  // Creation Line that is the rotation axis
  Line RotationAxis(Vector(0.,0.,0.), state->params.Axis.get());

  for (std::vector<atom *>::iterator iter = state->selectedAtoms.begin(); iter != state->selectedAtoms.end(); ++iter) {
    (*iter)->setPosition(RotationAxis.rotateVector((*iter)->getPosition(), -radian));
  }

  return ActionState::ptr(_state);
}

ActionState::ptr AtomRotateAroundOriginByAngleAction::performRedo(ActionState::ptr _state){
  AtomRotateAroundOriginByAngleState *state = assert_cast<AtomRotateAroundOriginByAngleState*>(_state.get());

  // convert from degrees to radian
  double radian = params.angle.get() * M_PI/180.;

  // Creation Line that is the rotation axis
  Line RotationAxis(Vector(0.,0.,0.), state->params.Axis.get());

  for (std::vector<atom *>::iterator iter = state->selectedAtoms.begin(); iter != state->selectedAtoms.end(); ++iter) {
    (*iter)->setPosition(RotationAxis.rotateVector((*iter)->getPosition(), radian));
  }

  return ActionState::ptr(_state);
}

bool AtomRotateAroundOriginByAngleAction::canUndo() {
  return true;
}

bool AtomRotateAroundOriginByAngleAction::shouldUndo() {
  return true;
}
/** =========== end of function ====================== */
