/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2010-2012 University of Bonn. All rights reserved.
 * 
 *
 *   This file is part of MoleCuilder.
 *
 *    MoleCuilder is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    MoleCuilder is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with MoleCuilder.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * SurfaceCorrelationAction.cpp
 *
 *  Created on: May 9, 2010
 *      Author: heber
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "CodePatterns/MemDebug.hpp"

#include "Analysis/analysis_correlation.hpp"
#include "CodePatterns/Log.hpp"
#include "Descriptors/MoleculeIdDescriptor.hpp"
#include "Element/element.hpp"
#include "Element/periodentafel.hpp"
#include "LinearAlgebra/Vector.hpp"
#include "LinkedCell/linkedcell.hpp"
#include "LinkedCell/PointCloudAdaptor.hpp"
#include "molecule.hpp"
#include "Tesselation/boundary.hpp"
#include "Tesselation/tesselation.hpp"
#include "World.hpp"

#include <iostream>
#include <string>

#include "Actions/AnalysisAction/SurfaceCorrelationAction.hpp"

using namespace MoleCuilder;

// and construct the stuff
#include "SurfaceCorrelationAction.def"
#include "Action_impl_pre.hpp"

/** =========== define the function ====================== */
ActionState::ptr AnalysisSurfaceCorrelationAction::performCall() {
  int ranges[3] = {1, 1, 1};
  ofstream output;
  ofstream binoutput;
  string type;
  BinPairMap *binmap = NULL;

  // execute action
  output.open(params.outputname.get().string().c_str());
  binoutput.open(params.binoutputname.get().string().c_str());

  // check for selected molecules and create surfaces from them
  std::vector<atom *> atoms(World::getInstance().getSelectedAtoms());
  LinkedCell_deprecated * LCList = NULL;
  Tesselation * TesselStruct = NULL;
  const double radius = 4.;
  double LCWidth = 20.;
  if (params.BinEnd.get() > 0) {
    if (params.BinEnd.get() > 2.*radius)
      LCWidth = params.BinEnd.get();
    else
      LCWidth = 2.*radius;
  }
  if ( atoms.size() == 0) {
    STATUS("You have not select any atoms.");
    return Action::failure;
  }
  // create adaptor for the selected atoms
  PointCloudAdaptor< std::vector<atom *> > cloud(&atoms, std::string("Selected atoms"));

  // create tesselation
  LCList = new LinkedCell_deprecated(cloud, 2.*radius);
  TesselStruct = new Tesselation;
  (*TesselStruct)(cloud, radius);

  // correlate
  std::vector<molecule*> molecules = World::getInstance().getSelectedMolecules();
  std::cout << "There are " << molecules.size() << " selected molecules." << std::endl;
  CorrelationToSurfaceMap *surfacemap = NULL;
  if (params.periodic.get())
    surfacemap = PeriodicCorrelationToSurface( molecules, params.elements.get(), TesselStruct, LCList, ranges);
  else
    surfacemap = CorrelationToSurface( molecules, params.elements.get(), TesselStruct, LCList);
  delete LCList;
  OutputCorrelationMap<CorrelationToSurfaceMap>(&output, surfacemap, OutputCorrelationToSurface_Header, OutputCorrelationToSurface_Value);
  // check whether radius was appropriate
  {
    double start; double end;
    GetMinMax( surfacemap, start, end);
    if (LCWidth < end)
      ELOG(1, "Linked Cell width is smaller than the found range of values! Bins can only be correct up to: " << radius << ".");
  }
  binmap = BinData( surfacemap, params.BinWidth.get(), params.BinStart.get(), params.BinEnd.get() );
  OutputCorrelationMap<BinPairMap> ( &binoutput, binmap, OutputCorrelation_Header, OutputCorrelation_Value );
  delete TesselStruct;  // surfacemap contains refs to triangles! delete here, not earlier!
  delete(binmap);
  delete(surfacemap);
  output.close();
  binoutput.close();
  return Action::success;
}

ActionState::ptr AnalysisSurfaceCorrelationAction::performUndo(ActionState::ptr _state) {
  return Action::success;
}

ActionState::ptr AnalysisSurfaceCorrelationAction::performRedo(ActionState::ptr _state){
  return Action::success;
}

bool AnalysisSurfaceCorrelationAction::canUndo() {
  return true;
}

bool AnalysisSurfaceCorrelationAction::shouldUndo() {
  return true;
}
/** =========== end of function ====================== */
