/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2010-2012 University of Bonn. All rights reserved.
 * 
 *
 *   This file is part of MoleCuilder.
 *
 *    MoleCuilder is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    MoleCuilder is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with MoleCuilder.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * DipoleAngularCorrelationAction.cpp
 *
 *  Created on: Feb 11, 2011
 *      Author: heber
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "CodePatterns/MemDebug.hpp"

#include "Analysis/analysis_correlation.hpp"
#include "CodePatterns/Log.hpp"
#include "Descriptors/AtomOfMoleculeSelectionDescriptor.hpp"
#include "Descriptors/MoleculeFormulaDescriptor.hpp"
#include "Element/element.hpp"
#include "Element/periodentafel.hpp"
#include "LinearAlgebra/Vector.hpp"
#include "molecule.hpp"
#include "World.hpp"
#include "WorldTime.hpp"

#include <iostream>
#include <map>
#include <string>

#include "Actions/AnalysisAction/DipoleAngularCorrelationAction.hpp"

using namespace MoleCuilder;

// and construct the stuff
#include "DipoleAngularCorrelationAction.def"
#include "Action_impl_pre.hpp"

/** =========== define the function ====================== */
ActionState::ptr AnalysisDipoleAngularCorrelationAction::performCall() {
  //int ranges[3] = {1, 1, 1};
  string type;

  // get selected atoms
  std::vector<atom*> old_atom_selection = World::getInstance().getSelectedAtoms();
  std::vector<molecule*> old_molecule_selection = World::getInstance().getSelectedMolecules();

  // get current time step
  const unsigned int oldtime = WorldTime::getTime();

  // select atoms and obtain zero dipole orientation
  Formula DipoleFormula(params.DipoleFormula.get());
  World::getInstance().setTime(params.timestepzero.get());
  World::getInstance().clearMoleculeSelection(); // TODO: This should be done in setTime or where molecules are re-done
  World::getInstance().selectAllMolecules(MoleculeByFormula(DipoleFormula));
  std::vector<molecule *> molecules = World::getInstance().getSelectedMolecules();
  std::map<atomId_t, Vector> ZeroVector = CalculateZeroAngularDipole(molecules);

  // go through each step of common trajectory of all atoms in set
  World::getInstance().clearAtomSelection();
  World::getInstance().selectAllAtoms(AtomsByMoleculeSelection());
  std::vector<atom *> atoms = World::getInstance().getSelectedAtoms();
  if (atoms.empty()) {
    STATUS("Formula "+toString(DipoleFormula)+" selects no atoms.");
    return Action::failure;
  }
  range<size_t> timesteps = getMaximumTrajectoryBounds(atoms);
  ASSERT(params.timestepzero.get() < timesteps.first,
    "AnalysisDipoleAngularCorrelationAction::performCall() - time step zero "
    +toString(params.timestepzero.get())+" is beyond trajectory range ("
    +toString(timesteps.first)+") of some atoms.");
  for (size_t step = params.timestepzero.get(); step < timesteps.first; ++step) {
    // calculate dipoles relative to zero orientation
    DipoleAngularCorrelationMap *correlationmap = NULL;
    correlationmap = DipoleAngularCorrelation(DipoleFormula, step, ZeroVector, DontResetTime);

    // prepare step string in filename
    std::stringstream stepstream;
    stepstream << std::setw(4) << std::setfill('0') << step;
    const std::string stepname(stepstream.str());

    // output correlation map
    ofstream output;
    std::string filename = params.outputname.get().string()+"."+stepname+".dat";
    output.open(filename.c_str());
    OutputCorrelationMap<DipoleAngularCorrelationMap>(&output, correlationmap, OutputDipoleAngularCorrelation_Header, OutputDipoleAngularCorrelation_Value);
    output.close();

    // bin map
    BinPairMap *binmap = BinData( correlationmap, params.BinWidth.get(), params.BinStart.get(), params.BinEnd.get() );

    // free correlation map
    delete(correlationmap);

    // output binned map
    ofstream binoutput;
    std::string binfilename = params.binoutputname.get().string()+"."+stepname+".dat";
    binoutput.open(binfilename.c_str());
    OutputCorrelationMap<BinPairMap> ( &binoutput, binmap, OutputCorrelation_Header, OutputCorrelation_Value );
    binoutput.close();

    // free binned map
    delete(binmap);
  }

  // reset to old time step
  World::getInstance().setTime(oldtime);

  // reset to old selections
  World::getInstance().clearAtomSelection();
  BOOST_FOREACH(atom *_atom, old_atom_selection) {
    World::getInstance().selectAtom(_atom);
  }
  World::getInstance().clearMoleculeSelection();
  BOOST_FOREACH(molecule *_mol, old_molecule_selection) {
    World::getInstance().selectMolecule(_mol);
  }

  // exit
  STATUS("Dipole angular correlation calculation successful.");
  return Action::success;
}

ActionState::ptr AnalysisDipoleAngularCorrelationAction::performUndo(ActionState::ptr _state) {
  return Action::success;
}

ActionState::ptr AnalysisDipoleAngularCorrelationAction::performRedo(ActionState::ptr _state){
  return Action::success;
}

bool AnalysisDipoleAngularCorrelationAction::canUndo() {
  return true;
}

bool AnalysisDipoleAngularCorrelationAction::shouldUndo() {
  return true;
}
/** =========== end of function ====================== */
