/*
 * SubgraphDissectionAction.cpp
 *
 *  Created on: May 9, 2010
 *      Author: heber
 */

#include "Helpers/MemDebug.hpp"

#include "Actions/FragmentationAction/SubgraphDissectionAction.hpp"
#include "Actions/ActionRegistry.hpp"
#include "Descriptors/MoleculeDescriptor.hpp"

#include "atom.hpp"
#include "bond.hpp"
#include "bondgraph.hpp"
#include "config.hpp"
#include "log.hpp"
#include "molecule.hpp"
#include "stackclass.hpp"
#include "World.hpp"

#include <iostream>
#include <string>

using namespace std;

#include "UIElements/UIFactory.hpp"
#include "UIElements/Dialog.hpp"
#include "UIElements/ValueStorage.hpp"

const char FragmentationSubgraphDissectionAction::NAME[] = "subgraph-dissect";

FragmentationSubgraphDissectionAction::FragmentationSubgraphDissectionAction() :
  Action(NAME)
{}

FragmentationSubgraphDissectionAction::~FragmentationSubgraphDissectionAction()
{}

void FragmentationSubgraphDissection() {
  ActionRegistry::getInstance().getActionByName(FragmentationSubgraphDissectionAction::NAME)->call(Action::NonInteractive);
};

Dialog* FragmentationSubgraphDissectionAction::createDialog() {
  Dialog *dialog = UIFactory::getInstance().makeDialog();

  dialog->queryEmpty(NAME, MapOfActions::getInstance().getDescription(NAME));

  return dialog;
}


Action::state_ptr FragmentationSubgraphDissectionAction::performCall() {
  DoLog(1) && (Log() << Verbose(1) << "Dissecting molecular system into a set of disconnected subgraphs ... " << endl);
  // @TODO rather do the dissection afterwards
  MoleculeListClass *molecules = World::getInstance().getMolecules();
  config * const configuration = World::getInstance().getConfig();

  // 0a. remove all present molecules
  vector<molecule *> allmolecules = World::getInstance().getAllMolecules();
  for (vector<molecule *>::iterator MolRunner = allmolecules.begin(); MolRunner != allmolecules.end(); ++MolRunner) {
    molecules->erase(*MolRunner);
    World::getInstance().destroyMolecule(*MolRunner);
  }

  // 0b. remove all bonds and construct a molecule with all atoms
  molecule *mol = World::getInstance().createMolecule();
  vector <atom *> allatoms = World::getInstance().getAllAtoms();
  for(vector<atom *>::iterator AtomRunner = allatoms.begin(); AtomRunner != allatoms.end(); ++AtomRunner) {
    for(BondList::iterator BondRunner = (*AtomRunner)->ListOfBonds.begin(); !(*AtomRunner)->ListOfBonds.empty(); BondRunner = (*AtomRunner)->ListOfBonds.begin())
      delete(*BondRunner);
    mol->AddAtom(*AtomRunner);
  }

  // 1. create the bond structure of the single molecule
  if (configuration->BG != NULL) {
    if (!configuration->BG->ConstructBondGraph(mol)) {
      World::getInstance().destroyMolecule(mol);
      DoeLog(1) && (eLog()<< Verbose(1) << "There are no bonds." << endl);
      return Action::failure;
    }
  } else {
    DoeLog(1) && (eLog()<< Verbose(1) << "There is no BondGraph class present to create bonds." << endl);
    return Action::failure;
  }

  // 2. scan for connected subgraphs
  MoleculeLeafClass *Subgraphs = NULL;      // list of subgraphs from DFS analysis
  class StackClass<bond *> *BackEdgeStack = NULL;
  Subgraphs = mol->DepthFirstSearchAnalysis(BackEdgeStack);
  delete(BackEdgeStack);
  if ((Subgraphs == NULL) || (Subgraphs->next == NULL)) {
    World::getInstance().destroyMolecule(mol);
    DoeLog(1) && (eLog()<< Verbose(1) << "There are no atoms." << endl);
    return Action::failure;
  }

  // 3. dissect (the following construct is needed to have the atoms not in the order of the DFS, but in
  // the original one as parsed in)
  // TODO: Optimize this, when molecules just contain pointer list of global atoms!

  // 4a. create array of molecules to fill
  const int MolCount = Subgraphs->next->Count();
  char number[MAXSTRINGSIZE];
  molecule **moleculelist = new molecule *[MolCount];
  MoleculeLeafClass *MolecularWalker = Subgraphs;
  for (int i=0;i<MolCount;i++) {
    MolecularWalker = MolecularWalker->next;
    moleculelist[i] = World::getInstance().createMolecule();
    moleculelist[i]->ActiveFlag = true;
    strncpy(moleculelist[i]->name, mol->name, MAXSTRINGSIZE);
    if (MolCount > 1) {
      sprintf(number, "-%d", i+1);
      strncat(moleculelist[i]->name, number, MAXSTRINGSIZE - strlen(mol->name) - 1);
    }
    DoLog(1) && (Log() << Verbose(1) << "MolName is " << moleculelist[i]->name << ", id is " << moleculelist[i]->getId() << endl);
    for (molecule::iterator iter = MolecularWalker->Leaf->begin(); iter != MolecularWalker->Leaf->end(); ++iter) {
      DoLog(1) && (Log() << Verbose(1) << **iter << endl);
    }
    molecules->insert(moleculelist[i]);
  }

  // 4b. create and fill map of which atom is associated to which connected molecule (note, counting starts at 1)
  int FragmentCounter = 0;
  map<int, atom *> AtomToFragmentMap;
  MolecularWalker = Subgraphs;
  while (MolecularWalker->next != NULL) {
    MolecularWalker = MolecularWalker->next;
    for (molecule::iterator iter = MolecularWalker->Leaf->begin(); !MolecularWalker->Leaf->empty(); iter = MolecularWalker->Leaf->begin()) {
      atom * Walker = *iter;
      DoLog(1) && (Log() << Verbose(1) << "Re-linking " << Walker << "..." << endl);
      MolecularWalker->Leaf->erase(iter);
      moleculelist[FragmentCounter]->AddAtom(Walker);    // counting starts at 1
    }
    FragmentCounter++;
  }
  // TODO: When DepthFirstSearchAnalysis does not use AddCopyAtom() anymore, we don't need to delete all original atoms.
  // 4d. destroy the original molecule
  for (molecule::iterator AtomRunner = mol->begin(); !mol->empty(); AtomRunner = mol->begin())
    World::getInstance().destroyAtom(*AtomRunner);
  World::getInstance().destroyMolecule(mol);

  // 4d. we don't need to redo bonds, as they are connected subgraphs and still maintain their ListOfBonds, but we have to remove them from first..last list
  // TODO: check whether this is really not needed anymore
  // 4e. free Leafs
  MolecularWalker = Subgraphs;
  while (MolecularWalker->next != NULL) {
    MolecularWalker = MolecularWalker->next;
    delete(MolecularWalker->previous);
  }
  delete(MolecularWalker);
  delete[](moleculelist);
  DoLog(1) && (Log() << Verbose(1) << "I scanned " << FragmentCounter << " molecules." << endl);

  return Action::success;
}

Action::state_ptr FragmentationSubgraphDissectionAction::performUndo(Action::state_ptr _state) {
//  ParserLoadXyzState *state = assert_cast<ParserLoadXyzState*>(_state.get());

  return Action::failure;
//  string newName = state->mol->getName();
//  state->mol->setName(state->lastName);
//
//  return Action::state_ptr(new ParserLoadXyzState(state->mol,newName));
}

Action::state_ptr FragmentationSubgraphDissectionAction::performRedo(Action::state_ptr _state){
  return Action::failure;
}

bool FragmentationSubgraphDissectionAction::canUndo() {
  return false;
}

bool FragmentationSubgraphDissectionAction::shouldUndo() {
  return false;
}

const string FragmentationSubgraphDissectionAction::getName() {
  return NAME;
}
