/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2010 University of Bonn. All rights reserved.
 * Please see the LICENSE file or "Copyright notice" in builder.cpp for details.
 */

/*
 * OptionTrait.cpp
 *
 *  Created on: Oct 27, 2010
 *      Author: heber
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "CodePatterns/MemDebug.hpp"

#include <iostream>
#include <typeinfo>

#include <boost/shared_ptr.hpp>

#include "Actions/OptionRegistry.hpp"
#include "Actions/OptionTrait.hpp"

using namespace MoleCuilder;

/** Constructor for class OptionTrait.
 * note that OptionTrait automatically registers with OptionRegistry.
 * \param &_token name of option
 * \param &_type type of option
 * \param &_description description of what the option is for
 * \param &_defaultvalue default value, empty string if none
 * \param &_shortform short form to access option on CommmandLineUI, empty if none
 * \param *_instance type to be stored in this OptionTrait.
 */
OptionTrait::OptionTrait(const std::string &_token, std::type_info const * const _type, const std::string &_description, const std::string &_defaultvalue, const std::string &_shortform) :
    Token(_token),
    InternalType(_type),
    Description(_description),
    DefaultValue(_defaultvalue),
    ShortForm(_shortform)
{
  //std::cout << "OptionTrait::OptionTrait() on instance " << this << " with " << getName() << ", type " << getTypeName() << " and description " << getDescription() << std::endl;
}

/** Destructor for class OptionTrait.
 *
 */
OptionTrait::~OptionTrait()
{
  //std::cout << "OptionTrait::~OptionTrait() on instance " << this << " with name " << getName() << " called." << std::endl;
}

/** Returns Current Value for this ActionTrait.
 * \return OptionTrait::CurrentValue as std::string
 */
const std::string& OptionTrait::getDefaultValue() const
{
 return DefaultValue;
}

/** Returns Description for this ActionTrait.
 * \return OptionTrait::Description as std::string
 */
const std::string& OptionTrait::getDescription() const
{
  return Description;
}

/** Specific Getter for a ActionTrait::ShortForm.
 * If action has a short for, then combination is as "Token,ShortForm" (this is
 * the desired format for boost::program_options). If no short form exists in the map,
 * just Token will be returned
 * Note that we assert when action does not exist in CommandLineParser::DescriptionMap.
 * \return ActionTrait::Token,ActionTrait::ShortForm or Description of the action
 */
const std::string OptionTrait::getKeyAndShortForm() const
{
  std::stringstream output;
  output << Token;
  if (hasShortForm())
    output << "," << ShortForm;
  return output.str();
}


/** Returns ShortForm for this OptionTrait.
 * \return OptionTrait::ShortForm as std::string
 */
const std::string& OptionTrait::getShortForm() const
{
  return ShortForm;
}

/** Returns Type for this OptionTrait.
 * \return OptionTrait::InternalType as std::type_info reference
 */
const std::type_info * const OptionTrait::getType() const
{
  return InternalType;
}

/** Returns Name of the Type for this OptionTrait.
 * \return OptionTrait::InternalType->name()
 */
const std::string OptionTrait::getTypeName() const
{
  return InternalType->name();
}

/** States whether DefaultValue is present or not.
 * \return true - DefaultValue unequal to empty string present.
 */
bool OptionTrait::hasDefaultValue() const
{
  return (!DefaultValue.empty());
}

/** States whether ShortForm is present or not.
 * \return true - ShortForm unequal to empty string present.
 */
bool OptionTrait::hasShortForm() const
{
  return (!ShortForm.empty());
}

/** Returns token of the option.
 * \return name/token of option
 */
const std::string& OptionTrait::getName() const
{
  return Token;
}

/** Sets the short form to the given value.
 * \param &_shortform new short form
 */
void OptionTrait::setShortForm(const std::string _shortform)
{
  ShortForm = _shortform;
}

/** Output operator for OptionTrait.
 *
 * \param &out output stream to print to.
 */
std::ostream& operator<<(std::ostream &out, const OptionTrait &t)
{
  out << "OptionTrait(" << &t << "): Name(" << t.getName() << ")";
  out << ", Type(" << t.getTypeName() << ")";
  out << ", Default(" << t.getDefaultValue() << ")";
  out << ", ShortForm(" << t.getShortForm() << ")";
  out << "." << std::endl;
  return out;
}
