/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2010 University of Bonn. All rights reserved.
 * Please see the LICENSE file or "Copyright notice" in builder.cpp for details.
 */

/*
 * ActionTraits.cpp
 *
 *  Created on: Oct 26, 2010
 *      Author: heber
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "CodePatterns/MemDebug.hpp"

#include "Actions/ActionTraits.hpp"

#include "CodePatterns/Assert.hpp"

#include <iostream>
#include <sstream>
#include <string>
#include <typeinfo>

using namespace MoleCuilder;

/** Copy constructor for base class ActionTraits.
 * \param &_Traits source ActionTraits class to copy
 */
ActionTraits::ActionTraits(const std::string &_token) :
  OptionTrait(_token,
      &typeid(void),
      "this is an invisible action",
      "",
      ""
  )
{
  //std::cout << "ActionTraits::ActionTraits(string &) on instance " << this << " with " << getName() << ", type " << getTypeName() << " and description " << getDescription() << std::endl;
}

/** Copy constructor for base class ActionTraits.
 * we have to make our own copy in order to avoid references and deep-copy everything.
 * \param &_Traits source ActionTraits class to copy
 */
ActionTraits::ActionTraits(const ActionTraits &_Traits) :
  OptionTrait(_Traits),
  MenuTitle(_Traits.MenuTitle),
  MenuPosition(_Traits.MenuPosition)
{
  for (options_const_iterator iter = _Traits.Options.begin(); iter != _Traits.Options.end(); ++iter) {
    Options.insert(
        std::make_pair(
            iter->first,
            new OptionTrait(*iter->second)
        )
    );
  }
  //std::cout << "ActionTraits::ActionTraits(ActionTraits &) on instance " << this << " with " << getName() << ", type " << getTypeName() << " and description " << getDescription() << std::endl;
}

/** Copy constructor for base class ActionTraits.
 * we have to make our own copy in order to avoid references and deep-copy everything.
 * \param &_Traits source OptionTrait class to copy
 */
ActionTraits::ActionTraits(const OptionTrait &_Traits, const std::string _MenuTitle, const int _MenuPosition) :
  OptionTrait(_Traits),
  MenuTitle(_MenuTitle),
  MenuPosition(_MenuPosition)
{
  //std::cout << "ActionTraits::ActionTraits(OptionTrait &) on instance " << this << " with " << getName() << ", type " << getTypeName() << " and description " << getDescription() << std::endl;
}

/** Constructor for base class ActionTraits.
 * Deletes all present Options.
 */
ActionTraits::~ActionTraits()
{
  //std::cout << "ActionTraits::~ActionTraits() on instance " << this << " with name " << getName() << " called." << std::endl;
  for (options_iterator iter = Options.begin(); iter != Options.end(); ++iter) {
    //std::cout << "ActionTraits::~ActionTraits() removes option instance " << iter->second << " with name " << iter->first << "." << std::endl;
    delete iter->second;
  }
  Options.clear();
}


/** Returns menu title for this ActionTrait.
 * \return ActionTraits::MenuTitle as std::string
 */
const std::string& ActionTraits::getMenuName() const
{
  return MenuTitle;
}

/** Returns menu title for this ActionTrait.
 * \return ActionTraits::MenuPosition as std::string
 */
int ActionTraits::getMenuPosition() const
{
  return MenuPosition;
}

/** Returns Description for the given option of this ActionTrait.
 * \param &token token of option
 * \return reference to OptionTrait
 */
OptionTrait const & ActionTraits::getOption(const std::string &token) const
{
  ASSERT(Options.find(token) != Options.end(),
      "ActionTraits::getOption() - Option not found!");
  return *(Options.find(token)->second);
}

/** States whether given option (token) is present or not.
 * \param &token name of option
 * \return true - option present, false - not
 */
bool ActionTraits::hasOption(const std::string &token) const
{
  return (Options.find(token) != Options.end());
}

/** States whether this Action has options at all.
 * \return true - options present, false - no options
 */
bool ActionTraits::hasOptions() const
{
  return (Options.begin() != Options.end());
}

/** Forward iterator from beginning of list of options.
 * \return iterator
 */
ActionTraits::options_iterator ActionTraits::getBeginIter()
{
  return Options.begin();
}

/** Forward iterator at end of list of options.
 * \return iterator
 */
ActionTraits::options_iterator ActionTraits::getEndIter()
{
  return Options.end();
}

/** Constant forward iterator from beginning of list of options.
 * \return constant iterator
 */
ActionTraits::options_const_iterator ActionTraits::getBeginIter() const
{
  return Options.begin();
}

/** Constant forward iterator at end of list of options.
 * \return constant iterator
 */
ActionTraits::options_const_iterator ActionTraits::getEndIter() const
{
  return Options.end();
}

/** Output operator for ActionTrait.
 *
 * \param &out stream to output to
 */
std::ostream &operator<<(std::ostream &out, const ActionTraits &a)
{
  out << "ActionTrait(" << &a << ") with " << static_cast<const OptionTrait &>(a) << " has the following options: " << std::endl;
  for (ActionTraits::options_const_iterator iter = a.getBeginIter();
    iter != a.getEndIter();
    ++iter) {
    out << "\t - " << *iter->second << ")" << std::endl;
  }
  return out;
}

