/*
 * MapOfActions.cpp
 *
 *  Created on: 10.05.2010
 *      Author: heber
 */

#include "Helpers/MemDebug.hpp"

using namespace std;

#include "Patterns/Singleton_impl.hpp"
#include "Actions/MapOfActions.hpp"
#include "Helpers/Assert.hpp"

#include <boost/lexical_cast.hpp>
#include <boost/optional.hpp>
#include <boost/program_options.hpp>

#include "CommandLineParser.hpp"
#include "log.hpp"
#include "verbose.hpp"

#include "Actions/ActionRegistry.hpp"
#include "Actions/AnalysisAction/MolecularVolumeAction.hpp"
#include "Actions/AnalysisAction/PairCorrelationAction.hpp"
#include "Actions/AnalysisAction/PrincipalAxisSystemAction.hpp"
#include "Actions/AtomAction/AddAction.hpp"
#include "Actions/AtomAction/ChangeElementAction.hpp"
#include "Actions/AtomAction/RemoveAction.hpp"
#include "Actions/CmdAction/BondLengthTableAction.hpp"
#include "Actions/CmdAction/ElementDbAction.hpp"
#include "Actions/CmdAction/FastParsingAction.hpp"
#include "Actions/CmdAction/HelpAction.hpp"
#include "Actions/CmdAction/VerboseAction.hpp"
#include "Actions/CmdAction/VersionAction.hpp"
#include "Actions/FragmentationAction/DepthFirstSearchAction.hpp"
#include "Actions/FragmentationAction/SubgraphDissectionAction.hpp"
#include "Actions/FragmentationAction/FragmentationAction.hpp"
#include "Actions/MoleculeAction/BondFileAction.hpp"
#include "Actions/MoleculeAction/ChangeNameAction.hpp"
#include "Actions/MoleculeAction/FillWithMoleculeAction.hpp"
#include "Actions/MoleculeAction/LinearInterpolationofTrajectoriesAction.hpp"
#include "Actions/MoleculeAction/SaveAdjacencyAction.hpp"
#include "Actions/MoleculeAction/SaveBondsAction.hpp"
#include "Actions/MoleculeAction/SaveTemperatureAction.hpp"
#include "Actions/MoleculeAction/TranslateAction.hpp"
#include "Actions/MoleculeAction/VerletIntegrationAction.hpp"
#include "Actions/ParserAction/LoadXyzAction.hpp"
#include "Actions/ParserAction/SaveXyzAction.hpp"
#include "Actions/TesselationAction/ConvexEnvelopeAction.hpp"
#include "Actions/TesselationAction/NonConvexEnvelopeAction.hpp"
#include "Actions/WorldAction/AddEmptyBoundaryAction.hpp"
#include "Actions/WorldAction/BoundInBoxAction.hpp"
#include "Actions/WorldAction/CenterInBoxAction.hpp"
#include "Actions/WorldAction/CenterOnEdgeAction.hpp"
#include "Actions/WorldAction/ChangeBoxAction.hpp"
#include "Actions/WorldAction/InputAction.hpp"
#include "Actions/WorldAction/OutputAction.hpp"
#include "Actions/WorldAction/RemoveSphereOfAtomsAction.hpp"
#include "Actions/WorldAction/RepeatBoxAction.hpp"
#include "Actions/WorldAction/ScaleBoxAction.hpp"
#include "Actions/WorldAction/SetDefaultNameAction.hpp"
#include "Actions/WorldAction/SetGaussianBasisAction.hpp"
#include "Actions/Values.hpp"

void validate(boost::any& v, const std::vector<std::string>& values, VectorValue *, int)
{
  VectorValue VV;
  if (values.size() != 3) {
    cerr <<  "Specified vector does not have three components but " << values.size() << endl;
    throw boost::program_options::validation_error("Specified vector does not have three components");
  }
  VV.x = boost::lexical_cast<double>(values.at(0));
  VV.y = boost::lexical_cast<double>(values.at(1));
  VV.z = boost::lexical_cast<double>(values.at(2));
  v = boost::any(VectorValue(VV));
}

void validate(boost::any& v, const std::vector<std::string>& values, BoxValue *, int)
{
  BoxValue BV;
  if (values.size() != 6) {
    cerr <<  "Specified vector does not have three components but " << values.size() << endl;
    throw boost::program_options::validation_error("Specified symmetric box matrix does not have six components");
  }
  BV.xx = boost::lexical_cast<double>(values.at(0));
  BV.xy = boost::lexical_cast<double>(values.at(1));
  BV.xz = boost::lexical_cast<double>(values.at(2));
  BV.yy = boost::lexical_cast<double>(values.at(3));
  BV.yz = boost::lexical_cast<double>(values.at(4));
  BV.zz = boost::lexical_cast<double>(values.at(5));
  v = boost::any(BoxValue(BV));
}

/** Constructor of class MapOfActions.
 *
 */
MapOfActions::MapOfActions()
{
  // initialise lookup map
  CmdParserLookup[&generic] = &(CommandLineParser::getInstance().generic);
  CmdParserLookup[&config] = &(CommandLineParser::getInstance().config);
  CmdParserLookup[&hidden] = &(CommandLineParser::getInstance().hidden);
  CmdParserLookup[&visible] = &(CommandLineParser::getInstance().visible);

  // keys for actions
  DescriptionMap["add-atom"] = "add atom of specified element";
  DescriptionMap["bond-table"] = "setting name of the bond length table file";
  DescriptionMap["bond-file"] = "name of the bond file";
  DescriptionMap["boundary"] = "change box to add an empty boundary around all atoms";
  DescriptionMap["bound-in-box"] = "bound all atoms in the domain";
  DescriptionMap["center-edge"] = "center edge of all atoms on (0,0,0)";
  DescriptionMap["center-in-box"] = "center all atoms in the domain";
  DescriptionMap["change-box"] = "change the symmetrc matrix of the simulation domain";
  DescriptionMap["change-element"] = "change the element of an atom";
  DescriptionMap["change-molname"] = "change the name of a molecule";
  DescriptionMap["convex-envelope"] = "create the convex envelope for a molecule";
  DescriptionMap["default-molname"] = "set the default name of new molecules";
  DescriptionMap["depth-first-search"] = "Depth-First Search analysis of the molecular system";
  DescriptionMap["element-db"] = "setting the path where the element databases can be found";
  DescriptionMap["fastparsing"] = "setting whether trajectories shall be parsed completely (n) or just first step (y)";
  DescriptionMap["fill-molecule"] = "fill empty space of box with a filler molecule";
  DescriptionMap["fragment-mol"] = "create for a given molecule into fragments up to given order";
  DescriptionMap["help"] = "Give this help screen";
  DescriptionMap["input"] = "specify input files";
  DescriptionMap["linear-interpolate"] = "linear interpolation in discrete steps between start and end position of a molecule";
  DescriptionMap["nonconvex-envelope"] = "create the non-convex envelope for a molecule";
  DescriptionMap["molecular-volume"] = "calculate the volume of a given molecule";
  DescriptionMap["output"] = "specify output formats";
  DescriptionMap["pair-correlation"] = "pair correlation analysis between two elements, element and point or element and surface";
  DescriptionMap["parse-xyz"] = "parse xyz file into World";
  DescriptionMap["principal-axis-system"] = "calculate the principal axis system of the specified molecule";
  DescriptionMap["remove-atom"] = "remove a specified atom";
  DescriptionMap["remove-sphere"] = "remove sphere of atoms of around a specified atom";
  DescriptionMap["repeat-box"] = "create periodic copies of the simulation box per axis";
  DescriptionMap["rotate-to-pas"] = "calculate the principal axis system of the specified molecule and rotate specified axis to align with main axis";
  DescriptionMap["set-basis"] = "set the name of the gaussian basis set for MPQC";
  DescriptionMap["save-adjacency"] = "name of the adjacency file to write to";
  DescriptionMap["save-bonds"] = "name of the bonds file to write to";
  DescriptionMap["save-temperature"] = "name of the temperature file to write to";
  DescriptionMap["SaveXyz"] = "save world as xyz file";
  DescriptionMap["scale-box"] = "scale box and atomic positions inside";
  DescriptionMap["subgraph-dissect"] = "dissect the molecular system into molecules representing disconnected subgraphs";
  DescriptionMap["suspend-in-water"] = "suspend the given molecule in water such that in the domain the mean density is as specified";
  DescriptionMap["translate-mol"] = "translate molecule by given vector";
  DescriptionMap["verbose"] = "set verbosity level";
  DescriptionMap["verlet-integrate"] = "perform verlet integration of a given force file";
  DescriptionMap["version"] = "show version";
  // keys for values
  DescriptionMap["atom-by-id"] = "index of an atom";
  DescriptionMap["bin-output-file"] = "name of the bin output file";
  DescriptionMap["bin-end"] = "start of the last bin";
  DescriptionMap["bin-start"] = "start of the first bin";
  DescriptionMap["bin-width"] = "width of the bins";
  DescriptionMap["convex-file"] = "filename of the non-convex envelope";
  DescriptionMap["distance"] = "distance in space";
  DescriptionMap["distances"] = "list of three of distances in space, one for each axis direction";
  DescriptionMap["DoRotate"] = "whether to rotate or just report angles";
  DescriptionMap["element"] = "single element";
  DescriptionMap["elements"] = "set of elements";
  DescriptionMap["end-step"] = "last or end step";
  DescriptionMap["id-mapping"] = "whether the identity shall be used in mapping atoms onto atoms or some closest distance measure shall be used";
  DescriptionMap["input"] = "name of input file";
  DescriptionMap["length"] = "length in space";
  DescriptionMap["lengths"] = "list of three of lengths in space, one for each axis direction";
  DescriptionMap["MaxDistance"] = "maximum distance in space";
  DescriptionMap["molecule-by-id"] = "index of a molecule";
  DescriptionMap["molecule-by-name"] = "name of a molecule";
  DescriptionMap["nonconvex-file"] = "filename of the non-convex envelope";
  DescriptionMap["order"] = "order of a discretization, dissection, ...";
  DescriptionMap["output-file"] = "name of the output file";
  DescriptionMap["periodic"] = "system is constraint to periodic boundary conditions (y/n)";
  DescriptionMap["position"] = "position in R^3 space";
  DescriptionMap["sphere-radius"] = "radius of tesselation sphere";
  DescriptionMap["start-step"] = "first or start step";

  // short forms for the actions
  ShortFormMap["add-atom"] = "a";
  ShortFormMap["bond-table"] = "g";
  ShortFormMap["bond-file"] = "A";
  ShortFormMap["boundary"] = "c";
  ShortFormMap["change-box"] = "B";
  ShortFormMap["center-edge"] = "O";
  ShortFormMap["center-in-box"] = "b";
  ShortFormMap["change-element"] = "E";
  ShortFormMap["convex-envelope"] = "o";
  ShortFormMap["default-molname"] = "X";
  ShortFormMap["depth-first-search"] = "D";
  ShortFormMap["element-db"] = "e";
  ShortFormMap["fastparsing"] = "n";
  ShortFormMap["fill-molecule"] = "F";
  ShortFormMap["fragment-mol"] = "f";
  ShortFormMap["help"] = "h";
  ShortFormMap["input"] = "i";
  ShortFormMap["linear-interpolate"] = "L";
  ShortFormMap["nonconvex-envelope"] = "N";
  ShortFormMap["pair-correlation"] = "C";
  ShortFormMap["parse-xyz"] = "p";
  ShortFormMap["remove-atom"] = "r";
  ShortFormMap["remove-sphere"] = "R";
  ShortFormMap["repeat-box"] = "d";
  ShortFormMap["rotate-to-pas"] = "m";
  ShortFormMap["save-adjacency"] = "J";
  ShortFormMap["save-bonds"] = "j";
  ShortFormMap["save-temperature"] = "S";
  ShortFormMap["scale-box"] = "s";
  ShortFormMap["set-basis"] = "M";
  ShortFormMap["subgraph-dissect"] = "I";
  ShortFormMap["suspend-in-water"] = "u";
  ShortFormMap["translate-mol"] = "t";
  ShortFormMap["verbose"] = "v";
  ShortFormMap["verlet-integrate"] = "P";
  ShortFormMap["version"] = "V";

  // value types for the actions
  TypeMap["add-atom"] = Element;
  TypeMap["bond-file"] = String;
  TypeMap["bond-table"] = String;
  TypeMap["boundary"] = Vector;
  TypeMap["center-in-box"] = Box;
  TypeMap["change-box"] = Box;
  TypeMap["change-element"] = Atom;
  TypeMap["change-molname"] = String;
  TypeMap["convex-envelope"] = Molecule;
  TypeMap["default-molname"] = String;
  TypeMap["depth-first-search"] = Double;
  TypeMap["element-db"] = String;
  TypeMap["fastparsing"] = Boolean;
  TypeMap["fill-molecule"] = String;
  TypeMap["fragment-mol"] = String;
  TypeMap["input"] = String;
  TypeMap["linear-interpolate"] = String;
  TypeMap["molecular-volume"] = Molecule;
  TypeMap["nonconvex-envelope"] = Molecule;
  TypeMap["output"] = String;
  TypeMap["parse-xyz"] = String;
  TypeMap["pair-correlation"] = String;
  TypeMap["principal-axis-system"] = Molecule;
  TypeMap["remove-atom"] = Atom;
  TypeMap["remove-sphere"] = Double;
  TypeMap["repeat-box"] = Vector;
  TypeMap["rotate-to-pas"] = Molecule;
  TypeMap["save-adjacency"] = String;
  TypeMap["save-bonds"] = String;
  TypeMap["save-temperature"] = String;
  TypeMap["scale-box"] = Vector;
  TypeMap["set-basis"] = String;
  TypeMap["subgraph-dissect"] = None;
  TypeMap["suspend-in-water"] = Double;
  TypeMap["translate-mol"] = Vector;
  TypeMap["verlet-integrate"] = String;
  TypeMap["verbose"] = Integer;

  // value types for the values
  TypeMap["atom-by-id"] = Atom;
  TypeMap["bin-output-file"] = String;
  TypeMap["bin-end"] = Double;
  TypeMap["bin-start"] = Double;
  TypeMap["bin-width"] = Double;
  TypeMap["convex-file"] = String;
  TypeMap["distance"] = Double;
  TypeMap["distances"] = Vector;
  TypeMap["DoRotate"] = Boolean;
  TypeMap["element"] = Element;
  TypeMap["elements"] = ListOfElements;
  TypeMap["end-step"] = Integer;
  TypeMap["id-mapping"] = Boolean;
  TypeMap["length"] = Double;
  TypeMap["lengths"] = Vector;
  TypeMap["MaxDistance"] = Double;
  TypeMap["molecule-by-id"] = Molecule;
  TypeMap["molecule-by-name"] = Molecule;
  TypeMap["nonconvex-file"] = String;
  TypeMap["order"] = Integer;
  TypeMap["output-file"] = String;
  TypeMap["periodic"] = Boolean;
  TypeMap["position"] = Vector;
  TypeMap["sphere-radius"] = Double;
  TypeMap["start-step"] = Integer;

  // default values for any action that needs one (always string!)
  DefaultValue["bin-width"] = "0.5";
  DefaultValue["fastparsing"] = "0";
  DefaultValue["atom-by-id"] = "-1";
  DefaultValue["molecule-by-id"] = "-1";
  DefaultValue["periodic"] = "0";

  // put action into each menu category
  MenuContainsActionMap.insert( pair<std::string, std::string> ("analysis", "molecular-volume") );
  MenuContainsActionMap.insert( pair<std::string, std::string> ("analysis", "pair-correlation") );
  MenuContainsActionMap.insert( pair<std::string, std::string> ("analysis", "principal-axis-system") );

  MenuContainsActionMap.insert( pair<std::string, std::string> ("atom", "add-atom") );
  MenuContainsActionMap.insert( pair<std::string, std::string> ("atom", "change-element") );
  MenuContainsActionMap.insert( pair<std::string, std::string> ("atom", "remove-atom") );

  MenuContainsActionMap.insert( pair<std::string, std::string> ("command", "bond-table") );
  MenuContainsActionMap.insert( pair<std::string, std::string> ("command", "element-db") );
  MenuContainsActionMap.insert( pair<std::string, std::string> ("command", "fastparsing") );
  MenuContainsActionMap.insert( pair<std::string, std::string> ("command", "verbose") );
  MenuContainsActionMap.insert( pair<std::string, std::string> ("command", "version") );

  MenuContainsActionMap.insert( pair<std::string, std::string> ("fragmentation", "depth-first-search") );
  MenuContainsActionMap.insert( pair<std::string, std::string> ("fragmentation", "fragment-mol") );
  MenuContainsActionMap.insert( pair<std::string, std::string> ("fragmentation", "subgraph-dissect") );

  MenuContainsActionMap.insert( pair<std::string, std::string> ("molecule", "bond-file") );
  MenuContainsActionMap.insert( pair<std::string, std::string> ("molecule", "change-molname") );
  MenuContainsActionMap.insert( pair<std::string, std::string> ("molecule", "fill-molecule") );
  MenuContainsActionMap.insert( pair<std::string, std::string> ("molecule", "linear-interpolate") );
  MenuContainsActionMap.insert( pair<std::string, std::string> ("molecule", "rotate-to-pas") );
  MenuContainsActionMap.insert( pair<std::string, std::string> ("molecule", "save-adjacency") );
  MenuContainsActionMap.insert( pair<std::string, std::string> ("molecule", "save-bonds") );
  MenuContainsActionMap.insert( pair<std::string, std::string> ("molecule", "save-temperature") );
  MenuContainsActionMap.insert( pair<std::string, std::string> ("molecule", "suspend-in-water") );
  MenuContainsActionMap.insert( pair<std::string, std::string> ("molecule", "translate-mol") );
  MenuContainsActionMap.insert( pair<std::string, std::string> ("molecule", "verlet-integrate") );

  MenuContainsActionMap.insert( pair<std::string, std::string> ("parser", "parse-xyz") );
  MenuContainsActionMap.insert( pair<std::string, std::string> ("parser", "SaveXyz") );

  MenuContainsActionMap.insert( pair<std::string, std::string> ("tesselation", "convex-envelope") );
  MenuContainsActionMap.insert( pair<std::string, std::string> ("tesselation", "nonconvex-envelope") );

  MenuContainsActionMap.insert( pair<std::string, std::string> ("world", "boundary") );
  MenuContainsActionMap.insert( pair<std::string, std::string> ("world", "bound-in-box") );
  MenuContainsActionMap.insert( pair<std::string, std::string> ("world", "center-in-box") );
  MenuContainsActionMap.insert( pair<std::string, std::string> ("world", "center-edge") );
  MenuContainsActionMap.insert( pair<std::string, std::string> ("world", "change-box") );
  MenuContainsActionMap.insert( pair<std::string, std::string> ("world", "input") );
  MenuContainsActionMap.insert( pair<std::string, std::string> ("world", "output") );
  MenuContainsActionMap.insert( pair<std::string, std::string> ("world", "remove-sphere") );
  MenuContainsActionMap.insert( pair<std::string, std::string> ("world", "repeat-box") );
  MenuContainsActionMap.insert( pair<std::string, std::string> ("world", "scale-box") );
  MenuContainsActionMap.insert( pair<std::string, std::string> ("world", "default-molname") );
  MenuContainsActionMap.insert( pair<std::string, std::string> ("world", "set-basis") );

  // put actions into command line category
	generic.insert("add-atom");
  generic.insert("bond-file");
	generic.insert("bond-table");
  generic.insert("boundary");
//  generic.insert("bound-in-box");
  generic.insert("center-edge");
  generic.insert("center-in-box");
	generic.insert("change-box");
//  generic.insert("change-molname");
	generic.insert("change-element");
  generic.insert("convex-envelope");
	generic.insert("default-molname");
	generic.insert("depth-first-search");
	generic.insert("element-db");
	generic.insert("fastparsing");
  generic.insert("fill-molecule");
  generic.insert("fragment-mol");
  generic.insert("help");
  generic.insert("input");
  generic.insert("linear-interpolate");
//  generic.insert("molecular-volume");
  generic.insert("nonconvex-envelope");
  generic.insert("output");
	generic.insert("pair-correlation");
	generic.insert("parse-xyz");
//  generic.insert("principal-axis-system");
  generic.insert("remove-atom");
  generic.insert("remove-sphere");
  generic.insert("repeat-box");
  generic.insert("rotate-to-pas");
	generic.insert("save-adjacency");
  generic.insert("save-bonds");
  generic.insert("save-temperature");
  generic.insert("scale-box");
  generic.insert("set-basis");
	generic.insert("subgraph-dissect");
  generic.insert("suspend-in-water");
  generic.insert("translate-mol");
	generic.insert("verbose");
  generic.insert("verlet-integrate");
	generic.insert("version");

    // positional arguments
  generic.insert("input");
  inputfile.insert("input");

    // hidden arguments
  generic.insert("atom-by-id");
  generic.insert("bin-end");
  generic.insert("bin-output-file");
  generic.insert("bin-start");
  generic.insert("bin-width");
  generic.insert("convex-file");
  generic.insert("distance");
  generic.insert("DoRotate");
  generic.insert("distances");
  generic.insert("element");
  generic.insert("elements");
  generic.insert("end-step");
  generic.insert("id-mapping");
  generic.insert("lengths");
  generic.insert("MaxDistance");
  generic.insert("molecule-by-id");
  generic.insert("molecule-by-name");
  generic.insert("nonconvex-file");
  generic.insert("order");
  generic.insert("output-file");
  generic.insert("periodic");
  generic.insert("position");
  generic.insert("sphere-radius");
  generic.insert("start-step");
}

/** Destructor of class MapOfActions.
 *
 */
MapOfActions::~MapOfActions()
{
  DescriptionMap.clear();
}


void MapOfActions::populateActions()
{
  new AnalysisMolecularVolumeAction();
  new AnalysisPairCorrelationAction();
  new AnalysisPrincipalAxisSystemAction();

  new AtomAddAction();
  new AtomChangeElementAction();
  new AtomRemoveAction();

  new CommandLineBondLengthTableAction();
  new CommandLineElementDbAction();
  new CommandLineFastParsingAction();
  new CommandLineHelpAction();
  new CommandLineVerboseAction();
  new CommandLineVersionAction();

  new FragmentationDepthFirstSearchAction();
  new FragmentationFragmentationAction();
  new FragmentationSubgraphDissectionAction();

  new MoleculeBondFileAction();
  new MoleculeChangeNameAction();
  new MoleculeFillWithMoleculeAction();
  new MoleculeLinearInterpolationofTrajectoriesAction();
  new MoleculeSaveAdjacencyAction();
  new MoleculeSaveBondsAction();
  new MoleculeSaveTemperatureAction();
  new MoleculeTranslateAction();
  new MoleculeVerletIntegrationAction();

  new ParserLoadXyzAction();
  new ParserSaveXyzAction();

  new TesselationConvexEnvelopeAction();
  new TesselationNonConvexEnvelopeAction();

  new WorldAddEmptyBoundaryAction();
  new WorldBoundInBoxAction();
  new WorldCenterInBoxAction();
  new WorldCenterOnEdgeAction();
  new WorldChangeBoxAction();
  new WorldInputAction();
  new WorldOutputAction();
  new WorldRemoveSphereOfAtomsAction();
  new WorldRepeatBoxAction();
  new WorldScaleBoxAction();
  new WorldSetDefaultNameAction();
  new WorldSetGaussianBasisAction();
}


/** Adds all options to the CommandLineParser.
 *
 */
void MapOfActions::AddOptionsToParser()
{
  // add other options
  for (map< set<string>*, po::options_description* >::iterator ListRunner = CmdParserLookup.begin(); ListRunner != CmdParserLookup.end(); ++ListRunner) {
    for (set<string>::iterator OptionRunner = ListRunner->first->begin(); OptionRunner != ListRunner->first->end(); ++OptionRunner) {
      if (hasValue(*OptionRunner)) {
        DoLog(0) && (Log() << Verbose(0) << "Adding option " << *OptionRunner << " with type " << TypeMap[*OptionRunner] << " to CommandLineParser." << endl);
           switch((enum OptionTypes) TypeMap[*OptionRunner]) {
          default:
          case None:
            ListRunner->second->add_options()
              (getKeyAndShortForm(*OptionRunner).c_str(), getDescription(*OptionRunner).c_str())
              ;
            break;
          case Boolean:
            ListRunner->second->add_options()
              (getKeyAndShortForm(*OptionRunner).c_str(),
                  DefaultValue.find(*OptionRunner) != DefaultValue.end() ?
                        po::value< bool >()->default_value(atoi(DefaultValue[*OptionRunner].c_str())) :
                        po::value< bool >(),
                  getDescription(*OptionRunner).c_str())
              ;
            break;
          case Box:
            ListRunner->second->add_options()
              (getKeyAndShortForm(*OptionRunner).c_str(),
                  po::value<BoxValue>()->multitoken(),
                  getDescription(*OptionRunner).c_str())
              ;
            break;
          case Integer:
            ListRunner->second->add_options()
              (getKeyAndShortForm(*OptionRunner).c_str(),
                  DefaultValue.find(*OptionRunner) != DefaultValue.end() ?
                        po::value< int >()->default_value(atoi(DefaultValue[*OptionRunner].c_str())) :
                        po::value< int >(),
                  getDescription(*OptionRunner).c_str())
              ;
            break;
          case ListOfInts:
            ListRunner->second->add_options()
              (getKeyAndShortForm(*OptionRunner).c_str(),
                  po::value< vector<int> >()->multitoken(),
                  getDescription(*OptionRunner).c_str())
              ;
            break;
          case Double:
            ListRunner->second->add_options()
              (getKeyAndShortForm(*OptionRunner).c_str(),
                  DefaultValue.find(*OptionRunner) != DefaultValue.end() ?
                        po::value< double >()->default_value(atof(DefaultValue[*OptionRunner].c_str())) :
                        po::value< double >(),
                  getDescription(*OptionRunner).c_str())
              ;
            break;
          case ListOfDoubles:
            ListRunner->second->add_options()
              (getKeyAndShortForm(*OptionRunner).c_str(),
                  po::value< vector<double> >()->multitoken(),
                  getDescription(*OptionRunner).c_str())
              ;
            break;
          case String:
            ListRunner->second->add_options()
              (getKeyAndShortForm(*OptionRunner).c_str(),
                  DefaultValue.find(*OptionRunner) != DefaultValue.end() ?
                        po::value< std::string >()->default_value(DefaultValue[*OptionRunner]) :
                        po::value< std::string >(),
                  getDescription(*OptionRunner).c_str())
              ;
            break;
          case Axis:
            ListRunner->second->add_options()
              (getKeyAndShortForm(*OptionRunner).c_str(),
                  DefaultValue.find(*OptionRunner) != DefaultValue.end() ?
                        po::value< int >()->default_value(atoi(DefaultValue[*OptionRunner].c_str())) :
                        po::value< int >(),
                  getDescription(*OptionRunner).c_str())
              ;
            break;
          case Vector:
            ListRunner->second->add_options()
              (getKeyAndShortForm(*OptionRunner).c_str(),
                  po::value<VectorValue>()->multitoken(),
                  getDescription(*OptionRunner).c_str())
              ;
            break;
          case Molecule:
            ListRunner->second->add_options()
              (getKeyAndShortForm(*OptionRunner).c_str(),
                  DefaultValue.find(*OptionRunner) != DefaultValue.end() ?
                        po::value< int >()->default_value(atoi(DefaultValue[*OptionRunner].c_str())) :
                        po::value< int >(),
                  getDescription(*OptionRunner).c_str())
              ;
            break;
          case ListOfMolecules:
            ListRunner->second->add_options()
              (getKeyAndShortForm(*OptionRunner).c_str(),
                  po::value< vector<int> >()->multitoken(),
                  getDescription(*OptionRunner).c_str())
              ;
            break;
          case Atom:
            ListRunner->second->add_options()
              (getKeyAndShortForm(*OptionRunner).c_str(),
                  DefaultValue.find(*OptionRunner) != DefaultValue.end() ?
                        po::value< int >()->default_value(atoi(DefaultValue[*OptionRunner].c_str())) :
                        po::value< int >(),
                  getDescription(*OptionRunner).c_str())
              ;
            break;
          case ListOfAtoms:
            ListRunner->second->add_options()
              (getKeyAndShortForm(*OptionRunner).c_str(),
                  po::value< vector<int> >()->multitoken(),
                  getDescription(*OptionRunner).c_str())
              ;
            break;
          case Element:
            ListRunner->second->add_options()
              (getKeyAndShortForm(*OptionRunner).c_str(),
                  po::value< vector<int> >(),
                  getDescription(*OptionRunner).c_str())
              ;
            break;
          case ListOfElements:
            ListRunner->second->add_options()
              (getKeyAndShortForm(*OptionRunner).c_str(),
                  po::value< vector<int> >()->multitoken(),
                  getDescription(*OptionRunner).c_str())
              ;
            break;
        }
      } else {
        DoLog(0) && (Log() << Verbose(0) << "Adding option " << *OptionRunner << " to CommandLineParser." << endl);
        ListRunner->second->add_options()
          (getKeyAndShortForm(*OptionRunner).c_str(), getDescription(*OptionRunner).c_str())
          ;
      }
    }
  }
  // add positional arguments
  for (set<string>::iterator OptionRunner = inputfile.begin(); OptionRunner != inputfile.end(); ++OptionRunner) {
    DoLog(0) && (Log() << Verbose(0) << "Adding option " << *OptionRunner << " to positional CommandLineParser." << endl);
    CommandLineParser::getInstance().inputfile.add((*OptionRunner).c_str(), -1);
  }
  cout << "Name for position 1: " << CommandLineParser::getInstance().inputfile.name_for_position(1) << endl;
}

/** Getter for MapOfActions:DescriptionMap.
 * Note that we assert when action does not exist in CommandLineParser::DescriptionMap.
 * \param actionname name of the action to lookup
 * \return Description of the action
 */
std::string MapOfActions::getDescription(string actionname)
{
  ASSERT(DescriptionMap.find(actionname) != DescriptionMap.end(), "Unknown action name passed to MapOfActions::getDescription");
  return DescriptionMap[actionname];
}

/** Specific Getter for a MapOfActions:ShortFormMap.
 * If action has a short for, then combination is as "actionname,ShortForm" (this is
 * the desired format for boost::program_options). If no short form exists in the map,
 * just actionname will be returned
 * Note that we assert when action does not exist in CommandLineParser::DescriptionMap.
 * \param actionname name of the action to lookup
 * \return actionname,ShortForm or Description of the action
 */
std::string MapOfActions::getKeyAndShortForm(string actionname)
{
  stringstream output;
  ASSERT(DescriptionMap.find(actionname) != DescriptionMap.end(), "Unknown action name passed to MapOfActions::getDescriptionAndShortForm");
  output << actionname;
  if (ShortFormMap.find(actionname) != DescriptionMap.end())
    output << "," << ShortFormMap[actionname];
  return output.str();
}

/** Getter for MapOfActions:ShortFormMap.
 * Note that we assert when action does not exist CommandLineParser::ShortFormMap.
 * \param actionname name of the action to lookup
 * \return ShortForm of the action
 */
std::string MapOfActions::getShortForm(string actionname)
{
  ASSERT(ShortFormMap.find(actionname) != ShortFormMap.end(), "Unknown action name passed to MapOfActions::getShortForm");
  return ShortFormMap[actionname];
}

/** Returns whether the given action needs a value or not.
 * \param actionname name of the action to look up
 * \return true - value is needed, false - no value is stored in MapOfActions::TypeMap
 */
bool MapOfActions::hasValue(string actionname)
{
  return (TypeMap.find(actionname) != TypeMap.end());
}

/** Getter for MapOfActions::TypeMap.
 * \param actionname name of the action to look up
 * \return type of the action
 */
enum MapOfActions::OptionTypes MapOfActions::getValueType(string actionname)
{
  return TypeMap[actionname];
}

/** Searches whether action is registered with CommandLineParser.
 * Note that this method is only meant transitionally for ParseCommandLineOptions' removal.
 * I.e. All actions that are already handled by the new CommandLineUIFactory can be checked
 * by this function.
 * \param shortform command short form to look for
 * \return true - action has been registered, false - action has not been registered.
 */
bool MapOfActions::isShortFormPresent(string shortform)
{
  bool result = false;
  string actionname;
  for (map<std::string, std::string>::iterator ShortFormRunner = ShortFormMap.begin(); ShortFormRunner != ShortFormMap.end(); ++ShortFormRunner)
    if (ShortFormRunner->second == shortform) {
      actionname = ShortFormRunner->first;
      break;
    }
  result = result || (generic.find(actionname) != generic.end());
  result = result || (config.find(actionname) != config.end());
  result = result || (hidden.find(actionname) != hidden.end());
  result = result || (visible.find(actionname) != visible.end());
  result = result || (inputfile.find(actionname) != inputfile.end());
  return result;
}

/** Returns the inverse to MapOfActions::ShortFormMap, i.e. lookup actionname for its short form.
 * \return map from short form of action to name of action
 */
map <std::string, std::string> MapOfActions::getShortFormToActionMap()
{
  map <std::string, std::string> result;

  for (map<std::string, std::string>::iterator iter = ShortFormMap.begin(); iter != ShortFormMap.end();  ++iter)
    result[iter->second] = iter->first;

  return result;
}


CONSTRUCT_SINGLETON(MapOfActions)
