/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2013 University of Bonn. All rights reserved.
 * 
 *
 *   This file is part of MoleCuilder.
 *
 *    MoleCuilder is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    MoleCuilder is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with MoleCuilder.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * HydrogenPoolUnitTest.cpp
 *
 *  Created on: Mar 03, 2013
 *      Author: heber
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

using namespace std;

#include <cppunit/CompilerOutputter.h>
#include <cppunit/extensions/TestFactoryRegistry.h>
#include <cppunit/ui/text/TestRunner.h>

// include headers that implement a archive in simple text format
#include <boost/archive/text_oarchive.hpp>
#include <boost/archive/text_iarchive.hpp>

#include "HydrogenPoolUnitTest.hpp"

#include "CodePatterns/Assert.hpp"

#include "Fragmentation/Exporters/HydrogenPool.hpp"
#include "World.hpp"

#ifdef HAVE_TESTRUNNER
#include "UnitTestMain.hpp"
#endif /*HAVE_TESTRUNNER*/

/********************************************** Test classes **************************************/

// Registers the fixture into the 'registry'
CPPUNIT_TEST_SUITE_REGISTRATION( HydrogenPoolTest );


void HydrogenPoolTest::setUp()
{
  // failing asserts should be thrown
  ASSERT_DO(Assert::Throw);

  pool = new HydrogenPool();
}


void HydrogenPoolTest::tearDown()
{
  delete pool;

  World::purgeInstance();
}

/** UnitTest for leaseHydrogen() and releaseHydrogen.
 */
void HydrogenPoolTest::lease_releaseHydrogen_Test()
{
  // check simple test
  {
    CPPUNIT_ASSERT_EQUAL( 0, World::getInstance().numAtoms() );
    atom * const Walker = pool->leaseHydrogen();
    CPPUNIT_ASSERT( pool->HydrogenQueue.empty() );
    CPPUNIT_ASSERT( !pool->HydrogenInUse.empty() );
    CPPUNIT_ASSERT_EQUAL( 1, World::getInstance().numAtoms() );
    pool->releaseHydrogen(Walker);
    CPPUNIT_ASSERT( !pool->HydrogenQueue.empty() );
    CPPUNIT_ASSERT( pool->HydrogenInUse.empty() );
    pool->cleanup();
    CPPUNIT_ASSERT_EQUAL( 0, World::getInstance().numAtoms() );
  }
  // check via id test
  {
    CPPUNIT_ASSERT_EQUAL( 0, World::getInstance().numAtoms() );
    atom * const Walker = pool->leaseHydrogen();
    CPPUNIT_ASSERT( pool->HydrogenQueue.empty() );
    CPPUNIT_ASSERT( !pool->HydrogenInUse.empty() );
    CPPUNIT_ASSERT_EQUAL( 1, World::getInstance().numAtoms() );
    pool->releaseHydrogen(Walker->getId());
    CPPUNIT_ASSERT( !pool->HydrogenQueue.empty() );
    CPPUNIT_ASSERT( pool->HydrogenInUse.empty() );
    pool->cleanup();
    CPPUNIT_ASSERT_EQUAL( 0, World::getInstance().numAtoms() );
  }
  // check that can't cleanup with leased atoms
  {
    CPPUNIT_ASSERT_EQUAL( 0, World::getInstance().numAtoms() );
    atom * const Walker = pool->leaseHydrogen();
    std::cerr << "The following assertion is intended and does not entail a failure of the test." << std::endl;
#ifndef NDEBUG
    CPPUNIT_ASSERT_THROW(pool->cleanup(), Assert::AssertionFailure );
#else
    pool->cleanup();
#endif
    CPPUNIT_ASSERT_EQUAL( 1, World::getInstance().numAtoms() );
    pool->releaseHydrogen(Walker);
    pool->cleanup();
    CPPUNIT_ASSERT_EQUAL( 0, World::getInstance().numAtoms() );
  }
  // check returning from atom
  {
    CPPUNIT_ASSERT_EQUAL( 0, World::getInstance().numAtoms() );
    atom * const Walker = pool->leaseHydrogen();
    CPPUNIT_ASSERT_EQUAL( 1, World::getInstance().numAtoms() );
    atom * const InvalidWalker = NULL;
    std::cerr << "The following assertion is intended and does not entail a failure of the test." << std::endl;
#ifndef NDEBUG
    CPPUNIT_ASSERT_THROW(pool->releaseHydrogen(InvalidWalker), Assert::AssertionFailure );
#else
    pool->releaseHydrogen(InvalidWalker);
#endif
    CPPUNIT_ASSERT_EQUAL( 1, World::getInstance().numAtoms() );
    pool->releaseHydrogen(Walker);
    pool->cleanup();
    CPPUNIT_ASSERT_EQUAL( 0, World::getInstance().numAtoms() );
  }
}

/** UnitTest for requestHydrogen().
 */
void HydrogenPoolTest::requestHydrogen_Test()
{
  CPPUNIT_ASSERT_EQUAL( 0, World::getInstance().numAtoms() );
  pool->requestHydrogenIntoPool();
  CPPUNIT_ASSERT( !pool->HydrogenQueue.empty() );
  CPPUNIT_ASSERT_EQUAL( 1, World::getInstance().numAtoms() );
  pool->cleanup();
  CPPUNIT_ASSERT_EQUAL( 0, World::getInstance().numAtoms() );
}
