/** \file MoleculeListClass.cpp
 *
 * Function implementations for the class MoleculeListClass.
 *
 */

#include "Helpers/MemDebug.hpp"

#include <cstring>

#include "World.hpp"
#include "atom.hpp"
#include "bond.hpp"
#include "bondgraph.hpp"
#include "boundary.hpp"
#include "config.hpp"
#include "element.hpp"
#include "helpers.hpp"
#include "linkedcell.hpp"
#include "lists.hpp"
#include "log.hpp"
#include "molecule.hpp"
#include "memoryallocator.hpp"
#include "periodentafel.hpp"
#include "Helpers/Assert.hpp"

#include "Helpers/Assert.hpp"

/*********************************** Functions for class MoleculeListClass *************************/

/** Constructor for MoleculeListClass.
 */
MoleculeListClass::MoleculeListClass(World *_world) :
  Observable("MoleculeListClass"),
  world(_world)
{
  // empty lists
  ListOfMolecules.clear();
  MaxIndex = 1;
};

/** Destructor for MoleculeListClass.
 */
MoleculeListClass::~MoleculeListClass()
{
  DoLog(4) && (Log() << Verbose(4) << "Clearing ListOfMolecules." << endl);
  for(MoleculeList::iterator MolRunner = ListOfMolecules.begin(); MolRunner != ListOfMolecules.end(); ++MolRunner)
    (*MolRunner)->signOff(this);
  ListOfMolecules.clear(); // empty list
};

/** Insert a new molecule into the list and set its number.
 * \param *mol molecule to add to list.
 */
void MoleculeListClass::insert(molecule *mol)
{
  OBSERVE;
  mol->IndexNr = MaxIndex++;
  ListOfMolecules.push_back(mol);
  mol->signOn(this);
};

/** Erases a molecule from the list.
 * \param *mol molecule to add to list.
 */
void MoleculeListClass::erase(molecule *mol)
{
  OBSERVE;
  mol->signOff(this);
  ListOfMolecules.remove(mol);
};

/** Compare whether two molecules are equal.
 * \param *a molecule one
 * \param *n molecule two
 * \return lexical value (-1, 0, +1)
 */
int MolCompare(const void *a, const void *b)
{
  int *aList = NULL, *bList = NULL;
  int Count, Counter, aCounter, bCounter;
  int flag;

  // sort each atom list and put the numbers into a list, then go through
  //Log() << Verbose(0) << "Comparing fragment no. " << *(molecule **)a << " to " << *(molecule **)b << "." << endl;
  // Yes those types are awkward... but check it for yourself it checks out this way
  molecule *const *mol1_ptr= static_cast<molecule *const *>(a);
  molecule *mol1 = *mol1_ptr;
  molecule *const *mol2_ptr= static_cast<molecule *const *>(b);
  molecule *mol2 = *mol2_ptr;
  if (mol1->getAtomCount() < mol2->getAtomCount()) {
    return -1;
  } else {
    if (mol1->getAtomCount() > mol2->getAtomCount())
      return +1;
    else {
      Count = mol1->getAtomCount();
      aList = new int[Count];
      bList = new int[Count];

      // fill the lists
      Counter = 0;
      aCounter = 0;
      bCounter = 0;
      molecule::const_iterator aiter = mol1->begin();
      molecule::const_iterator biter = mol2->begin();
      for (;(aiter != mol1->end()) && (biter != mol2->end());
          ++aiter, ++biter) {
        if ((*aiter)->GetTrueFather() == NULL)
          aList[Counter] = Count + (aCounter++);
        else
          aList[Counter] = (*aiter)->GetTrueFather()->nr;
        if ((*biter)->GetTrueFather() == NULL)
          bList[Counter] = Count + (bCounter++);
        else
          bList[Counter] = (*biter)->GetTrueFather()->nr;
        Counter++;
      }
      // check if AtomCount was for real
      flag = 0;
      if ((aiter == mol1->end()) && (biter != mol2->end())) {
        flag = -1;
      } else {
        if ((aiter != mol1->end()) && (biter == mol2->end()))
          flag = 1;
      }
      if (flag == 0) {
        // sort the lists
        gsl_heapsort(aList, Count, sizeof(int), CompareDoubles);
        gsl_heapsort(bList, Count, sizeof(int), CompareDoubles);
        // compare the lists

        flag = 0;
        for (int i = 0; i < Count; i++) {
          if (aList[i] < bList[i]) {
            flag = -1;
          } else {
            if (aList[i] > bList[i])
              flag = 1;
          }
          if (flag != 0)
            break;
        }
      }
      delete[] (aList);
      delete[] (bList);
      return flag;
    }
  }
  return -1;
};

/** Output of a list of all molecules.
 * \param *out output stream
 */
void MoleculeListClass::Enumerate(ostream *out)
{
  periodentafel *periode = World::getInstance().getPeriode();
  std::map<atomicNumber_t,unsigned int> counts;
  double size=0;
  Vector Origin;

  // header
  (*out) << "Index\tName\t\tAtoms\tFormula\tCenter\tSize" << endl;
  (*out) << "-----------------------------------------------" << endl;
  if (ListOfMolecules.size() == 0)
    (*out) << "\tNone" << endl;
  else {
    Origin.Zero();
    for (MoleculeList::iterator ListRunner = ListOfMolecules.begin(); ListRunner != ListOfMolecules.end(); ListRunner++) {
      // count atoms per element and determine size of bounding sphere
      size=0.;
      for (molecule::const_iterator iter = (*ListRunner)->begin(); iter != (*ListRunner)->end(); ++iter) {
        counts[(*iter)->type->getNumber()]++;
        if ((*iter)->x.DistanceSquared(Origin) > size)
          size = (*iter)->x.DistanceSquared(Origin);
      }
      // output Index, Name, number of atoms, chemical formula
      (*out) << ((*ListRunner)->ActiveFlag ? "*" : " ") << (*ListRunner)->IndexNr << "\t" << (*ListRunner)->name << "\t\t" << (*ListRunner)->getAtomCount() << "\t";

      std::map<atomicNumber_t,unsigned int>::reverse_iterator iter;
      for(iter=counts.rbegin(); iter!=counts.rend();++iter){
        atomicNumber_t Z =(*iter).first;
        (*out) << periode->FindElement(Z)->getSymbol() << (*iter).second;
      }
      // Center and size
      (*out) << "\t" << (*ListRunner)->Center << "\t" << sqrt(size) << endl;
    }
  }
};

/** Returns the molecule with the given index \a index.
 * \param index index of the desired molecule
 * \return pointer to molecule structure, NULL if not found
 */
molecule * MoleculeListClass::ReturnIndex(int index)
{
  for(MoleculeList::iterator ListRunner = ListOfMolecules.begin(); ListRunner != ListOfMolecules.end(); ListRunner++)
    if ((*ListRunner)->IndexNr == index)
      return (*ListRunner);
  return NULL;
};

/** Simple merge of two molecules into one.
 * \param *mol destination molecule
 * \param *srcmol source molecule
 * \return true - merge successful, false - merge failed (probably due to non-existant indices
 */
bool MoleculeListClass::SimpleMerge(molecule *mol, molecule *srcmol)
{
  if (srcmol == NULL)
    return false;

  // put all molecules of src into mol
  for (molecule::iterator iter = srcmol->begin(); !srcmol->empty(); iter=srcmol->begin()) {
    atom * const Walker = *iter;
    srcmol->UnlinkAtom(Walker);
    mol->AddAtom(Walker);
  }

  // remove src
  ListOfMolecules.remove(srcmol);
  World::getInstance().destroyMolecule(srcmol);
  return true;
};

/** Simple add of one molecules into another.
 * \param *mol destination molecule
 * \param *srcmol source molecule
 * \return true - merge successful, false - merge failed (probably due to non-existant indices
 */
bool MoleculeListClass::SimpleAdd(molecule *mol, molecule *srcmol)
{
  if (srcmol == NULL)
    return false;

  // put all molecules of src into mol
  atom *Walker = NULL;
  for (molecule::iterator iter = srcmol->begin(); iter != srcmol->end(); ++iter) {
    Walker = mol->AddCopyAtom((*iter));
    Walker->father = Walker;
  }

  return true;
};

/** Simple merge of a given set of molecules into one.
 * \param *mol destination molecule
 * \param *src index of set of source molecule
 * \param N number of source molecules
 * \return true - merge successful, false - some merges failed (probably due to non-existant indices)
 */
bool MoleculeListClass::SimpleMultiMerge(molecule *mol, int *src, int N)
{
  bool status = true;
  // check presence of all source molecules
  for (int i=0;i<N;i++) {
    molecule *srcmol = ReturnIndex(src[i]);
    status = status && SimpleMerge(mol, srcmol);
  }
  insert(mol);
  return status;
};

/** Simple add of a given set of molecules into one.
 * \param *mol destination molecule
 * \param *src index of set of source molecule
 * \param N number of source molecules
 * \return true - merge successful, false - some merges failed (probably due to non-existant indices)
 */
bool MoleculeListClass::SimpleMultiAdd(molecule *mol, int *src, int N)
{
  bool status = true;
  // check presence of all source molecules
  for (int i=0;i<N;i++) {
    molecule *srcmol = ReturnIndex(src[i]);
    status = status && SimpleAdd(mol, srcmol);
  }
  return status;
};

/** Scatter merge of a given set of molecules into one.
 * Scatter merge distributes the molecules in such a manner that they don't overlap.
 * \param *mol destination molecule
 * \param *src index of set of source molecule
 * \param N number of source molecules
 * \return true - merge successful, false - merge failed (probably due to non-existant indices
 * \TODO find scatter center for each src molecule
 */
bool MoleculeListClass::ScatterMerge(molecule *mol, int *src, int N)
{
  // check presence of all source molecules
  for (int i=0;i<N;i++) {
    // get pointer to src molecule
    molecule *srcmol = ReturnIndex(src[i]);
    if (srcmol == NULL)
      return false;
  }
  // adapt each Center
  for (int i=0;i<N;i++) {
    // get pointer to src molecule
    molecule *srcmol = ReturnIndex(src[i]);
    //srcmol->Center.Zero();
    srcmol->Translate(&srcmol->Center);
  }
  // perform a simple multi merge
  SimpleMultiMerge(mol, src, N);
  return true;
};

/** Embedding merge of a given set of molecules into one.
 * Embedding merge inserts one molecule into the other.
 * \param *mol destination molecule (fixed one)
 * \param *srcmol source molecule (variable one, where atoms are taken from)
 * \return true - merge successful, false - merge failed (probably due to non-existant indices)
 * \TODO linked cell dimensions for boundary points has to be as big as inner diameter!
 */
bool MoleculeListClass::EmbedMerge(molecule *mol, molecule *srcmol)
{
  LinkedCell *LCList = NULL;
  Tesselation *TesselStruct = NULL;
  if ((srcmol == NULL) || (mol == NULL)) {
    DoeLog(1) && (eLog()<< Verbose(1) << "Either fixed or variable molecule is given as NULL." << endl);
    return false;
  }

  // calculate envelope for *mol
  LCList = new LinkedCell(mol, 8.);
  FindNonConvexBorder(mol, TesselStruct, (const LinkedCell *&)LCList, 4., NULL);
  if (TesselStruct == NULL) {
    DoeLog(1) && (eLog()<< Verbose(1) << "Could not tesselate the fixed molecule." << endl);
    return false;
  }
  delete(LCList);
  LCList = new LinkedCell(TesselStruct, 8.);  // re-create with boundary points only!

  // prepare index list for bonds
  atom ** CopyAtoms = new atom*[srcmol->getAtomCount()];
  for(int i=0;i<srcmol->getAtomCount();i++)
    CopyAtoms[i] = NULL;

  // for each of the source atoms check whether we are in- or outside and add copy atom
  int nr=0;
  for (molecule::const_iterator iter = srcmol->begin(); iter != srcmol->end(); ++iter) {
    DoLog(2) && (Log() << Verbose(2) << "INFO: Current Walker is " << **iter << "." << endl);
    if (!TesselStruct->IsInnerPoint((*iter)->x, LCList)) {
      CopyAtoms[(*iter)->nr] = (*iter)->clone();
      mol->AddAtom(CopyAtoms[(*iter)->nr]);
      nr++;
    } else {
      // do nothing
    }
  }
  DoLog(1) && (Log() << Verbose(1) << nr << " of " << srcmol->getAtomCount() << " atoms have been merged.");

  // go through all bonds and add as well
  for(molecule::iterator AtomRunner = srcmol->begin(); AtomRunner != srcmol->end(); ++AtomRunner)
    for(BondList::iterator BondRunner = (*AtomRunner)->ListOfBonds.begin(); BondRunner != (*AtomRunner)->ListOfBonds.end(); ++BondRunner)
      if ((*BondRunner)->leftatom == *AtomRunner) {
        DoLog(3) && (Log() << Verbose(3) << "Adding Bond between " << *CopyAtoms[(*BondRunner)->leftatom->nr] << " and " << *CopyAtoms[(*BondRunner)->rightatom->nr]<< "." << endl);
        mol->AddBond(CopyAtoms[(*BondRunner)->leftatom->nr], CopyAtoms[(*BondRunner)->rightatom->nr], (*BondRunner)->BondDegree);
      }
  delete(LCList);
  return true;
};

/** Simple output of the pointers in ListOfMolecules.
 * \param *out output stream
 */
void MoleculeListClass::Output(ofstream *out)
{
  DoLog(1) && (Log() << Verbose(1) << "MoleculeList: ");
  for (MoleculeList::iterator ListRunner = ListOfMolecules.begin(); ListRunner != ListOfMolecules.end(); ListRunner++)
    DoLog(0) && (Log() << Verbose(0) << *ListRunner << "\t");
  DoLog(0) && (Log() << Verbose(0) << endl);
};

/** Calculates necessary hydrogen correction due to unwanted interaction between saturated ones.
 * If for a pair of two hydrogen atoms a and b, at least is a saturated one, and a and b are not
 * bonded to the same atom, then we add for this pair a correction term constructed from a Morse
 * potential function fit to QM calculations with respecting to the interatomic hydrogen distance.
 * \param &path path to file
 */
bool MoleculeListClass::AddHydrogenCorrection(std::string &path)
{
  bond *Binder = NULL;
  double ***FitConstant = NULL, **correction = NULL;
  int a, b;
  ofstream output;
  ifstream input;
  string line;
  stringstream zeile;
  double distance;
  char ParsedLine[1023];
  double tmp;
  char *FragmentNumber = NULL;

  DoLog(1) && (Log() << Verbose(1) << "Saving hydrogen saturation correction ... ");
  // 0. parse in fit constant files that should have the same dimension as the final energy files
  // 0a. find dimension of matrices with constants
  line = path;
  line += "1";
  line += FITCONSTANTSUFFIX;
  input.open(line.c_str());
  if (input.fail()) {
    DoLog(1) && (Log() << Verbose(1) << endl << "Unable to open " << line << ", is the directory correct?" << endl);
    return false;
  }
  a = 0;
  b = -1; // we overcount by one
  while (!input.eof()) {
    input.getline(ParsedLine, 1023);
    zeile.str(ParsedLine);
    int i = 0;
    while (!zeile.eof()) {
      zeile >> distance;
      i++;
    }
    if (i > a)
      a = i;
    b++;
  }
  DoLog(0) && (Log() << Verbose(0) << "I recognized " << a << " columns and " << b << " rows, ");
  input.close();

  // 0b. allocate memory for constants
  FitConstant = new double**[3];
  for (int k = 0; k < 3; k++) {
    FitConstant[k] = new double*[a];
    for (int i = a; i--;) {
      FitConstant[k][i] = new double[b];
      for (int j = b; j--;) {
        FitConstant[k][i][j] = 0.;
      }
    }
  }
  // 0c. parse in constants
  for (int i = 0; i < 3; i++) {
    line = path;
    line.append("/");
    line += FRAGMENTPREFIX;
    sprintf(ParsedLine, "%d", i + 1);
    line += ParsedLine;
    line += FITCONSTANTSUFFIX;
    input.open(line.c_str());
    if (input == NULL) {
      DoeLog(0) && (eLog()<< Verbose(0) << endl << "Unable to open " << line << ", is the directory correct?" << endl);
      performCriticalExit();
      return false;
    }
    int k = 0, l;
    while ((!input.eof()) && (k < b)) {
      input.getline(ParsedLine, 1023);
      //Log() << Verbose(0) << "Current Line: " << ParsedLine << endl;
      zeile.str(ParsedLine);
      zeile.clear();
      l = 0;
      while ((!zeile.eof()) && (l < a)) {
        zeile >> FitConstant[i][l][k];
        //Log() << Verbose(0) << FitConstant[i][l][k] << "\t";
        l++;
      }
      //Log() << Verbose(0) << endl;
      k++;
    }
    input.close();
  }
  for (int k = 0; k < 3; k++) {
    DoLog(0) && (Log() << Verbose(0) << "Constants " << k << ":" << endl);
    for (int j = 0; j < b; j++) {
      for (int i = 0; i < a; i++) {
        DoLog(0) && (Log() << Verbose(0) << FitConstant[k][i][j] << "\t");
      }
      DoLog(0) && (Log() << Verbose(0) << endl);
    }
    DoLog(0) && (Log() << Verbose(0) << endl);
  }

  // 0d. allocate final correction matrix
  correction = new double*[a];
  for (int i = a; i--;)
    correction[i] = new double[b];

  // 1a. go through every molecule in the list
  for (MoleculeList::iterator ListRunner = ListOfMolecules.begin(); ListRunner != ListOfMolecules.end(); ListRunner++) {
    // 1b. zero final correction matrix
    for (int k = a; k--;)
      for (int j = b; j--;)
        correction[k][j] = 0.;
    // 2. take every hydrogen that is a saturated one
    for (molecule::const_iterator iter = (*ListRunner)->begin(); iter != (*ListRunner)->end(); ++iter) {
      //Log() << Verbose(1) << "(*iter): " << *(*iter) << " with first bond " << *((*iter)->ListOfBonds.begin()) << "." << endl;
      if (((*iter)->type->Z == 1) && (((*iter)->father == NULL)
          || ((*iter)->father->type->Z != 1))) { // if it's a hydrogen
        for (molecule::const_iterator runner = (*ListRunner)->begin(); runner != (*ListRunner)->end(); ++runner) {
          //Log() << Verbose(2) << "Runner: " << *(*runner) << " with first bond " << *((*iter)->ListOfBonds.begin()) << "." << endl;
          // 3. take every other hydrogen that is the not the first and not bound to same bonding partner
          Binder = *((*runner)->ListOfBonds.begin());
          if (((*runner)->type->Z == 1) && ((*runner)->nr > (*iter)->nr) && (Binder->GetOtherAtom((*runner)) != Binder->GetOtherAtom((*iter)))) { // (hydrogens have only one bonding partner!)
            // 4. evaluate the morse potential for each matrix component and add up
            distance = (*runner)->x.distance((*iter)->x);
            //Log() << Verbose(0) << "Fragment " << (*ListRunner)->name << ": " << *(*runner) << "<= " << distance << "=>" << *(*iter) << ":" << endl;
            for (int k = 0; k < a; k++) {
              for (int j = 0; j < b; j++) {
                switch (k) {
                  case 1:
                  case 7:
                  case 11:
                    tmp = pow(FitConstant[0][k][j] * (1. - exp(-FitConstant[1][k][j] * (distance - FitConstant[2][k][j]))), 2);
                    break;
                  default:
                    tmp = FitConstant[0][k][j] * pow(distance, FitConstant[1][k][j]) + FitConstant[2][k][j];
                };
                correction[k][j] -= tmp; // ground state is actually lower (disturbed by additional interaction)
                //Log() << Verbose(0) << tmp << "\t";
              }
              //Log() << Verbose(0) << endl;
            }
            //Log() << Verbose(0) << endl;
          }
        }
      }
    }
    // 5. write final matrix to file
    line = path;
    line.append("/");
    line += FRAGMENTPREFIX;
    FragmentNumber = FixedDigitNumber(ListOfMolecules.size(), (*ListRunner)->IndexNr);
    line += FragmentNumber;
    delete[] (FragmentNumber);
    line += HCORRECTIONSUFFIX;
    output.open(line.c_str());
    output << "Time\t\tTotal\t\tKinetic\t\tNonLocal\tCorrelation\tExchange\tPseudo\t\tHartree\t\t-Gauss\t\tEwald\t\tIonKin\t\tETotal" << endl;
    for (int j = 0; j < b; j++) {
      for (int i = 0; i < a; i++)
        output << correction[i][j] << "\t";
      output << endl;
    }
    output.close();
  }
  for (int i = a; i--;)
    delete[](correction[i]);
  delete[](correction);

  line = path;
  line.append("/");
  line += HCORRECTIONSUFFIX;
  output.open(line.c_str());
  output << "Time\t\tTotal\t\tKinetic\t\tNonLocal\tCorrelation\tExchange\tPseudo\t\tHartree\t\t-Gauss\t\tEwald\t\tIonKin\t\tETotal" << endl;
  for (int j = 0; j < b; j++) {
    for (int i = 0; i < a; i++)
      output << 0 << "\t";
    output << endl;
  }
  output.close();
  // 6. free memory of parsed matrices
  for (int k = 0; k < 3; k++) {
    for (int i = a; i--;) {
      delete[](FitConstant[k][i]);
    }
    delete[](FitConstant[k]);
  }
  delete[](FitConstant);
  DoLog(0) && (Log() << Verbose(0) << "done." << endl);
  return true;
};

/** Store force indices, i.e. the connection between the nuclear index in the total molecule config and the respective atom in fragment config.
 * \param &path path to file
 * \param *SortIndex Index to map from the BFS labeling to the sequence how of Ion_Type in the config
 * \return true - file written successfully, false - writing failed
 */
bool MoleculeListClass::StoreForcesFile(std::string &path, int *SortIndex)
{
  bool status = true;
  string filename(path);
  filename += FORCESFILE;
  ofstream ForcesFile(filename.c_str());
  periodentafel *periode=World::getInstance().getPeriode();

  // open file for the force factors
  DoLog(1) && (Log() << Verbose(1) << "Saving  force factors ... ");
  if (!ForcesFile.fail()) {
    //Log() << Verbose(1) << "Final AtomicForcesList: ";
    //output << prefix << "Forces" << endl;
    for (MoleculeList::iterator ListRunner = ListOfMolecules.begin(); ListRunner != ListOfMolecules.end(); ListRunner++) {
      periodentafel::const_iterator elemIter;
      for(elemIter=periode->begin();elemIter!=periode->end();++elemIter){
        if ((*ListRunner)->ElementsInMolecule[(*elemIter).first]) { // if this element got atoms
          for(molecule::iterator atomIter = (*ListRunner)->begin(); atomIter !=(*ListRunner)->end();++atomIter){
            if ((*atomIter)->type->getNumber() == (*elemIter).first) {
              if (((*atomIter)->GetTrueFather() != NULL) && ((*atomIter)->GetTrueFather() != (*atomIter))) {// if there is a rea
                //Log() << Verbose(0) << "Walker is " << *Walker << " with true father " << *( Walker->GetTrueFather()) << ", it
                ForcesFile << SortIndex[(*atomIter)->GetTrueFather()->nr] << "\t";
              } else
                // otherwise a -1 to indicate an added saturation hydrogen
                ForcesFile << "-1\t";
            }
          }
        }
      }
      ForcesFile << endl;
    }
    ForcesFile.close();
    DoLog(1) && (Log() << Verbose(1) << "done." << endl);
  } else {
    status = false;
    DoLog(1) && (Log() << Verbose(1) << "failed to open file " << filename << "." << endl);
  }
  ForcesFile.close();

  return status;
};

/** Writes a config file for each molecule in the given \a **FragmentList.
 * \param *out output stream for debugging
 * \param &prefix path and prefix to the fragment config files
 * \param *SortIndex Index to map from the BFS labeling to the sequence how of Ion_Type in the config
 * \return true - success (each file was written), false - something went wrong.
 */
bool MoleculeListClass::OutputConfigForListOfFragments(std::string &prefix, int *SortIndex)
{
  ofstream outputFragment;
  std::string FragmentName;
  char PathBackup[MAXSTRINGSIZE];
  bool result = true;
  bool intermediateResult = true;
  Vector BoxDimension;
  char *FragmentNumber = NULL;
  char *path = NULL;
  int FragmentCounter = 0;
  ofstream output;
  double cell_size_backup[6];
  double * const cell_size = World::getInstance().getDomain();

  // backup cell_size
  for (int i=0;i<6;i++)
    cell_size_backup[i] = cell_size[i];
  // store the fragments as config and as xyz
  for (MoleculeList::iterator ListRunner = ListOfMolecules.begin(); ListRunner != ListOfMolecules.end(); ListRunner++) {
    // save default path as it is changed for each fragment
    path = World::getInstance().getConfig()->GetDefaultPath();
    if (path != NULL)
      strcpy(PathBackup, path);
    else {
      DoeLog(0) && (eLog()<< Verbose(0) << "OutputConfigForListOfFragments: NULL default path obtained from config!" << endl);
      performCriticalExit();
    }

    // correct periodic
    (*ListRunner)->ScanForPeriodicCorrection();

    // output xyz file
    FragmentNumber = FixedDigitNumber(ListOfMolecules.size(), FragmentCounter++);
    FragmentName = prefix + FragmentNumber + ".conf.xyz";
    outputFragment.open(FragmentName.c_str(), ios::out);
    DoLog(2) && (Log() << Verbose(2) << "Saving bond fragment No. " << FragmentNumber << "/" << FragmentCounter - 1 << " as XYZ ...");
    if ((intermediateResult = (*ListRunner)->OutputXYZ(&outputFragment)))
      DoLog(0) && (Log() << Verbose(0) << " done." << endl);
    else
      DoLog(0) && (Log() << Verbose(0) << " failed." << endl);
    result = result && intermediateResult;
    outputFragment.close();
    outputFragment.clear();

    // list atoms in fragment for debugging
    DoLog(2) && (Log() << Verbose(2) << "Contained atoms: ");
    for (molecule::const_iterator iter = (*ListRunner)->begin(); iter != (*ListRunner)->end(); ++iter) {
      DoLog(0) && (Log() << Verbose(0) << (*iter)->getName() << " ");
    }
    DoLog(0) && (Log() << Verbose(0) << endl);

    // center on edge
    (*ListRunner)->CenterEdge(&BoxDimension);
    (*ListRunner)->SetBoxDimension(&BoxDimension); // update Box of atoms by boundary
    int j = -1;
    for (int k = 0; k < NDIM; k++) {
      j += k + 1;
      BoxDimension[k] = 2.5 * (World::getInstance().getConfig()->GetIsAngstroem() ? 1. : 1. / AtomicLengthToAngstroem);
      cell_size[j] = BoxDimension[k] * 2.;
    }
    (*ListRunner)->Translate(&BoxDimension);

    // also calculate necessary orbitals
    (*ListRunner)->CountElements(); // this is a bugfix, atoms should shoulds actually be added correctly to this fragment
    //(*ListRunner)->CalculateOrbitals(*World::getInstance().getConfig);

    // change path in config
    FragmentName = PathBackup;
    FragmentName += "/";
    FragmentName += FRAGMENTPREFIX;
    FragmentName += FragmentNumber;
    FragmentName += "/";
    World::getInstance().getConfig()->SetDefaultPath(FragmentName.c_str());

    // and save as config
    FragmentName = prefix + FragmentNumber + ".conf";
    DoLog(2) && (Log() << Verbose(2) << "Saving bond fragment No. " << FragmentNumber << "/" << FragmentCounter - 1 << " as config ...");
    if ((intermediateResult = World::getInstance().getConfig()->Save(FragmentName.c_str(), (*ListRunner)->elemente, (*ListRunner))))
      DoLog(0) && (Log() << Verbose(0) << " done." << endl);
    else
      DoLog(0) && (Log() << Verbose(0) << " failed." << endl);
    result = result && intermediateResult;

    // restore old config
    World::getInstance().getConfig()->SetDefaultPath(PathBackup);

    // and save as mpqc input file
    FragmentName = prefix + FragmentNumber + ".conf";
    DoLog(2) && (Log() << Verbose(2) << "Saving bond fragment No. " << FragmentNumber << "/" << FragmentCounter - 1 << " as mpqc input ...");
    if ((intermediateResult = World::getInstance().getConfig()->SaveMPQC(FragmentName.c_str(), (*ListRunner))))
      DoLog(2) && (Log() << Verbose(2) << " done." << endl);
    else
      DoLog(0) && (Log() << Verbose(0) << " failed." << endl);

    result = result && intermediateResult;
    //outputFragment.close();
    //outputFragment.clear();
    delete[](FragmentNumber);
  }
  DoLog(0) && (Log() << Verbose(0) << " done." << endl);

  // printing final number
  DoLog(2) && (Log() << Verbose(2) << "Final number of fragments: " << FragmentCounter << "." << endl);

  // restore cell_size
  for (int i=0;i<6;i++)
    cell_size[i] = cell_size_backup[i];

  return result;
};

/** Counts the number of molecules with the molecule::ActiveFlag set.
 * \return number of molecules with ActiveFlag set to true.
 */
int MoleculeListClass::NumberOfActiveMolecules()
{
  int count = 0;
  for (MoleculeList::iterator ListRunner = ListOfMolecules.begin(); ListRunner != ListOfMolecules.end(); ListRunner++)
    count += ((*ListRunner)->ActiveFlag ? 1 : 0);
  return count;
};

/** Dissects given \a *mol into connected subgraphs and inserts them as new molecules but with old atoms into \a this.
 * \param *out output stream for debugging
 * \param *periode periodentafel
 * \param *configuration config with BondGraph
 */
void MoleculeListClass::DissectMoleculeIntoConnectedSubgraphs(const periodentafel * const periode, config * const configuration)
{
  // 0a. remove all present molecules
  vector<molecule *> allmolecules = World::getInstance().getAllMolecules();
  for (vector<molecule *>::iterator MolRunner = allmolecules.begin(); MolRunner != allmolecules.end(); ++MolRunner) {
    erase(*MolRunner);
    World::getInstance().destroyMolecule(*MolRunner);
  }
  // 0b. remove all bonds and construct a molecule with all atoms
  molecule *mol = World::getInstance().createMolecule();
  vector <atom *> allatoms = World::getInstance().getAllAtoms();
  for(vector<atom *>::iterator AtomRunner = allatoms.begin(); AtomRunner != allatoms.end(); ++AtomRunner) {
    for(BondList::iterator BondRunner = (*AtomRunner)->ListOfBonds.begin(); !(*AtomRunner)->ListOfBonds.empty(); BondRunner = (*AtomRunner)->ListOfBonds.begin())
      delete(*BondRunner);
    mol->AddAtom(*AtomRunner);
  }

  // 1. dissect the molecule into connected subgraphs
  if (configuration->BG != NULL) {
    if (!configuration->BG->ConstructBondGraph(mol)) {
      World::getInstance().destroyMolecule(mol);
      DoeLog(1) && (eLog()<< Verbose(1) << "There are no bonds." << endl);
      return;
    }
  } else {
    DoeLog(1) && (eLog()<< Verbose(1) << "There is no BondGraph class present to create bonds." << endl);
    return;
  }

  // 2. scan for connected subgraphs
  MoleculeLeafClass *Subgraphs = NULL;      // list of subgraphs from DFS analysis
  class StackClass<bond *> *BackEdgeStack = NULL;
  Subgraphs = mol->DepthFirstSearchAnalysis(BackEdgeStack);
  delete(BackEdgeStack);
  if ((Subgraphs == NULL) || (Subgraphs->next == NULL)) {
    World::getInstance().destroyMolecule(mol);
    DoeLog(1) && (eLog()<< Verbose(1) << "There are no atoms." << endl);
    return;
  }

  // 3. dissect (the following construct is needed to have the atoms not in the order of the DFS, but in
  // the original one as parsed in)
  // TODO: Optimize this, when molecules just contain pointer list of global atoms!

  // 4a. create array of molecules to fill
  const int MolCount = Subgraphs->next->Count();
  char number[MAXSTRINGSIZE];
  molecule **molecules = new molecule *[MolCount];
  MoleculeLeafClass *MolecularWalker = Subgraphs;
  for (int i=0;i<MolCount;i++) {
    MolecularWalker = MolecularWalker->next;
    molecules[i] = World::getInstance().createMolecule();
    molecules[i]->ActiveFlag = true;
    strncpy(molecules[i]->name, mol->name, MAXSTRINGSIZE);
    if (MolCount > 1) {
      sprintf(number, "-%d", i+1);
      strncat(molecules[i]->name, number, MAXSTRINGSIZE - strlen(mol->name) - 1);
    }
    DoLog(1) && (Log() << Verbose(1) << "MolName is " << molecules[i]->name << ", id is " << molecules[i]->getId() << endl);
    for (molecule::iterator iter = MolecularWalker->Leaf->begin(); iter != MolecularWalker->Leaf->end(); ++iter) {
      DoLog(1) && (Log() << Verbose(1) << **iter << endl);
    }
    insert(molecules[i]);
  }

  // 4b. create and fill map of which atom is associated to which connected molecule (note, counting starts at 1)
  int FragmentCounter = 0;
  map<int, atom *> AtomToFragmentMap;
  MolecularWalker = Subgraphs;
  while (MolecularWalker->next != NULL) {
    MolecularWalker = MolecularWalker->next;
    for (molecule::iterator iter = MolecularWalker->Leaf->begin(); !MolecularWalker->Leaf->empty(); iter = MolecularWalker->Leaf->begin()) {
      atom * Walker = *iter;
      DoLog(1) && (Log() << Verbose(1) << "Re-linking " << Walker << "..." << endl);
      MolecularWalker->Leaf->erase(iter);
      molecules[FragmentCounter]->AddAtom(Walker);    // counting starts at 1
    }
    FragmentCounter++;
  }
  World::getInstance().destroyMolecule(mol);

  // 4d. we don't need to redo bonds, as they are connected subgraphs and still maintain their ListOfBonds, but we have to remove them from first..last list
  // TODO: check whether this is really not needed anymore
  // 4e. free Leafs
  MolecularWalker = Subgraphs;
  while (MolecularWalker->next != NULL) {
    MolecularWalker = MolecularWalker->next;
    delete(MolecularWalker->previous);
  }
  delete(MolecularWalker);
  delete[](molecules);
  DoLog(1) && (Log() << Verbose(1) << "I scanned " << FragmentCounter << " molecules." << endl);
};

/** Count all atoms in each molecule.
 * \return number of atoms in the MoleculeListClass.
 * TODO: the inner loop should be done by some (double molecule::CountAtom()) function
 */
int MoleculeListClass::CountAllAtoms() const
{
  int AtomNo = 0;
  for (MoleculeList::const_iterator MolWalker = ListOfMolecules.begin(); MolWalker != ListOfMolecules.end(); MolWalker++) {
    AtomNo += (*MolWalker)->size();
  }
  return AtomNo;
}

/***********
 * Methods Moved here from the menus
 */

void MoleculeListClass::flipChosen() {
  int j;
  Log() << Verbose(0) << "Enter index of molecule: ";
  cin >> j;
  for(MoleculeList::iterator ListRunner = ListOfMolecules.begin(); ListRunner != ListOfMolecules.end(); ListRunner++)
    if ((*ListRunner)->IndexNr == j)
      (*ListRunner)->ActiveFlag = !(*ListRunner)->ActiveFlag;
}

void MoleculeListClass::createNewMolecule(periodentafel *periode) {
  OBSERVE;
  molecule *mol = NULL;
  mol = World::getInstance().createMolecule();
  insert(mol);
};

void MoleculeListClass::loadFromXYZ(periodentafel *periode){
  molecule *mol = NULL;
  Vector center;
  char filename[MAXSTRINGSIZE];
  Log() << Verbose(0) << "Format should be XYZ with: ShorthandOfElement\tX\tY\tZ" << endl;
  mol = World::getInstance().createMolecule();
  do {
    Log() << Verbose(0) << "Enter file name: ";
    cin >> filename;
  } while (!mol->AddXYZFile(filename));
  mol->SetNameFromFilename(filename);
  // center at set box dimensions
  mol->CenterEdge(&center);
  World::getInstance().getDomain()[0] = center[0];
  World::getInstance().getDomain()[1] = 0;
  World::getInstance().getDomain()[2] = center[1];
  World::getInstance().getDomain()[3] = 0;
  World::getInstance().getDomain()[4] = 0;
  World::getInstance().getDomain()[5] = center[2];
  insert(mol);
}

void MoleculeListClass::setMoleculeFilename() {
  char filename[MAXSTRINGSIZE];
  int nr;
  molecule *mol = NULL;
  do {
    Log() << Verbose(0) << "Enter index of molecule: ";
    cin >> nr;
    mol = ReturnIndex(nr);
  } while (mol == NULL);
  Log() << Verbose(0) << "Enter name: ";
  cin >> filename;
  mol->SetNameFromFilename(filename);
}

void MoleculeListClass::parseXYZIntoMolecule(){
  char filename[MAXSTRINGSIZE];
  int nr;
  molecule *mol = NULL;
  mol = NULL;
  do {
   Log() << Verbose(0) << "Enter index of molecule: ";
   cin >> nr;
   mol = ReturnIndex(nr);
  } while (mol == NULL);
  Log() << Verbose(0) << "Format should be XYZ with: ShorthandOfElement\tX\tY\tZ" << endl;
  do {
   Log() << Verbose(0) << "Enter file name: ";
   cin >> filename;
  } while (!mol->AddXYZFile(filename));
  mol->SetNameFromFilename(filename);
};

void MoleculeListClass::eraseMolecule(){
  int nr;
  molecule *mol = NULL;
  Log() << Verbose(0) << "Enter index of molecule: ";
  cin >> nr;
  for(MoleculeList::iterator ListRunner = ListOfMolecules.begin(); ListRunner != ListOfMolecules.end(); ListRunner++)
    if (nr == (*ListRunner)->IndexNr) {
      mol = *ListRunner;
      ListOfMolecules.erase(ListRunner);
      World::getInstance().destroyMolecule(mol);
      break;
    }
};


/******************************************* Class MoleculeLeafClass ************************************************/

/** Constructor for MoleculeLeafClass root leaf.
 * \param *Up Leaf on upper level
 * \param *PreviousLeaf NULL - We are the first leaf on this level, otherwise points to previous in list
 */
//MoleculeLeafClass::MoleculeLeafClass(MoleculeLeafClass *Up = NULL, MoleculeLeafClass *Previous = NULL)
MoleculeLeafClass::MoleculeLeafClass(MoleculeLeafClass *PreviousLeaf = NULL)
{
  //  if (Up != NULL)
  //    if (Up->DownLeaf == NULL) // are we the first down leaf for the upper leaf?
  //      Up->DownLeaf = this;
  //  UpLeaf = Up;
  //  DownLeaf = NULL;
  Leaf = NULL;
  previous = PreviousLeaf;
  if (previous != NULL) {
    MoleculeLeafClass *Walker = previous->next;
    previous->next = this;
    next = Walker;
  } else {
    next = NULL;
  }
};

/** Destructor for MoleculeLeafClass.
 */
MoleculeLeafClass::~MoleculeLeafClass()
{
  //  if (DownLeaf != NULL) {// drop leaves further down
  //    MoleculeLeafClass *Walker = DownLeaf;
  //    MoleculeLeafClass *Next;
  //    do {
  //      Next = Walker->NextLeaf;
  //      delete(Walker);
  //      Walker = Next;
  //    } while (Walker != NULL);
  //    // Last Walker sets DownLeaf automatically to NULL
  //  }
  // remove the leaf itself
  if (Leaf != NULL) {
    World::getInstance().destroyMolecule(Leaf);
    Leaf = NULL;
  }
  // remove this Leaf from level list
  if (previous != NULL)
    previous->next = next;
  //  } else { // we are first in list (connects to UpLeaf->DownLeaf)
  //    if ((NextLeaf != NULL) && (NextLeaf->UpLeaf == NULL))
  //      NextLeaf->UpLeaf = UpLeaf;  // either null as we are top level or the upleaf of the first node
  //    if (UpLeaf != NULL)
  //      UpLeaf->DownLeaf = NextLeaf;  // either null as we are only leaf or NextLeaf if we are just the first
  //  }
  //  UpLeaf = NULL;
  if (next != NULL) // are we last in list
    next->previous = previous;
  next = NULL;
  previous = NULL;
};

/** Adds \a molecule leaf to the tree.
 * \param *ptr ptr to molecule to be added
 * \param *Previous previous MoleculeLeafClass referencing level and which on the level
 * \return true - success, false - something went wrong
 */
bool MoleculeLeafClass::AddLeaf(molecule *ptr, MoleculeLeafClass *Previous)
{
  return false;
};

/** Fills the bond structure of this chain list subgraphs that are derived from a complete \a *reference molecule.
 * Calls this routine in each MoleculeLeafClass::next subgraph if it's not NULL.
 * \param *out output stream for debugging
 * \param *reference reference molecule with the bond structure to be copied
 * \param **&ListOfLocalAtoms Lookup table for this subgraph and index of each atom in \a *reference, may be NULL on start, then it is filled
 * \param FreeList true - ***ListOfLocalAtoms is free'd before return, false - it is not
 * \return true - success, false - faoilure
 */
bool MoleculeLeafClass::FillBondStructureFromReference(const molecule * const reference, atom **&ListOfLocalAtoms, bool FreeList)
{
  atom *OtherWalker = NULL;
  atom *Father = NULL;
  bool status = true;
  int AtomNo;

  DoLog(1) && (Log() << Verbose(1) << "Begin of FillBondStructureFromReference." << endl);
  // fill ListOfLocalAtoms if NULL was given
  if (!FillListOfLocalAtoms(ListOfLocalAtoms, reference->getAtomCount(), FreeList)) {
    DoLog(1) && (Log() << Verbose(1) << "Filling of ListOfLocalAtoms failed." << endl);
    return false;
  }

  if (status) {
    DoLog(1) && (Log() << Verbose(1) << "Creating adjacency list for subgraph " << Leaf << "." << endl);
    // remove every bond from the list
    for(molecule::iterator AtomRunner = Leaf->begin(); AtomRunner != Leaf->end(); ++AtomRunner)
      for(BondList::iterator BondRunner = (*AtomRunner)->ListOfBonds.begin(); !(*AtomRunner)->ListOfBonds.empty(); BondRunner = (*AtomRunner)->ListOfBonds.begin())
        if ((*BondRunner)->leftatom == *AtomRunner)
          delete((*BondRunner));

    for(molecule::const_iterator iter = Leaf->begin(); iter != Leaf->end(); ++iter) {
      Father = (*iter)->GetTrueFather();
      AtomNo = Father->nr; // global id of the current walker
      for (BondList::const_iterator Runner = Father->ListOfBonds.begin(); Runner != Father->ListOfBonds.end(); (++Runner)) {
        OtherWalker = ListOfLocalAtoms[(*Runner)->GetOtherAtom((*iter)->GetTrueFather())->nr]; // local copy of current bond partner of walker
        if (OtherWalker != NULL) {
          if (OtherWalker->nr > (*iter)->nr)
            Leaf->AddBond((*iter), OtherWalker, (*Runner)->BondDegree);
        } else {
          DoLog(1) && (Log() << Verbose(1) << "OtherWalker = ListOfLocalAtoms[" << (*Runner)->GetOtherAtom((*iter)->GetTrueFather())->nr << "] is NULL!" << endl);
          status = false;
        }
      }
    }
  }

  if ((FreeList) && (ListOfLocalAtoms != NULL)) {
    // free the index lookup list
    delete[](ListOfLocalAtoms);
  }
  DoLog(1) && (Log() << Verbose(1) << "End of FillBondStructureFromReference." << endl);
  return status;
};

/** Fills the root stack for sites to be used as root in fragmentation depending on order or adaptivity criteria
 * Again, as in \sa FillBondStructureFromReference steps recursively through each Leaf in this chain list of molecule's.
 * \param *out output stream for debugging
 * \param *&RootStack stack to be filled
 * \param *AtomMask defines true/false per global Atom::nr to mask in/out each nuclear site
 * \param &FragmentCounter counts through the fragments in this MoleculeLeafClass
 * \return true - stack is non-empty, fragmentation necessary, false - stack is empty, no more sites to update
 */
bool MoleculeLeafClass::FillRootStackForSubgraphs(KeyStack *&RootStack, bool *AtomMask, int &FragmentCounter)
{
  atom *Father = NULL;

  if (RootStack != NULL) {
    // find first root candidates
    if (&(RootStack[FragmentCounter]) != NULL) {
      RootStack[FragmentCounter].clear();
      for(molecule::const_iterator iter = Leaf->begin(); iter != Leaf->end(); ++iter) {
        Father = (*iter)->GetTrueFather();
        if (AtomMask[Father->nr]) // apply mask
#ifdef ADDHYDROGEN
          if ((*iter)->type->Z != 1) // skip hydrogen
#endif
          RootStack[FragmentCounter].push_front((*iter)->nr);
      }
      if (next != NULL)
        next->FillRootStackForSubgraphs(RootStack, AtomMask, ++FragmentCounter);
    } else {
      DoLog(1) && (Log() << Verbose(1) << "Rootstack[" << FragmentCounter << "] is NULL." << endl);
      return false;
    }
    FragmentCounter--;
    return true;
  } else {
    DoLog(1) && (Log() << Verbose(1) << "Rootstack is NULL." << endl);
    return false;
  }
};

/** Fills a lookup list of father's Atom::nr -> atom for each subgraph.
 * \param *out output stream from debugging
 * \param **&ListOfLocalAtoms Lookup table for each subgraph and index of each atom in global molecule, may be NULL on start, then it is filled
 * \param GlobalAtomCount number of atoms in the complete molecule
 * \param &FreeList true - ***ListOfLocalAtoms is free'd before return, false - it is not
 * \return true - success, false - failure (ListOfLocalAtoms != NULL)
 */
bool MoleculeLeafClass::FillListOfLocalAtoms(atom **&ListOfLocalAtoms, const int GlobalAtomCount, bool &FreeList)
{
  bool status = true;

  if (ListOfLocalAtoms == NULL) { // allocate and fill list of this fragment/subgraph
    status = status && Leaf->CreateFatherLookupTable(ListOfLocalAtoms, GlobalAtomCount);
    FreeList = FreeList && true;
  } else
    return false;

  return status;
};

/** The indices per keyset are compared to the respective father's Atom::nr in each subgraph and thus put into \a **&FragmentList.
 * \param *out output stream fro debugging
 * \param *reference reference molecule with the bond structure to be copied
 * \param *KeySetList list with all keysets
 * \param ***ListOfLocalAtoms Lookup table for each subgraph and index of each atom in global molecule, may be NULL on start, then it is filled
 * \param **&FragmentList list to be allocated and returned
 * \param &FragmentCounter counts the fragments as we move along the list
 * \param FreeList true - ***ListOfLocalAtoms is free'd before return, false - it is not
 * \retuen true - success, false - failure
 */
bool MoleculeLeafClass::AssignKeySetsToFragment(molecule *reference, Graph *KeySetList, atom ***&ListOfLocalAtoms, Graph **&FragmentList, int &FragmentCounter, bool FreeList)
{
  bool status = true;
  int KeySetCounter = 0;

  DoLog(1) && (Log() << Verbose(1) << "Begin of AssignKeySetsToFragment." << endl);
  // fill ListOfLocalAtoms if NULL was given
  if (!FillListOfLocalAtoms(ListOfLocalAtoms[FragmentCounter], reference->getAtomCount(), FreeList)) {
    DoLog(1) && (Log() << Verbose(1) << "Filling of ListOfLocalAtoms failed." << endl);
    return false;
  }

  // allocate fragment list
  if (FragmentList == NULL) {
    KeySetCounter = Count();
    FragmentList = new Graph*[KeySetCounter];
    for (int i=0;i<KeySetCounter;i++)
      FragmentList[i] = NULL;
    KeySetCounter = 0;
  }

  if ((KeySetList != NULL) && (KeySetList->size() != 0)) { // if there are some scanned keysets at all
    // assign scanned keysets
    if (FragmentList[FragmentCounter] == NULL)
      FragmentList[FragmentCounter] = new Graph;
    KeySet *TempSet = new KeySet;
    for (Graph::iterator runner = KeySetList->begin(); runner != KeySetList->end(); runner++) { // key sets contain global numbers!
      if (ListOfLocalAtoms[FragmentCounter][reference->FindAtom(*((*runner).first.begin()))->nr] != NULL) {// as we may assume that that bond structure is unchanged, we only test the first key in each set
        // translate keyset to local numbers
        for (KeySet::iterator sprinter = (*runner).first.begin(); sprinter != (*runner).first.end(); sprinter++)
          TempSet->insert(ListOfLocalAtoms[FragmentCounter][reference->FindAtom(*sprinter)->nr]->nr);
        // insert into FragmentList
        FragmentList[FragmentCounter]->insert(GraphPair(*TempSet, pair<int, double> (KeySetCounter++, (*runner).second.second)));
      }
      TempSet->clear();
    }
    delete (TempSet);
    if (KeySetCounter == 0) {// if there are no keysets, delete the list
      DoLog(1) && (Log() << Verbose(1) << "KeySetCounter is zero, deleting FragmentList." << endl);
      delete (FragmentList[FragmentCounter]);
    } else
      DoLog(1) && (Log() << Verbose(1) << KeySetCounter << " keysets were assigned to subgraph " << FragmentCounter << "." << endl);
    FragmentCounter++;
    if (next != NULL)
      next->AssignKeySetsToFragment(reference, KeySetList, ListOfLocalAtoms, FragmentList, FragmentCounter, FreeList);
    FragmentCounter--;
  } else
    DoLog(1) && (Log() << Verbose(1) << "KeySetList is NULL or empty." << endl);

  if ((FreeList) && (ListOfLocalAtoms != NULL)) {
    // free the index lookup list
    delete[](ListOfLocalAtoms[FragmentCounter]);
  }
  DoLog(1) && (Log() << Verbose(1) << "End of AssignKeySetsToFragment." << endl);
  return status;
};

/** Translate list into global numbers (i.e. ones that are valid in "this" molecule, not in MolecularWalker->Leaf)
 * \param *out output stream for debugging
 * \param **FragmentList Graph with local numbers per fragment
 * \param &FragmentCounter counts the fragments as we move along the list
 * \param &TotalNumberOfKeySets global key set counter
 * \param &TotalGraph Graph to be filled with global numbers
 */
void MoleculeLeafClass::TranslateIndicesToGlobalIDs(Graph **FragmentList, int &FragmentCounter, int &TotalNumberOfKeySets, Graph &TotalGraph)
{
  DoLog(1) && (Log() << Verbose(1) << "Begin of TranslateIndicesToGlobalIDs." << endl);
  KeySet *TempSet = new KeySet;
  if (FragmentList[FragmentCounter] != NULL) {
    for (Graph::iterator runner = FragmentList[FragmentCounter]->begin(); runner != FragmentList[FragmentCounter]->end(); runner++) {
      for (KeySet::iterator sprinter = (*runner).first.begin(); sprinter != (*runner).first.end(); sprinter++)
        TempSet->insert((Leaf->FindAtom(*sprinter))->GetTrueFather()->nr);
      TotalGraph.insert(GraphPair(*TempSet, pair<int, double> (TotalNumberOfKeySets++, (*runner).second.second)));
      TempSet->clear();
    }
    delete (TempSet);
  } else {
    DoLog(1) && (Log() << Verbose(1) << "FragmentList is NULL." << endl);
  }
  if (next != NULL)
    next->TranslateIndicesToGlobalIDs(FragmentList, ++FragmentCounter, TotalNumberOfKeySets, TotalGraph);
  FragmentCounter--;
  DoLog(1) && (Log() << Verbose(1) << "End of TranslateIndicesToGlobalIDs." << endl);
};

/** Simply counts the number of items in the list, from given MoleculeLeafClass.
 * \return number of items
 */
int MoleculeLeafClass::Count() const
{
  if (next != NULL)
    return next->Count() + 1;
  else
    return 1;
};

