/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2015 Frederik Heber. All rights reserved.
 *
 *
 *   This file is part of MoleCuilder.
 *
 *    MoleCuilder is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    MoleCuilder is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with MoleCuilder.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * QtMoleculeItem.cpp
 *
 *  Created on: Jan 17, 2015
 *      Author: heber
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "QtMoleculeItem.hpp"

#include <QModelIndex>

#include "UIElements/Views/Qt4/MoleculeList/QtMoleculeList.hpp"

#include "CodePatterns/MemDebug.hpp"

#include "CodePatterns/Observer/Notification.hpp"

#include "molecule.hpp"

// some attributes need to be easier to find for molecules
// these attributes are skipped so far
const int QtMoleculeItem::COLUMNCOUNT = COLUMNTYPES_MAX;
const char *QtMoleculeItem::COLUMNNAMES[QtMoleculeItem::COLUMNCOUNT]={"Name","Visibility", "Atoms","Formula","Occurrence"/*,"Size"*/};


QList<QStandardItem *> createMoleculeItemRow(const molecule *_mol)
{
  QList<QStandardItem *> molItems;
  QStandardItem *mainitem = new QtMoleculeItem(_mol);
  mainitem->setText(QString(_mol->getName().c_str()));
  mainitem->setFlags(mainitem->flags() | Qt::ItemIsSelectable);
  molItems << mainitem;

  QStandardItem *visitem = new QStandardItem();
  visitem->setCheckState(Qt::Unchecked);
  visitem->setFlags(visitem->flags() | Qt::ItemIsUserCheckable);
  molItems << visitem;

  molItems << new QStandardItem(QString::number(_mol->getAtomCount()));
  molItems << new QStandardItem(QString(_mol->getFormula().toString().c_str()));
  molItems << new QStandardItem(QString(""));
  return molItems;
}

QtMoleculeItem::QtMoleculeItem(const molecule *_mol) :
  Observer("QtMoleculeItem"),
  mol(_mol),
  IsSignedOn(false)
{
  mol->signOn(this, molecule::AtomInserted);
  mol->signOn(this, molecule::AtomRemoved);
  mol->signOn(this, molecule::MoleculeNameChanged);
  IsSignedOn = true;
}

QtMoleculeItem::~QtMoleculeItem()
{
  if (IsSignedOn) {
    mol->signOff(this, molecule::AtomInserted);
    mol->signOff(this, molecule::AtomRemoved);
    mol->signOff(this, molecule::MoleculeNameChanged);
  }
}


void QtMoleculeItem::update(Observable *publisher)
{}

void QtMoleculeItem::recieveNotification(Observable *publisher, Notification_ptr notification)
{
  if (dynamic_cast<molecule *>(publisher) != NULL) {
    switch (notification->getChannelNo()) {
      case molecule::AtomInserted:
      case molecule::AtomRemoved:
      {
        // change atomcount
        QStandardItem *count_item = parent()->child(index().row(), ATOMCOUNT);
        count_item->setText(QString::number(mol->getAtomCount()));

        // change formula
        const std::string molecule_formula = mol->getFormula().toString();
        QStandardItem *formula_item = parent()->child(index().row(), FORMULA);
        formula_item->setText(QString(molecule_formula.c_str()));

        // re-position in model's tree
//        static_cast<QtMoleculeList*>(model())->readdItem(this, molecule_formula);
        break;
      }
      case molecule::MoleculeNameChanged:
      {
        // change name
        QStandardItem *name_item = parent()->child(index().row(), NAME);
        name_item->setText(QString(mol->getName().c_str()));
        break;
      }
      default:
        ASSERT(0, "QtMoleculeItem::recieveNotification() - cannot get here, not subscribed to channel "
            +toString(notification->getChannelNo()));
        break;
    }
  }
}

void QtMoleculeItem::subjectKilled(Observable *publisher)
{
  IsSignedOn = false;
//  // we have to signal removal
//  emit removeMolecule(this);
}
