/*
 * Fragment.hpp
 *
 *  Created on: Aug 8, 2012
 *      Author: heber
 */

#ifndef FRAGMENT_HPP_
#define FRAGMENT_HPP_


// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <boost/serialization/export.hpp>
#include <boost/serialization/vector.hpp>
#include <boost/serialization/utility.hpp>

#include <iosfwd>
#include <vector>

class FragmentTest;

class Fragment {
  //!> grant ostream operator access
  friend std::ostream & operator<<(std::ostream &ost, const Fragment &f);
  //!> grant unit test access
  friend class FragmentTest;
public:
  typedef std::vector<double> position_t;
  typedef std::vector< position_t > positions_t;
  typedef double charge_t;
  typedef std::vector< charge_t > charges_t;
  typedef std::pair< position_t, charge_t> nucleus_t;
  typedef std::vector< nucleus_t > nuclei_t;

  /** Default constructor of class Fragment.
   *
   */
  Fragment();

  /** Default constructor of class Fragment.
   *
   */
  Fragment(const nuclei_t &_nuclei) :
    nuclei(_nuclei)
  {}

  /** Constructor of class Fragment.
   *
   * @param _positions given positions
   * @param _charges given charges
   */
  Fragment(const positions_t &_positions, const charges_t &_charges);

  /** Adding another fragment onto this one.
   *
   * \note The operation is area-conserving, i.e. the new area is the sum of
   * both areas.
   *
   * @param other other fragment
   * @return ref to this instance
   */
  Fragment& operator+=(const Fragment &other);

  /** Assignment operator.
   *
   * @param other other fragment to make ourselves equal to
   * @return ref to this instance
   */
  Fragment& operator=(const Fragment &other);

  /** Subtracting another fragment from this one.
   *
   * @param other other fragment
   * @return ref to this instance
   */
  Fragment& operator-=(const Fragment &other);

  /** Getter for all stored positions.
   *
   * @return vector of positions
   */
  positions_t getPositions() const;

  /** Getter for all stored charges.
   *
   * @return vector of charges
   */
  charges_t getCharges() const;

  /** Equality operator.
   *
   * @param other other instance to check against
   * @return true - both are equal, false - some nucleus_t differ
   */
  bool operator==(const Fragment& other) const;

  bool operator!=(const Fragment& other) const
  {
    return (!(*this == other));
  }

  /** Creates type nucleus_t from given \a position and \a charge.
   *
   * @param position position of nucleus to create
   * @param charge charge of nucleus to create
   * @return nucleus with given \a position and \a charge
   */
  static nucleus_t createNucleus(const position_t &position, const double charge);

  /** Helper function to check whether two positions are equal.
   *
   * @param a first position
   * @param b second position
   * @return a equals b within numerical precision
   */
  static bool isPositionEqual(const position_t &a, const position_t &b);

private:
  /** Helper function that checks whether this nuclei \b position is present.
   *
   * This operation is \f${\cal O}(n)\f$
   *
   * @param n nuclei to check
   * @return true - is contained, false - is not contained
   */
  bool containsNuclei(const nucleus_t &n) const;

  /** Seeks through all nuclei and removes one with matching \b position if found.
   *
   * @param n nuclei to remove
   */
  void removeNuclei(const nucleus_t &n);

private:
  nuclei_t nuclei;

private:
  friend class boost::serialization::access;
  // serialization
  template <typename Archive>
  void serialize(Archive& ar, const unsigned int version)
  {
    ar & nuclei;
  }
};

// we need to give this class a unique key for serialization
BOOST_CLASS_EXPORT_KEY(Fragment)

/** Equality operator for two nuclei.
 *
 * @param a first nuclei
 * @param b second nuclei
 * @return true - both have same position and charge, false - either charge or position is different
 */
bool operator==(const Fragment::nucleus_t &a, const Fragment::nucleus_t &b);

std::ostream & operator<<(std::ostream &ost, const Fragment::nucleus_t &n);

std::ostream & operator<<(std::ostream &ost, const Fragment &f);

template<typename T> T ZeroInstance();
template<> Fragment ZeroInstance<Fragment>();

#endif /* FRAGMENT_HPP_ */
