/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2014 Frederik Heber. All rights reserved.
 * 
 *
 *   This file is part of MoleCuilder.
 *
 *    MoleCuilder is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    MoleCuilder is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with MoleCuilder.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * MirrorAction.cpp
 *
 *  Created on: Sep 10, 2014
 *      Author: heber
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "CodePatterns/MemDebug.hpp"

#include "CodePatterns/Log.hpp"
#include "Atom/atom.hpp"
#include "LinearAlgebra/Plane.hpp"
#include "LinearAlgebra/Vector.hpp"
#include "CodePatterns/Verbose.hpp"
#include "World.hpp"

#include <iostream>
#include <fstream>
#include <string>

#include "Actions/AtomAction/MirrorAction.hpp"

using namespace MoleCuilder;

// and construct the stuff
#include "MirrorAction.def"
#include "Action_impl_pre.hpp"
/** =========== define the function ====================== */
ActionState::ptr AtomMirrorAction::performCall() {
  Box &domain = World::getInstance().getDomain();
  std::vector<atom *> selectedAtoms = World::getInstance().getSelectedAtoms();

  Plane *MirrorPlane = NULL;
  std::vector<atom *>::iterator iter = selectedAtoms.begin();
  try {
    MirrorPlane = new Plane(params.normal.get(), params.offset.get());
    for (; iter != selectedAtoms.end(); ++iter) {
      (*iter)->setPosition(MirrorPlane->mirrorVector((*iter)->getPosition()));
      if (params.periodic.get())
        (*iter)->setPosition(domain.enforceBoundaryConditions((*iter)->getPosition()));
    }
  } catch( LinearAlgebraException &e) {
    // flip back
    for (std::vector<atom *>::iterator undoiter = selectedAtoms.begin();
        undoiter != iter; ++undoiter)
      (*iter)->setPosition(MirrorPlane->mirrorVector((*iter)->getPosition()));
    delete MirrorPlane;
    return Action::failure;
  }
  delete MirrorPlane;

  return ActionState::ptr(new AtomMirrorState(selectedAtoms, params));
}

ActionState::ptr AtomMirrorAction::performUndo(ActionState::ptr _state) {
  AtomMirrorState *state = assert_cast<AtomMirrorState*>(_state.get());
  Box &domain = World::getInstance().getDomain();

  Plane MirrorPlane(state->params.normal.get(), state->params.offset.get());
  for (std::vector<atom *>::iterator iter = state->selectedAtoms.begin(); iter != state->selectedAtoms.end(); ++iter) {
    (*iter)->setPosition(MirrorPlane.mirrorVector((*iter)->getPosition()));
    if (state->params.periodic.get())
      (*iter)->setPosition(domain.enforceBoundaryConditions((*iter)->getPosition()));
  }

  return ActionState::ptr(_state);
}

ActionState::ptr AtomMirrorAction::performRedo(ActionState::ptr _state){
  AtomMirrorState *state = assert_cast<AtomMirrorState*>(_state.get());
  Box &domain = World::getInstance().getDomain();

  Plane MirrorPlane(state->params.normal.get(), state->params.offset.get());
  for (std::vector<atom *>::iterator iter = state->selectedAtoms.begin(); iter != state->selectedAtoms.end(); ++iter) {
    (*iter)->setPosition(MirrorPlane.mirrorVector((*iter)->getPosition()));
    if (state->params.periodic.get())
      (*iter)->setPosition(domain.enforceBoundaryConditions((*iter)->getPosition()));
  }

  return ActionState::ptr(_state);
}

bool AtomMirrorAction::canUndo() {
  return true;
}

bool AtomMirrorAction::shouldUndo() {
  return true;
}
/** =========== end of function ====================== */
