/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2015 Frederik Heber. All rights reserved.
 *
 *
 *   This file is part of MoleCuilder.
 *
 *    MoleCuilder is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    MoleCuilder is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with MoleCuilder.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * QtMoleculeListView.cpp
 *
 *  Created on: Jan 17, 2015
 *      Author: heber
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "QtMoleculeListView.hpp"

#include <QtCore/QMetaType>

#include "UIElements/Views/Qt4/MoleculeList/QtMoleculeList.hpp"
#include "UIElements/Views/Qt4/MoleculeList/QtMoleculeItem.hpp"

#include "CodePatterns/MemDebug.hpp"

#include "Actions/SelectionAction/Molecules/MoleculeByIdAction.hpp"
#include "Actions/SelectionAction/Molecules/NotMoleculeByIdAction.hpp"
#include "molecule.hpp"
#include "World.hpp"

QtMoleculeListView::QtMoleculeListView(QWidget * _parent) :
  QTreeView(_parent),
  selecting(false)
{
  setSelectionMode(QAbstractItemView::MultiSelection);

  qRegisterMetaType<QItemSelection>("QItemSelection");
}

QtMoleculeListView::~QtMoleculeListView()
{
}

void QtMoleculeListView::setModel(QtMoleculeList *_moleculelist)
{
  QTreeView::setModel(_moleculelist);
  // clicking a molecule means calling SelectionAction
  connect(
      selectionModel(), SIGNAL(selectionChanged(const QItemSelection &, const QItemSelection &)),
      this, SLOT(rowsSelected(const QItemSelection &, const QItemSelection &)), Qt::DirectConnection);
  connect(_moleculelist, SIGNAL(MayStartSelections()),
      this, SLOT(activateSelections()), Qt::DirectConnection);
  connect(_moleculelist, SIGNAL(MayNotStartSelections()),
      this, SLOT(deactivateSelections()), Qt::DirectConnection);

  connect(&_moleculelist->observer, SIGNAL(SelectionChanged(QtObservedMolecule::ptr)),
      this, SLOT(selectionChanged(QtObservedMolecule::ptr)));
}

QModelIndex QtMoleculeListView::setIndexToLastColumn(const QModelIndex &_index) const
{
  QModelIndex return_index;
  QModelIndex parent_index = _index.parent();
  ASSERT (parent_index.isValid(),
      "QtMoleculeListView::setIndexToLastColumn() - _index has no valid parent.");
    return_index = parent_index.child(_index.row(), QtMoleculeItem::OCCURRENCE);
//    return_index =
//        model()->invisibleRootItem()->child(
//            _index.row(),
//            QtMoleculeItem::OCCURRENCE)->index();
  return return_index;
}

void QtMoleculeListView::rowsSelected(
    const QItemSelection& selected, const QItemSelection& deselected)
{
  if (selecting)
    return;

  selecting = true;

  // Select all molecules which belong to newly selected rows.
  QtMoleculeList *moleculelist = dynamic_cast<QtMoleculeList *>(model());
  QModelIndex index;
  {
    QModelIndexList items = selected.indexes();
    molids_t ids;
    ids.reserve(items.size());
    foreach (index, items)
      if ((index.column() == 0) && (selectionModel()->isSelected(index))) {
        const moleculeId_t mol_id = moleculelist->observer.getIdtoIndex(
            moleculelist->IndexToMoleculeId(index));
        const molecule * const mol = const_cast<const World &>(World::getInstance()).
            getMolecule(MoleculeById(mol_id));
        // check for invalid molecule
        if (mol_id < 0)
          continue;
        // means we are looking at deselection because of removal (in World)
        if (mol == NULL)
          continue;
        if (!World::getInstance().isSelected(mol))
            ids.push_back(mol_id);
        //std::cout << "select molecule" << std::endl;
      }
    if (!ids.empty())
      MoleCuilder::SelectionMoleculeById(ids);
  }

  // Unselect all molecules which belong to newly unselected rows.
  {
    QModelIndexList items = deselected.indexes();
    molids_t ids;
    ids.reserve(items.size());
    foreach (index, items)
      if ((index.column() == 0) && (!selectionModel()->isSelected(index))) {
        const moleculeId_t mol_id = moleculelist->observer.getIdtoIndex(
            moleculelist->IndexToMoleculeId(index));
        const molecule * const mol = const_cast<const World &>(World::getInstance()).
            getMolecule(MoleculeById(mol_id));
        // check for invalid molecule
        if (mol_id < 0)
          continue;
        // means we are looking at deselection because of removal (in World)
        if (mol == NULL)
          continue;
        if (World::getInstance().isSelected(mol))
            ids.push_back(mol_id);
        //std::cout << "unselect molecule" << std::endl;
      }
    if (!ids.empty())
      MoleCuilder::SelectionNotMoleculeById(ids);
  }

  selecting = false;
}

void QtMoleculeListView::activateSelections()
{
  selecting = false;
}

void QtMoleculeListView::deactivateSelections()
{
  selecting = true;
}

void QtMoleculeListView::selectionChanged(const QtObservedMolecule::ptr _molecule)
{
  if (_molecule->getMolSelected())
    MoleculeSelected(_molecule->getIndex());
  else
    MoleculeUnselected(_molecule->getIndex());
}

void QtMoleculeListView::MoleculeSelected(ObservedValue_Index_t _id)
{
  if (selecting)
    return;

  selecting = true;

  const QtMoleculeList *moleculelist = dynamic_cast<const QtMoleculeList *>(model());
  if (moleculelist->isMoleculeItemPresent(_id)) {
    QModelIndex index = moleculelist->MoleculeIdToIndex(_id);
//    ASSERT( !selectionModel()->isSelected(index),
//        "QtMoleculeListView::MoleculeSelected() - row to molecule "
//        +toString(mol->getMolIndex())+" is already selected.");

    // select the full row
    expand(index);
    selectionModel()->select(index, QItemSelectionModel::Select | QItemSelectionModel::Rows);
  }

  selecting = false;
}

void QtMoleculeListView::MoleculeUnselected(ObservedValue_Index_t _id)
{
  if (selecting)
    return;

  selecting = true;

  const QtMoleculeList *moleculelist = dynamic_cast<const QtMoleculeList *>(model());
  if (moleculelist->isMoleculeItemPresent(_id)) {
    QModelIndex index = moleculelist->MoleculeIdToIndex(_id);
//    ASSERT( selectionModel()->isSelected(index),
//        "QtMoleculeListView::MoleculeSelected() - row to molecule "
//        +toString(mol->getMolIndex())+" is already unselected.");

    // unselect the full row
    expand(index);
    selectionModel()->select(index, QItemSelectionModel::Deselect | QItemSelectionModel::Rows);
  }

  selecting = false;
}
