/*
 * FragmentationResultContainer.hpp
 *
 *  Created on: Mar 8, 2013
 *      Author: heber
 */

#ifndef FRAGMENTATIONRESULTCONTAINER_HPP_
#define FRAGMENTATIONRESULTCONTAINER_HPP_

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <boost/serialization/access.hpp>
#include "CodePatterns/Singleton.hpp"

#include <boost/filesystem/path.hpp>
#include <map>

#include "CodePatterns/Observer/Observable.hpp"

#ifdef HAVE_JOBMARKET
#include "JobMarket/types.hpp"
#else
typedef size_t JobId_t;
#endif

#include "Fragmentation/KeySetsContainer.hpp"
#include "Fragmentation/Summation/Containers/MPQCData.hpp"
#ifdef HAVE_VMG
#include "Fragmentation/Summation/Containers/VMGData.hpp"
#endif

/** This class stores results from lengthy fragmentation calculations, e.g.
 * coming out of FragmentationAutomationAction.
 *
 * The idea is that we can play around with results without having to recalculate
 * results in between. Hence, we place results into this container which we may
 * full serialize.
 */
class FragmentationResultContainer :
    public Singleton<FragmentationResultContainer>,
    public Observable
{
  //!> Singleton patterns needs access to private cstor/dtor.
  friend class Singleton<FragmentationResultContainer>;
private:
  FragmentationResultContainer() :
    Observable("FragmentationResultContainer"),
    ResultsType(BothRanges)
  {}
  ~FragmentationResultContainer() {}

public:
  //!> typedef for short range data container
  typedef std::map<JobId_t, MPQCData> shortrangedata_t;
#ifdef HAVE_VMG
  //!> typedef for long range data container
  typedef std::map<JobId_t, VMGData> longrangedata_t;
#endif

#ifdef HAVE_VMG
  /** Adds a a set of both short and long range results.
   *
   * Adds all the containers to the ones present in this instance.
   *
   * \param _keysets
   * \param _forcekeysetskeysets
   * \param _shortrangedata
   * \param _longrangedata
   */
  void addFullResults(
      const KeySetsContainer &_keysets,
      const KeySetsContainer &_forcekeysets,
      const shortrangedata_t &_shortrangedata,
      const longrangedata_t &_longrangedata
      );
#endif

  /** Adds a a set of short range results only.
   *
   * Adds all the containers to the ones present in this instance.
   *
   * \param _keysets
   * \param _forcekeysets
   * \param _shortrangedata
   */
  void addShortRangeResults(
      const KeySetsContainer &_keysets,
      const KeySetsContainer &_forcekeysets,
      const shortrangedata_t &_shortrangedata
      );

  /** Adds given cycles to internal keyset list.
   *
   * \param _cycles keysets of cycles to add
   */
  void addCycles(const KeySetsContainer &_cycles)
  { cycles.insert(_cycles); }

  /** Clears all internal containers.
   *
   * \note Also resets ResultsType.
   */
  void clear()
  {
    OBSERVE;
    keysets.clear();
    forcekeysets.clear();
    shortrangedata.clear();
#ifdef HAVE_VMG
    longrangedata.clear();
#endif
    ResultsType = BothRanges;
  }

  const KeySetsContainer &getKeySets() const { return keysets; }
  const KeySetsContainer &getCycles() const { return cycles; }
  const KeySetsContainer &getForceKeySets() const  { return forcekeysets; }
  const shortrangedata_t& getShortRangeResults() const  { return shortrangedata; }
#ifdef HAVE_VMG
  const longrangedata_t& getLongRangeResults() const;
#endif
  bool areFullRangeResultsPresent() const { return (ResultsType == BothRanges); }

private:
  //!> indicates whether we contain short range only or both results
  enum ResultsType_t {
    ShortRangeOnly,
    BothRanges,
  } ResultsType;

  //!> container for all KeySet's without hydrogens to the jobs
  KeySetsContainer keysets;
  //!> container for all KeySet's with all except saturation hydrogen to the jobs
  KeySetsContainer forcekeysets;
  //!> container for all short-range results
  std::map<JobId_t, MPQCData> shortrangedata;
  //! container of all cycle keysets
  KeySetsContainer cycles;

#ifdef HAVE_VMG
  //!> container for all long-range results
  std::map<JobId_t, VMGData> longrangedata;
#endif

  friend class boost::serialization::access;
  // serialization
  template <typename Archive>
  void serialize(Archive& ar, const unsigned int version)
  {
    ar & ResultsType;
    ar & keysets;
    ar & forcekeysets;
    ar & shortrangedata;
    if (version > 0)
      ar & cycles;
#ifdef HAVE_VMG
    ar & longrangedata;
#endif
  }
};

// version for serialized information associated to FragmentationResultContainer
BOOST_CLASS_VERSION(FragmentationResultContainer, 1)

#endif /* FRAGMENTATIONRESULTCONTAINER_HPP_ */
