/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2010 University of Bonn. All rights reserved.
 * Please see the LICENSE file or "Copyright notice" in builder.cpp for details.
 */

/*
 * NotAllAtomsInsideCuboidAction.cpp
 *
 *  Created on: Aug 9, 2010
 *      Author: heber
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "CodePatterns/MemDebug.hpp"

#include "Descriptors/AtomSelectionDescriptor.hpp"
#include "Descriptors/AtomShapeDescriptor.hpp"
#include "atom.hpp"
#include "CodePatterns/Log.hpp"
#include "CodePatterns/Verbose.hpp"
#include "LinearAlgebra/RealSpaceMatrix.hpp"
#include "LinearAlgebra/Vector.hpp"
#include "Shapes/BaseShapes.hpp"
#include "Shapes/Shape.hpp"
#include "Shapes/ShapeOps.hpp"
#include "World.hpp"

#include <boost/foreach.hpp>
#include <iostream>
#include <string>

using namespace std;

#include "NotAllAtomsInsideCuboidAction.hpp"

// and construct the stuff
#include "NotAllAtomsInsideCuboidAction.def"
#include "Action_impl_pre.hpp"
/** =========== define the function ====================== */
Action::state_ptr SelectionNotAllAtomsInsideCuboidAction::performCall() {
  RealSpaceMatrix RotationMatrix;

  // obtain information
  getParametersfromValueStorage();
  RotationMatrix.setRotation(params.Xangle, params.Yangle, params.Zangle);

  DoLog(1) && (Log() << Verbose(1) << "Unselecting all atoms inside a rotated " << RotationMatrix << " cuboid at " << params.position << " and extension of " << params.extension << "." << endl);
  Shape s = translate(transform(stretch(Cuboid(),params.extension),RotationMatrix),params.position);
  std::vector<atom *> unselectedAtoms = World::getInstance().getAllAtoms((!AtomsBySelection()) && AtomByShape(s));
  World::getInstance().unselectAllAtoms(AtomByShape(s));
  return Action::state_ptr(new SelectionNotAllAtomsInsideCuboidState(unselectedAtoms, s, params));
}

Action::state_ptr SelectionNotAllAtomsInsideCuboidAction::performUndo(Action::state_ptr _state) {
  SelectionNotAllAtomsInsideCuboidState *state = assert_cast<SelectionNotAllAtomsInsideCuboidState*>(_state.get());

  World::getInstance().selectAllAtoms(AtomByShape(state->s));
  BOOST_FOREACH(atom *_atom, state->unselectedAtoms)
    World::getInstance().unselectAtom(_atom);

  return Action::state_ptr(_state);
}

Action::state_ptr SelectionNotAllAtomsInsideCuboidAction::performRedo(Action::state_ptr _state){
  SelectionNotAllAtomsInsideCuboidState *state = assert_cast<SelectionNotAllAtomsInsideCuboidState*>(_state.get());
  RealSpaceMatrix RotationMatrix;

  World::getInstance().unselectAllAtoms(AtomByShape(state->s));

  return Action::state_ptr(_state);
}

bool SelectionNotAllAtomsInsideCuboidAction::canUndo() {
  return true;
}

bool SelectionNotAllAtomsInsideCuboidAction::shouldUndo() {
  return true;
}
/** =========== end of function ====================== */
