/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2010 University of Bonn. All rights reserved.
 * Please see the LICENSE file or "Copyright notice" in builder.cpp for details.
 */

/*
 * PrincipalAxisSystemAction.cpp
 *
 *  Created on: May 12, 2010
 *      Author: heber
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "CodePatterns/MemDebug.hpp"

#include "CodePatterns/Log.hpp"
#include "CodePatterns/Verbose.hpp"
#include "LinearAlgebra/RealSpaceMatrix.hpp"
#include "LinearAlgebra/Vector.hpp"
#include "element.hpp"
#include "molecule.hpp"

#include <iostream>
#include <string>

using namespace std;

#include "Actions/AnalysisAction/PrincipalAxisSystemAction.hpp"

// and construct the stuff
#include "PrincipalAxisSystemAction.def"
#include "Action_impl_pre.hpp"

/** =========== define the function ====================== */
Action::state_ptr AnalysisPrincipalAxisSystemAction::performCall() {
  RealSpaceMatrix InertiaTensor;

  DoLog(0) && (Log() << Verbose(0) << "Evaluating prinicipal axis." << endl);
  for (World::MoleculeSelectionIterator iter = World::getInstance().beginMoleculeSelection(); iter != World::getInstance().endMoleculeSelection(); ++iter) {
    molecule *mol = iter->second;
    Vector *CenterOfGravity = mol->DetermineCenterOfGravity();

    // reset inertia tensor
    InertiaTensor.setZero();

    // sum up inertia tensor
    for (molecule::const_iterator iter = mol->begin(); iter != mol->end(); ++iter) {
      Vector x = (*iter)->getPosition();
      x -= *CenterOfGravity;
      const double mass = (*iter)->getType()->getMass();
      InertiaTensor.at(0,0) += mass*(x[1]*x[1] + x[2]*x[2]);
      InertiaTensor.at(0,1) += mass*(-x[0]*x[1]);
      InertiaTensor.at(0,2) += mass*(-x[0]*x[2]);
      InertiaTensor.at(1,0) += mass*(-x[1]*x[0]);
      InertiaTensor.at(1,1) += mass*(x[0]*x[0] + x[2]*x[2]);
      InertiaTensor.at(1,2) += mass*(-x[1]*x[2]);
      InertiaTensor.at(2,0) += mass*(-x[2]*x[0]);
      InertiaTensor.at(2,1) += mass*(-x[2]*x[1]);
      InertiaTensor.at(2,2) += mass*(x[0]*x[0] + x[1]*x[1]);
    }
    // print InertiaTensor for debugging
    DoLog(0) && (Log() << Verbose(0) << "The inertia tensor is:" << InertiaTensor << endl);
  }
  return Action::success;
}

Action::state_ptr AnalysisPrincipalAxisSystemAction::performUndo(Action::state_ptr _state) {
  return Action::success;
}

Action::state_ptr AnalysisPrincipalAxisSystemAction::performRedo(Action::state_ptr _state){
  return Action::success;
}

bool AnalysisPrincipalAxisSystemAction::canUndo() {
  return true;
}

bool AnalysisPrincipalAxisSystemAction::shouldUndo() {
  return true;
}
/** =========== end of function ====================== */
