/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2012 University of Bonn. All rights reserved.
 * Copyright (C)  2013 Frederik Heber. All rights reserved.
 * 
 *
 *   This file is part of MoleCuilder.
 *
 *    MoleCuilder is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    MoleCuilder is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with MoleCuilder.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * PairPotential_HarmonicUnitTest.cpp
 *
 *  Created on: Oct 04, 2012
 *      Author: heber
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

using namespace std;

#include <cppunit/CompilerOutputter.h>
#include <cppunit/extensions/TestFactoryRegistry.h>
#include <cppunit/ui/text/TestRunner.h>

#include "PairPotential_HarmonicUnitTest.hpp"

#include <boost/assign.hpp>

#include "CodePatterns/Assert.hpp"

#include "FunctionApproximation/FunctionArgument.hpp"
#include "Potentials/Specifics/PairPotential_Harmonic.hpp"
#include "Potentials/helpers.hpp"

using namespace boost::assign;

#ifdef HAVE_TESTRUNNER
#include "UnitTestMain.hpp"
#endif /*HAVE_TESTRUNNER*/

/********************************************** Test classes **************************************/

// Registers the fixture into the 'registry'
CPPUNIT_TEST_SUITE_REGISTRATION( PairPotential_HarmonicTest );


void PairPotential_HarmonicTest::setUp()
{
  // failing asserts should be thrown
  ASSERT_DO(Assert::Throw);

  input += 0., 1., 2., 3., 4., 5., 6., 7., 8., 9.;
  output +=
      .5*1.,
      .5*0.,
      .5*1.,
      .5*4.,
      .5*9.,
      .5*16.,
      .5*25.,
      .5*36.,
      .5*49.,
      .5*64.;

  CPPUNIT_ASSERT_EQUAL( input.size(), output.size() );
}


void PairPotential_HarmonicTest::tearDown()
{
}

/** UnitTest for operator()
 */
void PairPotential_HarmonicTest::operatorTest()
{
  PairPotential_Harmonic::ParticleTypes_t types =
      boost::assign::list_of<PairPotential_Harmonic::ParticleType_t>
        (0)(1)
      ;
  PairPotential_Harmonic harmonic(types, .5,1.);
  for (size_t index = 0; index < input.size(); ++index) {
    argument_t arg(argument_t::indices_t(0,1), argument_t::types_t(0,1), input[index]);
    CPPUNIT_ASSERT(
        Helpers::isEqual(
            output[index],
            harmonic( FunctionModel::arguments_t(1,arg) )[0]
        )
    );
  }
}

/** UnitTest for derivative()
 */
void PairPotential_HarmonicTest::derivativeTest()
{
  PairPotential_Harmonic::ParticleTypes_t types =
      boost::assign::list_of<PairPotential_Harmonic::ParticleType_t>
        (0)(1)
      ;
  PairPotential_Harmonic harmonic(types, .5,1.);
  argument_t arg(argument_t::indices_t(0,1), argument_t::types_t(0,1), 1.);
  CPPUNIT_ASSERT(
      Helpers::isEqual(
          0.,
          harmonic.derivative(
              FunctionModel::arguments_t(1,arg)
          )[0]
      )
  );
}


/** UnitTest for parameter_derivative()
 */
void PairPotential_HarmonicTest::parameter_derivativeTest()
{
  PairPotential_Harmonic::ParticleTypes_t types =
      boost::assign::list_of<PairPotential_Harmonic::ParticleType_t>
        (0)(1)
      ;
  PairPotential_Harmonic harmonic(types, .5,1.);
  argument_t arg(argument_t::indices_t(0,1), argument_t::types_t(0,1), 1.);
  CPPUNIT_ASSERT(
      Helpers::isEqual(
          0.,
          harmonic.parameter_derivative(
              FunctionModel::arguments_t(1,arg),
              0
          )[0]
      )
  );
  CPPUNIT_ASSERT(
      Helpers::isEqual(
          0.,
          harmonic.parameter_derivative(
              FunctionModel::arguments_t(1,arg),
              1
          )[0]
      )
  );
}
