/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2010-2012 University of Bonn. All rights reserved.
 *
 *
 *   This file is part of MoleCuilder.
 *
 *    MoleCuilder is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    MoleCuilder is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with MoleCuilder.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * ShapeFactoryUnitTest.cpp
 *
 *  Created on: Sep 5, 2012
 *      Author: ankele
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <cppunit/CompilerOutputter.h>
#include <cppunit/extensions/TestFactoryRegistry.h>
#include <cppunit/ui/text/TestRunner.h>

#include <cmath>
#include <limits>

#ifdef HAVE_TESTRUNNER
#include "UnitTestMain.hpp"
#endif /*HAVE_TESTRUNNER*/

#include "CodePatterns/Assert.hpp"

#include "Helpers/defs.hpp"
#include "Shapes/BaseShapes_impl.hpp"
#include "Shapes/Shape_impl.hpp"
#include "Shapes/ShapeFactory.hpp"

#include "ShapeFactoryUnitTest.hpp"

// Registers the fixture into the 'registry'
CPPUNIT_TEST_SUITE_REGISTRATION( ShapeFactoryTest );

void ShapeFactoryTest::setUp()
{
  // failing asserts should be thrown
  ASSERT_DO(Assert::Throw);

  ShapeFactory::getInstance();
}

void ShapeFactoryTest::tearDown()
{
  ShapeFactory::purgeInstance();
}

void ShapeFactoryTest::typeTest()
{
  ShapeFactory::getInstance().setAngle(0., 0., 0.);
  ShapeFactory::getInstance().setStretch(Vector(1., 1., 1.));
  ShapeFactory::getInstance().setTranslation(Vector(0., 0., 0.));

  ShapeFactory::getInstance().setType(ShapeFactory::NOWHERE);
  CPPUNIT_ASSERT( dynamic_cast<Nowhere_impl*>(ShapeFactory::getInstance().produce().getImpl().get()) );

  ShapeFactory::getInstance().setType(ShapeFactory::EVERYWHERE);
  CPPUNIT_ASSERT( dynamic_cast<Everywhere_impl*>(ShapeFactory::getInstance().produce().getImpl().get()) );

  ShapeFactory::getInstance().setType(ShapeFactory::SPHERE);
  CPPUNIT_ASSERT( dynamic_cast<Sphere_impl*>(ShapeFactory::getInstance().produce().getImpl().get()) );

  ShapeFactory::getInstance().setType(ShapeFactory::CUBE);
  CPPUNIT_ASSERT( dynamic_cast<Cuboid_impl*>(ShapeFactory::getInstance().produce().getImpl().get()) );

  ShapeFactory::getInstance().setType(ShapeFactory::CYLINDER);
  CPPUNIT_ASSERT( dynamic_cast<Cylinder_impl*>(ShapeFactory::getInstance().produce().getImpl().get()) );
}

void ShapeFactoryTest::resizeTest()
{
  ShapeFactory::getInstance().setType(ShapeFactory::SPHERE);
  ShapeFactory::getInstance().setAngle(0., 0., 0.);
  ShapeFactory::getInstance().setStretch(Vector(2., 2., 2.));
  ShapeFactory::getInstance().setTranslation(Vector(0., 0., 0.));
  Shape s = ShapeFactory::getInstance().produce();

  CPPUNIT_ASSERT( s.isInside( Vector(1.5,0.,0.) ) );
  CPPUNIT_ASSERT( s.isInside( Vector(0.,1.5,0.) ) );
  CPPUNIT_ASSERT( s.isInside( Vector(0.,0.,1.5) ) );
  CPPUNIT_ASSERT( s.isOnSurface( Vector(2.,0.,0.) ) );
  CPPUNIT_ASSERT( s.isOnSurface( Vector(0.,2.,0.) ) );
  CPPUNIT_ASSERT( s.isOnSurface( Vector(0.,0.,2.) ) );
}

void ShapeFactoryTest::translateTest()
{
  ShapeFactory::getInstance().setType(ShapeFactory::SPHERE);
  ShapeFactory::getInstance().setAngle(0., 0., 0.);
  ShapeFactory::getInstance().setStretch(Vector(1., 1., 1.));
  ShapeFactory::getInstance().setTranslation(Vector(1., 0., 0.));
  Shape s = ShapeFactory::getInstance().produce();

  CPPUNIT_ASSERT( s.isInside( Vector(1.5,0.,0.) ) );
  CPPUNIT_ASSERT( s.isInside( Vector(1.,.5,0.) ) );
  CPPUNIT_ASSERT( s.isInside( Vector(1.,0.,.5) ) );
  CPPUNIT_ASSERT( s.isOnSurface( Vector(2.,0.,0.) ) );
  CPPUNIT_ASSERT( s.isOnSurface( Vector(1.,1.,0.) ) );
  CPPUNIT_ASSERT( s.isOnSurface( Vector(1.,0.,1.) ) );
  CPPUNIT_ASSERT( !s.isInside( Vector(-.5,0.,0.) ) );
  CPPUNIT_ASSERT( !s.isInside( Vector(0.,-.5,0.) ) );
  CPPUNIT_ASSERT( !s.isInside( Vector(0.,0.,-.5) ) );
}

void ShapeFactoryTest::stretchTest()
{
  ShapeFactory::getInstance().setType(ShapeFactory::SPHERE);
  ShapeFactory::getInstance().setAngle(0., 0., 0.);
  ShapeFactory::getInstance().setStretch(Vector(2., 1., 1.));
  ShapeFactory::getInstance().setTranslation(Vector(0., 0., 0.));
  Shape s = ShapeFactory::getInstance().produce();

  CPPUNIT_ASSERT( s.isInside( Vector(1.5,0.,0.) ) );
  CPPUNIT_ASSERT( !s.isInside( Vector(0.,1.5,0.) ) );
  CPPUNIT_ASSERT( !s.isInside( Vector(0.,0.,1.5) ) );
  CPPUNIT_ASSERT( s.isOnSurface( Vector(2.,0.,0.) ) );
  CPPUNIT_ASSERT( s.isOnSurface( Vector(0.,1.,0.) ) );
  CPPUNIT_ASSERT( s.isOnSurface( Vector(0.,1.,0.) ) );
}

void ShapeFactoryTest::transformTest()
{
  ShapeFactory::getInstance().setType(ShapeFactory::SPHERE);
  ShapeFactory::getInstance().setAngle(45., 0., 0.);
  ShapeFactory::getInstance().setStretch(Vector(1., 1., 1.));
  ShapeFactory::getInstance().setTranslation(Vector(0., 0., 0.));
  Shape s = ShapeFactory::getInstance().produce();

  CPPUNIT_ASSERT( s.isInside( Vector(.5,0.,0.) ) );
  CPPUNIT_ASSERT( s.isInside( Vector(0.,.5,0.) ) );
  CPPUNIT_ASSERT( s.isInside( Vector(0.,0.,.5) ) );
  CPPUNIT_ASSERT( s.isOnSurface( Vector(1.,0.,0.) ) );
  CPPUNIT_ASSERT( s.isOnSurface( Vector(0.,1.,0.) ) );
  CPPUNIT_ASSERT( s.isOnSurface( Vector(0.,0.,1.) ) );
}

void ShapeFactoryTest::getterSetterTest()
{
  ShapeFactory::getInstance().setType(ShapeFactory::SPHERE);
  CPPUNIT_ASSERT_EQUAL( ShapeFactory::getInstance().getType(), ShapeFactory::SPHERE );
  ShapeFactory::getInstance().setType(ShapeFactory::CUBE);
  CPPUNIT_ASSERT_EQUAL( ShapeFactory::getInstance().getType(), ShapeFactory::CUBE );
#ifndef NDEBUG
  CPPUNIT_ASSERT_THROW( ShapeFactory::getInstance().setType((ShapeFactory::ShapeType)-1), Assert::AssertionFailure );
#endif


  ShapeFactory::getInstance().setAngle(45., 60., 90.);
  CPPUNIT_ASSERT_EQUAL( ShapeFactory::getInstance().getAngle()[0], 45. );
  CPPUNIT_ASSERT_EQUAL( ShapeFactory::getInstance().getAngle()[1], 60. );
  CPPUNIT_ASSERT_EQUAL( ShapeFactory::getInstance().getAngle()[2], 90. );


  ShapeFactory::getInstance().setStretch(Vector(2., 3., 4.));
  CPPUNIT_ASSERT_EQUAL( ShapeFactory::getInstance().getStretch(), Vector(2., 3., 4.) );
#ifndef NDEBUG
  CPPUNIT_ASSERT_THROW( ShapeFactory::getInstance().setStretch(Vector(0., 0., 0.)), Assert::AssertionFailure );
#endif
#ifndef NDEBUG
  CPPUNIT_ASSERT_THROW( ShapeFactory::getInstance().setStretch(Vector(1., 1., 0.)), Assert::AssertionFailure );
#endif

  ShapeFactory::getInstance().setTranslation(Vector(1., 2., 3.));
  CPPUNIT_ASSERT_EQUAL( ShapeFactory::getInstance().getTranslation(), Vector(1., 2., 3.) );
}

