/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2010 University of Bonn. All rights reserved.
 * Please see the LICENSE file or "Copyright notice" in builder.cpp for details.
 */

/*
 * BoxUnittest.cpp
 *
 *  Created on: Jul 30, 2010
 *      Author: crueger
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "BoxUnittest.hpp"

#include <cppunit/CompilerOutputter.h>
#include <cppunit/extensions/TestFactoryRegistry.h>
#include <cppunit/ui/text/TestRunner.h>

#ifdef HAVE_TESTRUNNER
#include "UnitTestMain.hpp"
#endif /*HAVE_TESTRUNNER*/

#include "LinearAlgebra/Vector.hpp"
#include "LinearAlgebra/Matrix.hpp"
#include "Box.hpp"
#include "Helpers/Assert.hpp"

/********************************************** Test classes **************************************/

// Registers the fixture into the 'registry'
CPPUNIT_TEST_SUITE_REGISTRATION( BoxUnittest );

void BoxUnittest::setUp(){
  ASSERT_DO(Assert::Throw);
  unit = new Matrix;
  unit->setIdentity();
  zero = new Matrix;
  invertible = new Matrix;
  invertible->diagonal() = Vector(1,2,3);
  uninvertible = new Matrix;
  uninvertible->column(0) = Vector(1,0,1);
  uninvertible->column(2) = Vector(1,0,1);

  Matrix boxMat;
  unitBox = new Box;
  stretchedBox1 = new Box;
  boxMat.setIdentity();
  boxMat.diagonal() = Vector(1,2,3);
  stretchedBox1->setM(boxMat);

  stretchedBox2 = new Box;
  boxMat.setIdentity();
  boxMat.diagonal() = Vector(2,3,1);
  stretchedBox2->setM(boxMat);

  stretchedBox3 = new Box;
  boxMat.setIdentity();
  boxMat.diagonal() = Vector(3,1,2);
  stretchedBox3->setM(boxMat);

  stretchedBox4 = new Box;
  boxMat.setIdentity();
  boxMat.diagonal() = Vector(2,2,2);
  stretchedBox4->setM(boxMat);

  tiltedBox1 = new Box;
  boxMat.setIdentity();
  boxMat.column(0) = Vector(1,0,1);
  tiltedBox1->setM(boxMat);

  tiltedBox2 = new Box;
  boxMat.setIdentity();
  boxMat.column(0) = Vector(1,1,1);
  tiltedBox2->setM(boxMat);

  tiltedBox3 = new Box;
  boxMat.setIdentity();
  boxMat.column(1) = Vector(0,1,1);
  tiltedBox3->setM(boxMat);

  tiltedBox4 = new Box;
  boxMat.setIdentity();
  boxMat.column(0) = Vector(1,1,1);
  boxMat.column(1) = Vector(0,1,1);
  tiltedBox4->setM(boxMat);

}

void BoxUnittest::tearDown(){
  delete unit;
  delete zero;
  delete invertible;
  delete uninvertible;

  delete unitBox;
  delete stretchedBox1;
  delete stretchedBox2;
  delete stretchedBox3;
  delete stretchedBox4;
  delete tiltedBox1;
  delete tiltedBox2;
  delete tiltedBox3;
  delete tiltedBox4;

}

void BoxUnittest::setBoxTest(){
  Box testBox;
  CPPUNIT_ASSERT_NO_THROW(testBox.setM(*unit));
  CPPUNIT_ASSERT_NO_THROW(testBox = *unit);
  CPPUNIT_ASSERT_NO_THROW(testBox.setM(*invertible));
  CPPUNIT_ASSERT_NO_THROW(testBox = *invertible);
#ifndef NDEBUG
  CPPUNIT_ASSERT_THROW(testBox.setM(*zero),Assert::AssertionFailure);
  CPPUNIT_ASSERT_THROW(testBox = *zero,Assert::AssertionFailure);
  CPPUNIT_ASSERT_THROW(testBox.setM(*uninvertible),Assert::AssertionFailure);
  CPPUNIT_ASSERT_THROW(testBox = *uninvertible,Assert::AssertionFailure);
#endif
}

void BoxUnittest::translateInOutTest(){
  Vector testVector;
  {
    testVector=Vector(0,0,0);
    CPPUNIT_ASSERT_EQUAL(unitBox->translateOut(unitBox->translateIn(testVector)),testVector);
    CPPUNIT_ASSERT_EQUAL(unitBox->translateIn(unitBox->translateOut(testVector)),testVector);
    CPPUNIT_ASSERT_EQUAL(stretchedBox1->translateOut(stretchedBox1->translateIn(testVector)),testVector);
    CPPUNIT_ASSERT_EQUAL(stretchedBox1->translateIn(stretchedBox1->translateOut(testVector)),testVector);
    CPPUNIT_ASSERT_EQUAL(stretchedBox2->translateOut(stretchedBox2->translateIn(testVector)),testVector);
    CPPUNIT_ASSERT_EQUAL(stretchedBox2->translateIn(stretchedBox2->translateOut(testVector)),testVector);
    CPPUNIT_ASSERT_EQUAL(stretchedBox3->translateOut(stretchedBox3->translateIn(testVector)),testVector);
    CPPUNIT_ASSERT_EQUAL(stretchedBox3->translateIn(stretchedBox3->translateOut(testVector)),testVector);
    CPPUNIT_ASSERT_EQUAL(stretchedBox4->translateOut(stretchedBox4->translateIn(testVector)),testVector);
    CPPUNIT_ASSERT_EQUAL(stretchedBox4->translateIn(stretchedBox4->translateOut(testVector)),testVector);
    CPPUNIT_ASSERT_EQUAL(tiltedBox1->translateOut(tiltedBox1->translateIn(testVector)),testVector);
    CPPUNIT_ASSERT_EQUAL(tiltedBox1->translateIn(tiltedBox1->translateOut(testVector)),testVector);
    CPPUNIT_ASSERT_EQUAL(tiltedBox2->translateOut(tiltedBox2->translateIn(testVector)),testVector);
    CPPUNIT_ASSERT_EQUAL(tiltedBox2->translateIn(tiltedBox2->translateOut(testVector)),testVector);
    CPPUNIT_ASSERT_EQUAL(tiltedBox3->translateOut(tiltedBox3->translateIn(testVector)),testVector);
    CPPUNIT_ASSERT_EQUAL(tiltedBox3->translateIn(tiltedBox3->translateOut(testVector)),testVector);
    CPPUNIT_ASSERT_EQUAL(tiltedBox4->translateOut(tiltedBox4->translateIn(testVector)),testVector);
    CPPUNIT_ASSERT_EQUAL(tiltedBox4->translateIn(tiltedBox4->translateOut(testVector)),testVector);
  }

  {
    testVector=Vector(0.5,0.5,0.5);
    CPPUNIT_ASSERT_EQUAL(unitBox->translateOut(unitBox->translateIn(testVector)),testVector);
    CPPUNIT_ASSERT_EQUAL(unitBox->translateIn(unitBox->translateOut(testVector)),testVector);
    CPPUNIT_ASSERT_EQUAL(stretchedBox1->translateOut(stretchedBox1->translateIn(testVector)),testVector);
    CPPUNIT_ASSERT_EQUAL(stretchedBox1->translateIn(stretchedBox1->translateOut(testVector)),testVector);
    CPPUNIT_ASSERT_EQUAL(stretchedBox2->translateOut(stretchedBox2->translateIn(testVector)),testVector);
    CPPUNIT_ASSERT_EQUAL(stretchedBox2->translateIn(stretchedBox2->translateOut(testVector)),testVector);
    CPPUNIT_ASSERT_EQUAL(stretchedBox3->translateOut(stretchedBox3->translateIn(testVector)),testVector);
    CPPUNIT_ASSERT_EQUAL(stretchedBox3->translateIn(stretchedBox3->translateOut(testVector)),testVector);
    CPPUNIT_ASSERT_EQUAL(stretchedBox4->translateOut(stretchedBox4->translateIn(testVector)),testVector);
    CPPUNIT_ASSERT_EQUAL(stretchedBox4->translateIn(stretchedBox4->translateOut(testVector)),testVector);
    CPPUNIT_ASSERT_EQUAL(tiltedBox1->translateOut(tiltedBox1->translateIn(testVector)),testVector);
    CPPUNIT_ASSERT_EQUAL(tiltedBox1->translateIn(tiltedBox1->translateOut(testVector)),testVector);
    CPPUNIT_ASSERT_EQUAL(tiltedBox2->translateOut(tiltedBox2->translateIn(testVector)),testVector);
    CPPUNIT_ASSERT_EQUAL(tiltedBox2->translateIn(tiltedBox2->translateOut(testVector)),testVector);
    CPPUNIT_ASSERT_EQUAL(tiltedBox3->translateOut(tiltedBox3->translateIn(testVector)),testVector);
    CPPUNIT_ASSERT_EQUAL(tiltedBox3->translateIn(tiltedBox3->translateOut(testVector)),testVector);
    CPPUNIT_ASSERT_EQUAL(tiltedBox4->translateOut(tiltedBox4->translateIn(testVector)),testVector);
    CPPUNIT_ASSERT_EQUAL(tiltedBox4->translateIn(tiltedBox4->translateOut(testVector)),testVector);
  }

  {
    testVector=Vector(1,1,1);
    CPPUNIT_ASSERT_EQUAL(unitBox->translateOut(unitBox->translateIn(testVector)),testVector);
    CPPUNIT_ASSERT_EQUAL(unitBox->translateIn(unitBox->translateOut(testVector)),testVector);
    CPPUNIT_ASSERT_EQUAL(stretchedBox1->translateOut(stretchedBox1->translateIn(testVector)),testVector);
    CPPUNIT_ASSERT_EQUAL(stretchedBox1->translateIn(stretchedBox1->translateOut(testVector)),testVector);
    CPPUNIT_ASSERT_EQUAL(stretchedBox2->translateOut(stretchedBox2->translateIn(testVector)),testVector);
    CPPUNIT_ASSERT_EQUAL(stretchedBox2->translateIn(stretchedBox2->translateOut(testVector)),testVector);
    CPPUNIT_ASSERT_EQUAL(stretchedBox3->translateOut(stretchedBox3->translateIn(testVector)),testVector);
    CPPUNIT_ASSERT_EQUAL(stretchedBox3->translateIn(stretchedBox3->translateOut(testVector)),testVector);
    CPPUNIT_ASSERT_EQUAL(stretchedBox4->translateOut(stretchedBox4->translateIn(testVector)),testVector);
    CPPUNIT_ASSERT_EQUAL(stretchedBox4->translateIn(stretchedBox4->translateOut(testVector)),testVector);
    CPPUNIT_ASSERT_EQUAL(tiltedBox1->translateOut(tiltedBox1->translateIn(testVector)),testVector);
    CPPUNIT_ASSERT_EQUAL(tiltedBox1->translateIn(tiltedBox1->translateOut(testVector)),testVector);
    CPPUNIT_ASSERT_EQUAL(tiltedBox2->translateOut(tiltedBox2->translateIn(testVector)),testVector);
    CPPUNIT_ASSERT_EQUAL(tiltedBox2->translateIn(tiltedBox2->translateOut(testVector)),testVector);
    CPPUNIT_ASSERT_EQUAL(tiltedBox3->translateOut(tiltedBox3->translateIn(testVector)),testVector);
    CPPUNIT_ASSERT_EQUAL(tiltedBox3->translateIn(tiltedBox3->translateOut(testVector)),testVector);
    CPPUNIT_ASSERT_EQUAL(tiltedBox4->translateOut(tiltedBox4->translateIn(testVector)),testVector);
    CPPUNIT_ASSERT_EQUAL(tiltedBox4->translateIn(tiltedBox4->translateOut(testVector)),testVector);
  }

  {
    testVector=Vector(2,1,1);
    CPPUNIT_ASSERT_EQUAL(unitBox->translateOut(unitBox->translateIn(testVector)),testVector);
    CPPUNIT_ASSERT_EQUAL(unitBox->translateIn(unitBox->translateOut(testVector)),testVector);
    CPPUNIT_ASSERT_EQUAL(stretchedBox1->translateOut(stretchedBox1->translateIn(testVector)),testVector);
    CPPUNIT_ASSERT_EQUAL(stretchedBox1->translateIn(stretchedBox1->translateOut(testVector)),testVector);
    CPPUNIT_ASSERT_EQUAL(stretchedBox2->translateOut(stretchedBox2->translateIn(testVector)),testVector);
    CPPUNIT_ASSERT_EQUAL(stretchedBox2->translateIn(stretchedBox2->translateOut(testVector)),testVector);
    CPPUNIT_ASSERT_EQUAL(stretchedBox3->translateOut(stretchedBox3->translateIn(testVector)),testVector);
    CPPUNIT_ASSERT_EQUAL(stretchedBox3->translateIn(stretchedBox3->translateOut(testVector)),testVector);
    CPPUNIT_ASSERT_EQUAL(stretchedBox4->translateOut(stretchedBox4->translateIn(testVector)),testVector);
    CPPUNIT_ASSERT_EQUAL(stretchedBox4->translateIn(stretchedBox4->translateOut(testVector)),testVector);
    CPPUNIT_ASSERT_EQUAL(tiltedBox1->translateOut(tiltedBox1->translateIn(testVector)),testVector);
    CPPUNIT_ASSERT_EQUAL(tiltedBox1->translateIn(tiltedBox1->translateOut(testVector)),testVector);
    CPPUNIT_ASSERT_EQUAL(tiltedBox2->translateOut(tiltedBox2->translateIn(testVector)),testVector);
    CPPUNIT_ASSERT_EQUAL(tiltedBox2->translateIn(tiltedBox2->translateOut(testVector)),testVector);
    CPPUNIT_ASSERT_EQUAL(tiltedBox3->translateOut(tiltedBox3->translateIn(testVector)),testVector);
    CPPUNIT_ASSERT_EQUAL(tiltedBox3->translateIn(tiltedBox3->translateOut(testVector)),testVector);
    CPPUNIT_ASSERT_EQUAL(tiltedBox4->translateOut(tiltedBox4->translateIn(testVector)),testVector);
    CPPUNIT_ASSERT_EQUAL(tiltedBox4->translateIn(tiltedBox4->translateOut(testVector)),testVector);
  }

  {
    testVector=Vector(1,2,1);
    CPPUNIT_ASSERT_EQUAL(unitBox->translateOut(unitBox->translateIn(testVector)),testVector);
    CPPUNIT_ASSERT_EQUAL(unitBox->translateIn(unitBox->translateOut(testVector)),testVector);
    CPPUNIT_ASSERT_EQUAL(stretchedBox1->translateOut(stretchedBox1->translateIn(testVector)),testVector);
    CPPUNIT_ASSERT_EQUAL(stretchedBox1->translateIn(stretchedBox1->translateOut(testVector)),testVector);
    CPPUNIT_ASSERT_EQUAL(stretchedBox2->translateOut(stretchedBox2->translateIn(testVector)),testVector);
    CPPUNIT_ASSERT_EQUAL(stretchedBox2->translateIn(stretchedBox2->translateOut(testVector)),testVector);
    CPPUNIT_ASSERT_EQUAL(stretchedBox3->translateOut(stretchedBox3->translateIn(testVector)),testVector);
    CPPUNIT_ASSERT_EQUAL(stretchedBox3->translateIn(stretchedBox3->translateOut(testVector)),testVector);
    CPPUNIT_ASSERT_EQUAL(stretchedBox4->translateOut(stretchedBox4->translateIn(testVector)),testVector);
    CPPUNIT_ASSERT_EQUAL(stretchedBox4->translateIn(stretchedBox4->translateOut(testVector)),testVector);
    CPPUNIT_ASSERT_EQUAL(tiltedBox1->translateOut(tiltedBox1->translateIn(testVector)),testVector);
    CPPUNIT_ASSERT_EQUAL(tiltedBox1->translateIn(tiltedBox1->translateOut(testVector)),testVector);
    CPPUNIT_ASSERT_EQUAL(tiltedBox2->translateOut(tiltedBox2->translateIn(testVector)),testVector);
    CPPUNIT_ASSERT_EQUAL(tiltedBox2->translateIn(tiltedBox2->translateOut(testVector)),testVector);
    CPPUNIT_ASSERT_EQUAL(tiltedBox3->translateOut(tiltedBox3->translateIn(testVector)),testVector);
    CPPUNIT_ASSERT_EQUAL(tiltedBox3->translateIn(tiltedBox3->translateOut(testVector)),testVector);
    CPPUNIT_ASSERT_EQUAL(tiltedBox4->translateOut(tiltedBox4->translateIn(testVector)),testVector);
    CPPUNIT_ASSERT_EQUAL(tiltedBox4->translateIn(tiltedBox4->translateOut(testVector)),testVector);
  }

  {
    testVector=Vector(1,1,2);
    CPPUNIT_ASSERT_EQUAL(unitBox->translateOut(unitBox->translateIn(testVector)),testVector);
    CPPUNIT_ASSERT_EQUAL(unitBox->translateIn(unitBox->translateOut(testVector)),testVector);
    CPPUNIT_ASSERT_EQUAL(stretchedBox1->translateOut(stretchedBox1->translateIn(testVector)),testVector);
    CPPUNIT_ASSERT_EQUAL(stretchedBox1->translateIn(stretchedBox1->translateOut(testVector)),testVector);
    CPPUNIT_ASSERT_EQUAL(stretchedBox2->translateOut(stretchedBox2->translateIn(testVector)),testVector);
    CPPUNIT_ASSERT_EQUAL(stretchedBox2->translateIn(stretchedBox2->translateOut(testVector)),testVector);
    CPPUNIT_ASSERT_EQUAL(stretchedBox3->translateOut(stretchedBox3->translateIn(testVector)),testVector);
    CPPUNIT_ASSERT_EQUAL(stretchedBox3->translateIn(stretchedBox3->translateOut(testVector)),testVector);
    CPPUNIT_ASSERT_EQUAL(stretchedBox4->translateOut(stretchedBox4->translateIn(testVector)),testVector);
    CPPUNIT_ASSERT_EQUAL(stretchedBox4->translateIn(stretchedBox4->translateOut(testVector)),testVector);
    CPPUNIT_ASSERT_EQUAL(tiltedBox1->translateOut(tiltedBox1->translateIn(testVector)),testVector);
    CPPUNIT_ASSERT_EQUAL(tiltedBox1->translateIn(tiltedBox1->translateOut(testVector)),testVector);
    CPPUNIT_ASSERT_EQUAL(tiltedBox2->translateOut(tiltedBox2->translateIn(testVector)),testVector);
    CPPUNIT_ASSERT_EQUAL(tiltedBox2->translateIn(tiltedBox2->translateOut(testVector)),testVector);
    CPPUNIT_ASSERT_EQUAL(tiltedBox3->translateOut(tiltedBox3->translateIn(testVector)),testVector);
    CPPUNIT_ASSERT_EQUAL(tiltedBox3->translateIn(tiltedBox3->translateOut(testVector)),testVector);
    CPPUNIT_ASSERT_EQUAL(tiltedBox4->translateOut(tiltedBox4->translateIn(testVector)),testVector);
    CPPUNIT_ASSERT_EQUAL(tiltedBox4->translateIn(tiltedBox4->translateOut(testVector)),testVector);
  }

  {
    testVector=Vector(3,1,1);
    CPPUNIT_ASSERT_EQUAL(unitBox->translateOut(unitBox->translateIn(testVector)),testVector);
    CPPUNIT_ASSERT_EQUAL(unitBox->translateIn(unitBox->translateOut(testVector)),testVector);
    CPPUNIT_ASSERT_EQUAL(stretchedBox1->translateOut(stretchedBox1->translateIn(testVector)),testVector);
    CPPUNIT_ASSERT_EQUAL(stretchedBox1->translateIn(stretchedBox1->translateOut(testVector)),testVector);
    CPPUNIT_ASSERT_EQUAL(stretchedBox2->translateOut(stretchedBox2->translateIn(testVector)),testVector);
    CPPUNIT_ASSERT_EQUAL(stretchedBox2->translateIn(stretchedBox2->translateOut(testVector)),testVector);
    CPPUNIT_ASSERT_EQUAL(stretchedBox3->translateOut(stretchedBox3->translateIn(testVector)),testVector);
    CPPUNIT_ASSERT_EQUAL(stretchedBox3->translateIn(stretchedBox3->translateOut(testVector)),testVector);
    CPPUNIT_ASSERT_EQUAL(stretchedBox4->translateOut(stretchedBox4->translateIn(testVector)),testVector);
    CPPUNIT_ASSERT_EQUAL(stretchedBox4->translateIn(stretchedBox4->translateOut(testVector)),testVector);
    CPPUNIT_ASSERT_EQUAL(tiltedBox1->translateOut(tiltedBox1->translateIn(testVector)),testVector);
    CPPUNIT_ASSERT_EQUAL(tiltedBox1->translateIn(tiltedBox1->translateOut(testVector)),testVector);
    CPPUNIT_ASSERT_EQUAL(tiltedBox2->translateOut(tiltedBox2->translateIn(testVector)),testVector);
    CPPUNIT_ASSERT_EQUAL(tiltedBox2->translateIn(tiltedBox2->translateOut(testVector)),testVector);
    CPPUNIT_ASSERT_EQUAL(tiltedBox3->translateOut(tiltedBox3->translateIn(testVector)),testVector);
    CPPUNIT_ASSERT_EQUAL(tiltedBox3->translateIn(tiltedBox3->translateOut(testVector)),testVector);
    CPPUNIT_ASSERT_EQUAL(tiltedBox4->translateOut(tiltedBox4->translateIn(testVector)),testVector);
    CPPUNIT_ASSERT_EQUAL(tiltedBox4->translateIn(tiltedBox4->translateOut(testVector)),testVector);
  }

  {
    testVector=Vector(1,3,1);
    CPPUNIT_ASSERT_EQUAL(unitBox->translateOut(unitBox->translateIn(testVector)),testVector);
    CPPUNIT_ASSERT_EQUAL(unitBox->translateIn(unitBox->translateOut(testVector)),testVector);
    CPPUNIT_ASSERT_EQUAL(stretchedBox1->translateOut(stretchedBox1->translateIn(testVector)),testVector);
    CPPUNIT_ASSERT_EQUAL(stretchedBox1->translateIn(stretchedBox1->translateOut(testVector)),testVector);
    CPPUNIT_ASSERT_EQUAL(stretchedBox2->translateOut(stretchedBox2->translateIn(testVector)),testVector);
    CPPUNIT_ASSERT_EQUAL(stretchedBox2->translateIn(stretchedBox2->translateOut(testVector)),testVector);
    CPPUNIT_ASSERT_EQUAL(stretchedBox3->translateOut(stretchedBox3->translateIn(testVector)),testVector);
    CPPUNIT_ASSERT_EQUAL(stretchedBox3->translateIn(stretchedBox3->translateOut(testVector)),testVector);
    CPPUNIT_ASSERT_EQUAL(stretchedBox4->translateOut(stretchedBox4->translateIn(testVector)),testVector);
    CPPUNIT_ASSERT_EQUAL(stretchedBox4->translateIn(stretchedBox4->translateOut(testVector)),testVector);
    CPPUNIT_ASSERT_EQUAL(tiltedBox1->translateOut(tiltedBox1->translateIn(testVector)),testVector);
    CPPUNIT_ASSERT_EQUAL(tiltedBox1->translateIn(tiltedBox1->translateOut(testVector)),testVector);
    CPPUNIT_ASSERT_EQUAL(tiltedBox2->translateOut(tiltedBox2->translateIn(testVector)),testVector);
    CPPUNIT_ASSERT_EQUAL(tiltedBox2->translateIn(tiltedBox2->translateOut(testVector)),testVector);
    CPPUNIT_ASSERT_EQUAL(tiltedBox3->translateOut(tiltedBox3->translateIn(testVector)),testVector);
    CPPUNIT_ASSERT_EQUAL(tiltedBox3->translateIn(tiltedBox3->translateOut(testVector)),testVector);
    CPPUNIT_ASSERT_EQUAL(tiltedBox4->translateOut(tiltedBox4->translateIn(testVector)),testVector);
    CPPUNIT_ASSERT_EQUAL(tiltedBox4->translateIn(tiltedBox4->translateOut(testVector)),testVector);
  }

  {
    testVector=Vector(1,1,3);
    CPPUNIT_ASSERT_EQUAL(unitBox->translateOut(unitBox->translateIn(testVector)),testVector);
    CPPUNIT_ASSERT_EQUAL(unitBox->translateIn(unitBox->translateOut(testVector)),testVector);
    CPPUNIT_ASSERT_EQUAL(stretchedBox1->translateOut(stretchedBox1->translateIn(testVector)),testVector);
    CPPUNIT_ASSERT_EQUAL(stretchedBox1->translateIn(stretchedBox1->translateOut(testVector)),testVector);
    CPPUNIT_ASSERT_EQUAL(stretchedBox2->translateOut(stretchedBox2->translateIn(testVector)),testVector);
    CPPUNIT_ASSERT_EQUAL(stretchedBox2->translateIn(stretchedBox2->translateOut(testVector)),testVector);
    CPPUNIT_ASSERT_EQUAL(stretchedBox3->translateOut(stretchedBox3->translateIn(testVector)),testVector);
    CPPUNIT_ASSERT_EQUAL(stretchedBox3->translateIn(stretchedBox3->translateOut(testVector)),testVector);
    CPPUNIT_ASSERT_EQUAL(stretchedBox4->translateOut(stretchedBox4->translateIn(testVector)),testVector);
    CPPUNIT_ASSERT_EQUAL(stretchedBox4->translateIn(stretchedBox4->translateOut(testVector)),testVector);
    CPPUNIT_ASSERT_EQUAL(tiltedBox1->translateOut(tiltedBox1->translateIn(testVector)),testVector);
    CPPUNIT_ASSERT_EQUAL(tiltedBox1->translateIn(tiltedBox1->translateOut(testVector)),testVector);
    CPPUNIT_ASSERT_EQUAL(tiltedBox2->translateOut(tiltedBox2->translateIn(testVector)),testVector);
    CPPUNIT_ASSERT_EQUAL(tiltedBox2->translateIn(tiltedBox2->translateOut(testVector)),testVector);
    CPPUNIT_ASSERT_EQUAL(tiltedBox3->translateOut(tiltedBox3->translateIn(testVector)),testVector);
    CPPUNIT_ASSERT_EQUAL(tiltedBox3->translateIn(tiltedBox3->translateOut(testVector)),testVector);
    CPPUNIT_ASSERT_EQUAL(tiltedBox4->translateOut(tiltedBox4->translateIn(testVector)),testVector);
    CPPUNIT_ASSERT_EQUAL(tiltedBox4->translateIn(tiltedBox4->translateOut(testVector)),testVector);
  }

  {
    testVector=Vector(2,2,2);
    CPPUNIT_ASSERT_EQUAL(unitBox->translateOut(unitBox->translateIn(testVector)),testVector);
    CPPUNIT_ASSERT_EQUAL(unitBox->translateIn(unitBox->translateOut(testVector)),testVector);
    CPPUNIT_ASSERT_EQUAL(stretchedBox1->translateOut(stretchedBox1->translateIn(testVector)),testVector);
    CPPUNIT_ASSERT_EQUAL(stretchedBox1->translateIn(stretchedBox1->translateOut(testVector)),testVector);
    CPPUNIT_ASSERT_EQUAL(stretchedBox2->translateOut(stretchedBox2->translateIn(testVector)),testVector);
    CPPUNIT_ASSERT_EQUAL(stretchedBox2->translateIn(stretchedBox2->translateOut(testVector)),testVector);
    CPPUNIT_ASSERT_EQUAL(stretchedBox3->translateOut(stretchedBox3->translateIn(testVector)),testVector);
    CPPUNIT_ASSERT_EQUAL(stretchedBox3->translateIn(stretchedBox3->translateOut(testVector)),testVector);
    CPPUNIT_ASSERT_EQUAL(stretchedBox4->translateOut(stretchedBox4->translateIn(testVector)),testVector);
    CPPUNIT_ASSERT_EQUAL(stretchedBox4->translateIn(stretchedBox4->translateOut(testVector)),testVector);
    CPPUNIT_ASSERT_EQUAL(tiltedBox1->translateOut(tiltedBox1->translateIn(testVector)),testVector);
    CPPUNIT_ASSERT_EQUAL(tiltedBox1->translateIn(tiltedBox1->translateOut(testVector)),testVector);
    CPPUNIT_ASSERT_EQUAL(tiltedBox2->translateOut(tiltedBox2->translateIn(testVector)),testVector);
    CPPUNIT_ASSERT_EQUAL(tiltedBox2->translateIn(tiltedBox2->translateOut(testVector)),testVector);
    CPPUNIT_ASSERT_EQUAL(tiltedBox3->translateOut(tiltedBox3->translateIn(testVector)),testVector);
    CPPUNIT_ASSERT_EQUAL(tiltedBox3->translateIn(tiltedBox3->translateOut(testVector)),testVector);
    CPPUNIT_ASSERT_EQUAL(tiltedBox4->translateOut(tiltedBox4->translateIn(testVector)),testVector);
    CPPUNIT_ASSERT_EQUAL(tiltedBox4->translateIn(tiltedBox4->translateOut(testVector)),testVector);
  }

}

void BoxUnittest::isInsideTest(){
  Vector testVector(0,0,0);
  CPPUNIT_ASSERT( unitBox->isInside(testVector));
  CPPUNIT_ASSERT( stretchedBox1->isInside(testVector));
  CPPUNIT_ASSERT( stretchedBox2->isInside(testVector));
  CPPUNIT_ASSERT( stretchedBox3->isInside(testVector));
  CPPUNIT_ASSERT( stretchedBox4->isInside(testVector));
  CPPUNIT_ASSERT( tiltedBox1->isInside(testVector));
  CPPUNIT_ASSERT( tiltedBox2->isInside(testVector));
  CPPUNIT_ASSERT( tiltedBox3->isInside(testVector));
  CPPUNIT_ASSERT( tiltedBox4->isInside(testVector));


  testVector = Vector(0.5,0.5,0.5);
  CPPUNIT_ASSERT( unitBox->isInside(testVector));
  CPPUNIT_ASSERT( stretchedBox1->isInside(testVector));
  CPPUNIT_ASSERT( stretchedBox2->isInside(testVector));
  CPPUNIT_ASSERT( stretchedBox3->isInside(testVector));
  CPPUNIT_ASSERT( stretchedBox4->isInside(testVector));
  CPPUNIT_ASSERT( tiltedBox1->isInside(testVector));
  CPPUNIT_ASSERT( tiltedBox2->isInside(testVector));
  CPPUNIT_ASSERT( tiltedBox3->isInside(testVector));
  CPPUNIT_ASSERT( tiltedBox4->isInside(testVector));

  testVector = Vector(1,1,1);
  CPPUNIT_ASSERT( unitBox->isInside(testVector));
  CPPUNIT_ASSERT( stretchedBox1->isInside(testVector));
  CPPUNIT_ASSERT( stretchedBox2->isInside(testVector));
  CPPUNIT_ASSERT( stretchedBox3->isInside(testVector));
  CPPUNIT_ASSERT( stretchedBox4->isInside(testVector));
  CPPUNIT_ASSERT( tiltedBox1->isInside(testVector));
  CPPUNIT_ASSERT( tiltedBox2->isInside(testVector));
  CPPUNIT_ASSERT( tiltedBox3->isInside(testVector));
  CPPUNIT_ASSERT( tiltedBox4->isInside(testVector));

  testVector = Vector(2,1,1);
  CPPUNIT_ASSERT(!unitBox->isInside(testVector));
  CPPUNIT_ASSERT(!stretchedBox1->isInside(testVector));
  CPPUNIT_ASSERT( stretchedBox2->isInside(testVector));
  CPPUNIT_ASSERT( stretchedBox3->isInside(testVector));
  CPPUNIT_ASSERT( stretchedBox4->isInside(testVector));
  CPPUNIT_ASSERT(!tiltedBox1->isInside(testVector));
  CPPUNIT_ASSERT(!tiltedBox2->isInside(testVector));
  CPPUNIT_ASSERT(!tiltedBox3->isInside(testVector));
  CPPUNIT_ASSERT(!tiltedBox4->isInside(testVector));

  testVector = Vector(1,2,1);
  CPPUNIT_ASSERT(!unitBox->isInside(testVector));
  CPPUNIT_ASSERT( stretchedBox1->isInside(testVector));
  CPPUNIT_ASSERT( stretchedBox2->isInside(testVector));
  CPPUNIT_ASSERT(!stretchedBox3->isInside(testVector));
  CPPUNIT_ASSERT( stretchedBox4->isInside(testVector));
  CPPUNIT_ASSERT(!tiltedBox1->isInside(testVector));
  CPPUNIT_ASSERT( tiltedBox2->isInside(testVector));
  CPPUNIT_ASSERT(!tiltedBox3->isInside(testVector));
  CPPUNIT_ASSERT(!tiltedBox4->isInside(testVector));

  testVector = Vector(1,1,2);
  CPPUNIT_ASSERT(!unitBox->isInside(testVector));
  CPPUNIT_ASSERT( stretchedBox1->isInside(testVector));
  CPPUNIT_ASSERT(!stretchedBox2->isInside(testVector));
  CPPUNIT_ASSERT( stretchedBox3->isInside(testVector));
  CPPUNIT_ASSERT( stretchedBox4->isInside(testVector));
  CPPUNIT_ASSERT( tiltedBox1->isInside(testVector));
  CPPUNIT_ASSERT( tiltedBox2->isInside(testVector));
  CPPUNIT_ASSERT( tiltedBox3->isInside(testVector));
  CPPUNIT_ASSERT( tiltedBox4->isInside(testVector));

  testVector = Vector(3,1,1);
  CPPUNIT_ASSERT(!unitBox->isInside(testVector));
  CPPUNIT_ASSERT(!stretchedBox1->isInside(testVector));
  CPPUNIT_ASSERT(!stretchedBox2->isInside(testVector));
  CPPUNIT_ASSERT( stretchedBox3->isInside(testVector));
  CPPUNIT_ASSERT(!stretchedBox4->isInside(testVector));
  CPPUNIT_ASSERT(!tiltedBox1->isInside(testVector));
  CPPUNIT_ASSERT(!tiltedBox2->isInside(testVector));
  CPPUNIT_ASSERT(!tiltedBox3->isInside(testVector));
  CPPUNIT_ASSERT(!tiltedBox4->isInside(testVector));

  testVector = Vector(1,3,1);
  CPPUNIT_ASSERT(!unitBox->isInside(testVector));
  CPPUNIT_ASSERT(!stretchedBox1->isInside(testVector));
  CPPUNIT_ASSERT( stretchedBox2->isInside(testVector));
  CPPUNIT_ASSERT(!stretchedBox3->isInside(testVector));
  CPPUNIT_ASSERT(!stretchedBox4->isInside(testVector));
  CPPUNIT_ASSERT(!tiltedBox1->isInside(testVector));
  CPPUNIT_ASSERT(!tiltedBox2->isInside(testVector));
  CPPUNIT_ASSERT(!tiltedBox3->isInside(testVector));
  CPPUNIT_ASSERT(!tiltedBox4->isInside(testVector));

  testVector = Vector(1,1,3);
  CPPUNIT_ASSERT(!unitBox->isInside(testVector));
  CPPUNIT_ASSERT( stretchedBox1->isInside(testVector));
  CPPUNIT_ASSERT(!stretchedBox2->isInside(testVector));
  CPPUNIT_ASSERT(!stretchedBox3->isInside(testVector));
  CPPUNIT_ASSERT(!stretchedBox4->isInside(testVector));
  CPPUNIT_ASSERT(!tiltedBox1->isInside(testVector));
  CPPUNIT_ASSERT(!tiltedBox2->isInside(testVector));
  CPPUNIT_ASSERT(!tiltedBox3->isInside(testVector));
  CPPUNIT_ASSERT(!tiltedBox4->isInside(testVector));

  testVector = Vector(2,2,2);
  CPPUNIT_ASSERT(!unitBox->isInside(testVector));
  CPPUNIT_ASSERT(!stretchedBox1->isInside(testVector));
  CPPUNIT_ASSERT(!stretchedBox2->isInside(testVector));
  CPPUNIT_ASSERT(!stretchedBox3->isInside(testVector));
  CPPUNIT_ASSERT( stretchedBox4->isInside(testVector));
  CPPUNIT_ASSERT(!tiltedBox1->isInside(testVector));
  CPPUNIT_ASSERT(!tiltedBox2->isInside(testVector));
  CPPUNIT_ASSERT(!tiltedBox3->isInside(testVector));
  CPPUNIT_ASSERT(!tiltedBox4->isInside(testVector));

  testVector = Vector(3,3,3);
  CPPUNIT_ASSERT(!unitBox->isInside(testVector));
  CPPUNIT_ASSERT(!stretchedBox1->isInside(testVector));
  CPPUNIT_ASSERT(!stretchedBox2->isInside(testVector));
  CPPUNIT_ASSERT(!stretchedBox3->isInside(testVector));
  CPPUNIT_ASSERT(!stretchedBox4->isInside(testVector));
  CPPUNIT_ASSERT(!tiltedBox1->isInside(testVector));
  CPPUNIT_ASSERT(!tiltedBox2->isInside(testVector));
  CPPUNIT_ASSERT(!tiltedBox3->isInside(testVector));
  CPPUNIT_ASSERT(!tiltedBox4->isInside(testVector));
}

bool testWrapExplode(VECTORSET(std::list) &set,Vector &point, Box* box){
  bool res = true;
  Vector wrappedPoint = box->WrapPeriodically(point);
  for(std::list<Vector>::iterator iter = set.begin(); iter!=set.end();++iter){
    Vector wrapped = box->WrapPeriodically(*iter);
    bool equals = (wrapped == wrappedPoint);
    res = res && equals;
    if(!equals){
      cout << "Wrapped vector " << wrapped << " produced from vector " << (*iter) << " does not match target " << wrappedPoint << endl;
    }
  }
  return res;
}

void BoxUnittest::WrapExplodeTest(){
  Vector testVector(0,0,0);
  VECTORSET(std::list) res;

  // we only can explode those vectors that are actually inside the box
  {
    testVector = Vector(0,0,0);
    res = unitBox->explode(testVector,1);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,unitBox));
    res = stretchedBox1->explode(testVector,1);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,stretchedBox1));
    res = stretchedBox2->explode(testVector,1);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,stretchedBox2));
    res = stretchedBox3->explode(testVector,1);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,stretchedBox3));
    res = stretchedBox4->explode(testVector,1);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,stretchedBox4));
    res = tiltedBox1->explode(testVector,1);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,tiltedBox1));
    res = tiltedBox2->explode(testVector,1);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,tiltedBox2));
    res = tiltedBox3->explode(testVector,1);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,tiltedBox3));
    res = tiltedBox4->explode(testVector,1);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,tiltedBox4));
  }

  {
    testVector = Vector(0.5,0.5,0.5);
    res = unitBox->explode(testVector,1);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,unitBox));
    res = stretchedBox1->explode(testVector,1);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,stretchedBox1));
    res = stretchedBox2->explode(testVector,1);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,stretchedBox2));
    res = stretchedBox3->explode(testVector,1);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,stretchedBox3));
    res = stretchedBox4->explode(testVector,1);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,stretchedBox4));
    res = tiltedBox1->explode(testVector,1);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,tiltedBox1));
    res = tiltedBox2->explode(testVector,1);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,tiltedBox2));
    res = tiltedBox3->explode(testVector,1);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,tiltedBox3));
    res = tiltedBox4->explode(testVector,1);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,tiltedBox4));
  }

  {
    testVector = Vector(1,1,1);
    res = unitBox->explode(testVector,1);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,unitBox));
    res = stretchedBox1->explode(testVector,1);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,stretchedBox1));
    res = stretchedBox2->explode(testVector,1);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,stretchedBox2));
    res = stretchedBox3->explode(testVector,1);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,stretchedBox3));
    res = stretchedBox4->explode(testVector,1);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,stretchedBox4));
    res = tiltedBox1->explode(testVector,1);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,tiltedBox1));
    res = tiltedBox2->explode(testVector,1);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,tiltedBox2));
    res = tiltedBox3->explode(testVector,1);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,tiltedBox3));
    res = tiltedBox4->explode(testVector,1);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,tiltedBox4));
  }

  {
    testVector = Vector(2,1,1);
    res = stretchedBox2->explode(testVector,1);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,stretchedBox2));
    res = stretchedBox3->explode(testVector,1);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,stretchedBox3));
    res = stretchedBox4->explode(testVector,1);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,stretchedBox4));
  }

  {
    testVector = Vector(1,2,1);
    res = stretchedBox1->explode(testVector,1);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,stretchedBox1));
    res = stretchedBox2->explode(testVector,1);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,stretchedBox2));
    res = stretchedBox4->explode(testVector,1);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,tiltedBox1));
    res = tiltedBox2->explode(testVector,1);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,tiltedBox2));
  }

  {
    testVector = Vector(1,1,2);
    res = stretchedBox1->explode(testVector,1);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,stretchedBox1));
    res = stretchedBox3->explode(testVector,1);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,stretchedBox3));
    res = stretchedBox4->explode(testVector,1);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,stretchedBox4));
    res = tiltedBox1->explode(testVector,1);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,tiltedBox1));
    res = tiltedBox2->explode(testVector,1);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,tiltedBox2));
    res = tiltedBox3->explode(testVector,1);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,tiltedBox3));
    res = tiltedBox4->explode(testVector,1);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,tiltedBox4));
  }

  {
    testVector = Vector(3,1,1);
    res = stretchedBox3->explode(testVector,1);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,stretchedBox3));
  }

  {
    testVector = Vector(1,3,1);
    res = stretchedBox2->explode(testVector,1);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,stretchedBox2));
  }

  {
    testVector = Vector(1,1,3);
    res = stretchedBox1->explode(testVector,1);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,stretchedBox1));
  }

  {
    testVector = Vector(2,2,2);
    res = stretchedBox4->explode(testVector,1);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,stretchedBox4));
  }

  // Higher level explosions

  {
    testVector = Vector(0,0,0);
    res = unitBox->explode(testVector,2);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,unitBox));
    res = stretchedBox1->explode(testVector,2);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,stretchedBox1));
    res = stretchedBox2->explode(testVector,2);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,stretchedBox2));
    res = stretchedBox3->explode(testVector,2);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,stretchedBox3));
    res = stretchedBox4->explode(testVector,2);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,stretchedBox4));
    res = tiltedBox1->explode(testVector,2);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,tiltedBox1));
    res = tiltedBox2->explode(testVector,2);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,tiltedBox2));
    res = tiltedBox3->explode(testVector,2);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,tiltedBox3));
    res = tiltedBox4->explode(testVector,2);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,tiltedBox4));
  }

  {
    testVector = Vector(0.5,0.5,0.5);
    res = unitBox->explode(testVector,2);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,unitBox));
    res = stretchedBox1->explode(testVector,2);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,stretchedBox1));
    res = stretchedBox2->explode(testVector,2);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,stretchedBox2));
    res = stretchedBox3->explode(testVector,2);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,stretchedBox3));
    res = stretchedBox4->explode(testVector,2);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,stretchedBox4));
    res = tiltedBox1->explode(testVector,2);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,tiltedBox1));
    res = tiltedBox2->explode(testVector,2);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,tiltedBox2));
    res = tiltedBox3->explode(testVector,2);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,tiltedBox3));
    res = tiltedBox4->explode(testVector,2);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,tiltedBox4));
  }

  {
    testVector = Vector(1,1,1);
    res = unitBox->explode(testVector,2);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,unitBox));
    res = stretchedBox1->explode(testVector,2);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,stretchedBox1));
    res = stretchedBox2->explode(testVector,2);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,stretchedBox2));
    res = stretchedBox3->explode(testVector,2);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,stretchedBox3));
    res = stretchedBox4->explode(testVector,2);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,stretchedBox4));
    res = tiltedBox1->explode(testVector,2);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,tiltedBox1));
    res = tiltedBox2->explode(testVector,2);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,tiltedBox2));
    res = tiltedBox3->explode(testVector,2);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,tiltedBox3));
    res = tiltedBox4->explode(testVector,2);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,tiltedBox4));
  }

  {
    testVector = Vector(2,1,1);
    res = stretchedBox2->explode(testVector,2);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,stretchedBox2));
    res = stretchedBox3->explode(testVector,2);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,stretchedBox3));
    res = stretchedBox4->explode(testVector,2);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,stretchedBox4));
  }

  {
    testVector = Vector(1,2,1);
    res = stretchedBox1->explode(testVector,2);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,stretchedBox1));
    res = stretchedBox2->explode(testVector,2);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,stretchedBox2));
    res = stretchedBox4->explode(testVector,2);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,tiltedBox1));
    res = tiltedBox2->explode(testVector,2);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,tiltedBox2));
  }

  {
    testVector = Vector(1,1,2);
    res = stretchedBox1->explode(testVector,2);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,stretchedBox1));
    res = stretchedBox3->explode(testVector,2);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,stretchedBox3));
    res = stretchedBox4->explode(testVector,2);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,stretchedBox4));
    res = tiltedBox1->explode(testVector,2);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,tiltedBox1));
    res = tiltedBox2->explode(testVector,2);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,tiltedBox2));
    res = tiltedBox3->explode(testVector,2);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,tiltedBox3));
    res = tiltedBox4->explode(testVector,2);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,tiltedBox4));
  }

  {
    testVector = Vector(3,1,1);
    res = stretchedBox3->explode(testVector,2);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,stretchedBox3));
  }

  {
    testVector = Vector(1,3,1);
    res = stretchedBox2->explode(testVector,2);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,stretchedBox2));
  }

  {
    testVector = Vector(1,1,3);
    res = stretchedBox1->explode(testVector,2);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,stretchedBox1));
  }

  {
    testVector = Vector(2,2,2);
    res = stretchedBox4->explode(testVector,2);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,stretchedBox4));
  }

  // one more set of higher level explosions

  {
    testVector = Vector(0,0,0);
    res = unitBox->explode(testVector,3);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,unitBox));
    res = stretchedBox1->explode(testVector,3);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,stretchedBox1));
    res = stretchedBox2->explode(testVector,3);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,stretchedBox2));
    res = stretchedBox3->explode(testVector,3);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,stretchedBox3));
    res = stretchedBox4->explode(testVector,3);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,stretchedBox4));
    res = tiltedBox1->explode(testVector,3);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,tiltedBox1));
    res = tiltedBox2->explode(testVector,3);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,tiltedBox2));
    res = tiltedBox3->explode(testVector,3);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,tiltedBox3));
    res = tiltedBox4->explode(testVector,3);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,tiltedBox4));
  }

  {
    testVector = Vector(0.5,0.5,0.5);
    res = unitBox->explode(testVector,3);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,unitBox));
    res = stretchedBox1->explode(testVector,3);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,stretchedBox1));
    res = stretchedBox2->explode(testVector,3);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,stretchedBox2));
    res = stretchedBox3->explode(testVector,3);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,stretchedBox3));
    res = stretchedBox4->explode(testVector,3);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,stretchedBox4));
    res = tiltedBox1->explode(testVector,3);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,tiltedBox1));
    res = tiltedBox2->explode(testVector,3);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,tiltedBox2));
    res = tiltedBox3->explode(testVector,3);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,tiltedBox3));
    res = tiltedBox4->explode(testVector,3);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,tiltedBox4));
  }

  {
    testVector = Vector(1,1,1);
    res = unitBox->explode(testVector,3);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,unitBox));
    res = stretchedBox1->explode(testVector,3);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,stretchedBox1));
    res = stretchedBox2->explode(testVector,3);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,stretchedBox2));
    res = stretchedBox3->explode(testVector,3);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,stretchedBox3));
    res = stretchedBox4->explode(testVector,3);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,stretchedBox4));
    res = tiltedBox1->explode(testVector,3);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,tiltedBox1));
    res = tiltedBox2->explode(testVector,3);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,tiltedBox2));
    res = tiltedBox3->explode(testVector,3);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,tiltedBox3));
    res = tiltedBox4->explode(testVector,3);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,tiltedBox4));
  }

  {
    testVector = Vector(2,1,1);
    res = stretchedBox2->explode(testVector,3);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,stretchedBox2));
    res = stretchedBox3->explode(testVector,3);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,stretchedBox3));
    res = stretchedBox4->explode(testVector,3);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,stretchedBox4));
  }

  {
    testVector = Vector(1,2,1);
    res = stretchedBox1->explode(testVector,3);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,stretchedBox1));
    res = stretchedBox2->explode(testVector,3);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,stretchedBox2));
    res = stretchedBox4->explode(testVector,3);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,tiltedBox1));
    res = tiltedBox2->explode(testVector,3);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,tiltedBox2));
  }

  {
    testVector = Vector(1,1,2);
    res = stretchedBox1->explode(testVector,3);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,stretchedBox1));
    res = stretchedBox3->explode(testVector,3);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,stretchedBox3));
    res = stretchedBox4->explode(testVector,3);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,stretchedBox4));
    res = tiltedBox1->explode(testVector,3);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,tiltedBox1));
    res = tiltedBox2->explode(testVector,3);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,tiltedBox2));
    res = tiltedBox3->explode(testVector,3);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,tiltedBox3));
    res = tiltedBox4->explode(testVector,3);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,tiltedBox4));
  }

  {
    testVector = Vector(3,1,1);
    res = stretchedBox3->explode(testVector,3);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,stretchedBox3));
  }

  {
    testVector = Vector(1,3,1);
    res = stretchedBox2->explode(testVector,3);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,stretchedBox2));
  }

  {
    testVector = Vector(1,1,3);
    res = stretchedBox1->explode(testVector,3);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,stretchedBox1));
  }

  {
    testVector = Vector(2,2,2);
    res = stretchedBox4->explode(testVector,3);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,stretchedBox4));
  }
}

void BoxUnittest::BoundaryBounceTest(){
  Vector testVector(0,0,0);
  VECTORSET(std::list) res;

  unitBox->setCondition(0,Box::Bounce);
  stretchedBox1->setCondition(0,Box::Bounce);
  stretchedBox2->setCondition(0,Box::Bounce);
  stretchedBox3->setCondition(0,Box::Bounce);
  stretchedBox4->setCondition(0,Box::Bounce);
  tiltedBox1->setCondition(0,Box::Bounce);
  tiltedBox2->setCondition(0,Box::Bounce);
  tiltedBox3->setCondition(0,Box::Bounce);
  tiltedBox4->setCondition(0,Box::Bounce);

  {
    testVector = Vector(0,0,0);
    res = unitBox->explode(testVector,1);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,unitBox));
    res = stretchedBox1->explode(testVector,1);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,stretchedBox1));
    res = stretchedBox2->explode(testVector,1);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,stretchedBox2));
    res = stretchedBox3->explode(testVector,1);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,stretchedBox3));
    res = stretchedBox4->explode(testVector,1);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,stretchedBox4));
    res = tiltedBox1->explode(testVector,1);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,tiltedBox1));
    res = tiltedBox2->explode(testVector,1);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,tiltedBox2));
    res = tiltedBox3->explode(testVector,1);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,tiltedBox3));
    res = tiltedBox4->explode(testVector,1);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,tiltedBox4));
  }

  {
    testVector = Vector(0.5,0.5,0.5);
    res = unitBox->explode(testVector,1);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,unitBox));
    res = stretchedBox1->explode(testVector,1);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,stretchedBox1));
    res = stretchedBox2->explode(testVector,1);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,stretchedBox2));
    res = stretchedBox3->explode(testVector,1);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,stretchedBox3));
    res = stretchedBox4->explode(testVector,1);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,stretchedBox4));
    res = tiltedBox1->explode(testVector,1);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,tiltedBox1));
    res = tiltedBox2->explode(testVector,1);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,tiltedBox2));
    res = tiltedBox3->explode(testVector,1);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,tiltedBox3));
    res = tiltedBox4->explode(testVector,1);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,tiltedBox4));
  }

  {
    testVector = Vector(0,0,0);
    res = unitBox->explode(testVector,2);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,unitBox));
    res = stretchedBox1->explode(testVector,2);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,stretchedBox1));
    res = stretchedBox2->explode(testVector,2);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,stretchedBox2));
    res = stretchedBox3->explode(testVector,2);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,stretchedBox3));
    res = stretchedBox4->explode(testVector,2);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,stretchedBox4));
    res = tiltedBox1->explode(testVector,2);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,tiltedBox1));
    res = tiltedBox2->explode(testVector,2);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,tiltedBox2));
    res = tiltedBox3->explode(testVector,2);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,tiltedBox3));
    res = tiltedBox4->explode(testVector,2);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,tiltedBox4));
  }

  {
    testVector = Vector(0.5,0.5,0.5);
    res = unitBox->explode(testVector,2);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,unitBox));
    res = stretchedBox1->explode(testVector,2);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,stretchedBox1));
    res = stretchedBox2->explode(testVector,2);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,stretchedBox2));
    res = stretchedBox3->explode(testVector,2);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,stretchedBox3));
    res = stretchedBox4->explode(testVector,2);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,stretchedBox4));
    res = tiltedBox1->explode(testVector,2);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,tiltedBox1));
    res = tiltedBox2->explode(testVector,2);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,tiltedBox2));
    res = tiltedBox3->explode(testVector,2);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,tiltedBox3));
    res = tiltedBox4->explode(testVector,2);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,tiltedBox4));
  }

  unitBox->setCondition(1,Box::Bounce);
  stretchedBox1->setCondition(1,Box::Bounce);
  stretchedBox2->setCondition(1,Box::Bounce);
  stretchedBox3->setCondition(1,Box::Bounce);
  stretchedBox4->setCondition(1,Box::Bounce);
  tiltedBox1->setCondition(1,Box::Bounce);
  tiltedBox2->setCondition(1,Box::Bounce);
  tiltedBox3->setCondition(1,Box::Bounce);
  tiltedBox4->setCondition(1,Box::Bounce);

  {
    testVector = Vector(0,0,0);
    res = unitBox->explode(testVector,1);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,unitBox));
    res = stretchedBox1->explode(testVector,1);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,stretchedBox1));
    res = stretchedBox2->explode(testVector,1);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,stretchedBox2));
    res = stretchedBox3->explode(testVector,1);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,stretchedBox3));
    res = stretchedBox4->explode(testVector,1);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,stretchedBox4));
    res = tiltedBox1->explode(testVector,1);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,tiltedBox1));
    res = tiltedBox2->explode(testVector,1);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,tiltedBox2));
    res = tiltedBox3->explode(testVector,1);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,tiltedBox3));
    res = tiltedBox4->explode(testVector,1);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,tiltedBox4));
  }

  {
    testVector = Vector(0.5,0.5,0.5);
    res = unitBox->explode(testVector,1);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,unitBox));
    res = stretchedBox1->explode(testVector,1);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,stretchedBox1));
    res = stretchedBox2->explode(testVector,1);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,stretchedBox2));
    res = stretchedBox3->explode(testVector,1);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,stretchedBox3));
    res = stretchedBox4->explode(testVector,1);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,stretchedBox4));
    res = tiltedBox1->explode(testVector,1);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,tiltedBox1));
    res = tiltedBox2->explode(testVector,1);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,tiltedBox2));
    res = tiltedBox3->explode(testVector,1);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,tiltedBox3));
    res = tiltedBox4->explode(testVector,1);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,tiltedBox4));
  }

  {
    testVector = Vector(0,0,0);
    res = unitBox->explode(testVector,2);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,unitBox));
    res = stretchedBox1->explode(testVector,2);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,stretchedBox1));
    res = stretchedBox2->explode(testVector,2);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,stretchedBox2));
    res = stretchedBox3->explode(testVector,2);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,stretchedBox3));
    res = stretchedBox4->explode(testVector,2);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,stretchedBox4));
    res = tiltedBox1->explode(testVector,2);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,tiltedBox1));
    res = tiltedBox2->explode(testVector,2);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,tiltedBox2));
    res = tiltedBox3->explode(testVector,2);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,tiltedBox3));
    res = tiltedBox4->explode(testVector,2);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,tiltedBox4));
  }

  {
    testVector = Vector(0.5,0.5,0.5);
    res = unitBox->explode(testVector,2);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,unitBox));
    res = stretchedBox1->explode(testVector,2);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,stretchedBox1));
    res = stretchedBox2->explode(testVector,2);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,stretchedBox2));
    res = stretchedBox3->explode(testVector,2);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,stretchedBox3));
    res = stretchedBox4->explode(testVector,2);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,stretchedBox4));
    res = tiltedBox1->explode(testVector,2);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,tiltedBox1));
    res = tiltedBox2->explode(testVector,2);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,tiltedBox2));
    res = tiltedBox3->explode(testVector,2);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,tiltedBox3));
    res = tiltedBox4->explode(testVector,2);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,tiltedBox4));
  }

  unitBox->setCondition(2,Box::Bounce);
  stretchedBox1->setCondition(2,Box::Bounce);
  stretchedBox2->setCondition(2,Box::Bounce);
  stretchedBox3->setCondition(2,Box::Bounce);
  stretchedBox4->setCondition(2,Box::Bounce);
  tiltedBox1->setCondition(2,Box::Bounce);
  tiltedBox2->setCondition(2,Box::Bounce);
  tiltedBox3->setCondition(2,Box::Bounce);
  tiltedBox4->setCondition(2,Box::Bounce);

  {
    testVector = Vector(0,0,0);
    res = unitBox->explode(testVector,1);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,unitBox));
    res = stretchedBox1->explode(testVector,1);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,stretchedBox1));
    res = stretchedBox2->explode(testVector,1);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,stretchedBox2));
    res = stretchedBox3->explode(testVector,1);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,stretchedBox3));
    res = stretchedBox4->explode(testVector,1);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,stretchedBox4));
    res = tiltedBox1->explode(testVector,1);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,tiltedBox1));
    res = tiltedBox2->explode(testVector,1);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,tiltedBox2));
    res = tiltedBox3->explode(testVector,1);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,tiltedBox3));
    res = tiltedBox4->explode(testVector,1);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,tiltedBox4));
  }

  {
    testVector = Vector(0.5,0.5,0.5);
    res = unitBox->explode(testVector,1);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,unitBox));
    res = stretchedBox1->explode(testVector,1);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,stretchedBox1));
    res = stretchedBox2->explode(testVector,1);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,stretchedBox2));
    res = stretchedBox3->explode(testVector,1);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,stretchedBox3));
    res = stretchedBox4->explode(testVector,1);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,stretchedBox4));
    res = tiltedBox1->explode(testVector,1);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,tiltedBox1));
    res = tiltedBox2->explode(testVector,1);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,tiltedBox2));
    res = tiltedBox3->explode(testVector,1);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,tiltedBox3));
    res = tiltedBox4->explode(testVector,1);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,tiltedBox4));
  }

  {
    testVector = Vector(0,0,0);
    res = unitBox->explode(testVector,2);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,unitBox));
    res = stretchedBox1->explode(testVector,2);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,stretchedBox1));
    res = stretchedBox2->explode(testVector,2);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,stretchedBox2));
    res = stretchedBox3->explode(testVector,2);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,stretchedBox3));
    res = stretchedBox4->explode(testVector,2);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,stretchedBox4));
    res = tiltedBox1->explode(testVector,2);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,tiltedBox1));
    res = tiltedBox2->explode(testVector,2);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,tiltedBox2));
    res = tiltedBox3->explode(testVector,2);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,tiltedBox3));
    res = tiltedBox4->explode(testVector,2);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,tiltedBox4));
  }

  {
    testVector = Vector(0.5,0.5,0.5);
    res = unitBox->explode(testVector,2);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,unitBox));
    res = stretchedBox1->explode(testVector,2);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,stretchedBox1));
    res = stretchedBox2->explode(testVector,2);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,stretchedBox2));
    res = stretchedBox3->explode(testVector,2);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,stretchedBox3));
    res = stretchedBox4->explode(testVector,2);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,stretchedBox4));
    res = tiltedBox1->explode(testVector,2);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,tiltedBox1));
    res = tiltedBox2->explode(testVector,2);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,tiltedBox2));
    res = tiltedBox3->explode(testVector,2);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,tiltedBox3));
    res = tiltedBox4->explode(testVector,2);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,tiltedBox4));
  }
}

void BoxUnittest::BoundaryIgnoreTest(){
  Vector testVector(0,0,0);
  VECTORSET(std::list) res;

  unitBox->setCondition(0,Box::Ignore);
  stretchedBox1->setCondition(0,Box::Ignore);
  stretchedBox2->setCondition(0,Box::Ignore);
  stretchedBox3->setCondition(0,Box::Ignore);
  stretchedBox4->setCondition(0,Box::Ignore);
  tiltedBox1->setCondition(0,Box::Ignore);
  tiltedBox2->setCondition(0,Box::Ignore);
  tiltedBox3->setCondition(0,Box::Ignore);
  tiltedBox4->setCondition(0,Box::Ignore);

  {
    testVector = Vector(0,0,0);
    res = unitBox->explode(testVector,1);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,unitBox));
    res = stretchedBox1->explode(testVector,1);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,stretchedBox1));
    res = stretchedBox2->explode(testVector,1);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,stretchedBox2));
    res = stretchedBox3->explode(testVector,1);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,stretchedBox3));
    res = stretchedBox4->explode(testVector,1);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,stretchedBox4));
    res = tiltedBox1->explode(testVector,1);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,tiltedBox1));
    res = tiltedBox2->explode(testVector,1);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,tiltedBox2));
    res = tiltedBox3->explode(testVector,1);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,tiltedBox3));
    res = tiltedBox4->explode(testVector,1);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,tiltedBox4));
  }

  {
    testVector = Vector(0.5,0.5,0.5);
    res = unitBox->explode(testVector,1);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,unitBox));
    res = stretchedBox1->explode(testVector,1);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,stretchedBox1));
    res = stretchedBox2->explode(testVector,1);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,stretchedBox2));
    res = stretchedBox3->explode(testVector,1);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,stretchedBox3));
    res = stretchedBox4->explode(testVector,1);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,stretchedBox4));
    res = tiltedBox1->explode(testVector,1);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,tiltedBox1));
    res = tiltedBox2->explode(testVector,1);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,tiltedBox2));
    res = tiltedBox3->explode(testVector,1);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,tiltedBox3));
    res = tiltedBox4->explode(testVector,1);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,tiltedBox4));
  }

  {
    testVector = Vector(0,0,0);
    res = unitBox->explode(testVector,2);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,unitBox));
    res = stretchedBox1->explode(testVector,2);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,stretchedBox1));
    res = stretchedBox2->explode(testVector,2);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,stretchedBox2));
    res = stretchedBox3->explode(testVector,2);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,stretchedBox3));
    res = stretchedBox4->explode(testVector,2);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,stretchedBox4));
    res = tiltedBox1->explode(testVector,2);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,tiltedBox1));
    res = tiltedBox2->explode(testVector,2);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,tiltedBox2));
    res = tiltedBox3->explode(testVector,2);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,tiltedBox3));
    res = tiltedBox4->explode(testVector,2);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,tiltedBox4));
  }

  {
    testVector = Vector(0.5,0.5,0.5);
    res = unitBox->explode(testVector,2);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,unitBox));
    res = stretchedBox1->explode(testVector,2);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,stretchedBox1));
    res = stretchedBox2->explode(testVector,2);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,stretchedBox2));
    res = stretchedBox3->explode(testVector,2);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,stretchedBox3));
    res = stretchedBox4->explode(testVector,2);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,stretchedBox4));
    res = tiltedBox1->explode(testVector,2);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,tiltedBox1));
    res = tiltedBox2->explode(testVector,2);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,tiltedBox2));
    res = tiltedBox3->explode(testVector,2);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,tiltedBox3));
    res = tiltedBox4->explode(testVector,2);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,tiltedBox4));
  }

  unitBox->setCondition(1,Box::Ignore);
  stretchedBox1->setCondition(1,Box::Ignore);
  stretchedBox2->setCondition(1,Box::Ignore);
  stretchedBox3->setCondition(1,Box::Ignore);
  stretchedBox4->setCondition(1,Box::Ignore);
  tiltedBox1->setCondition(1,Box::Ignore);
  tiltedBox2->setCondition(1,Box::Ignore);
  tiltedBox3->setCondition(1,Box::Ignore);
  tiltedBox4->setCondition(1,Box::Ignore);

  {
    testVector = Vector(0,0,0);
    res = unitBox->explode(testVector,1);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,unitBox));
    res = stretchedBox1->explode(testVector,1);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,stretchedBox1));
    res = stretchedBox2->explode(testVector,1);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,stretchedBox2));
    res = stretchedBox3->explode(testVector,1);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,stretchedBox3));
    res = stretchedBox4->explode(testVector,1);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,stretchedBox4));
    res = tiltedBox1->explode(testVector,1);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,tiltedBox1));
    res = tiltedBox2->explode(testVector,1);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,tiltedBox2));
    res = tiltedBox3->explode(testVector,1);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,tiltedBox3));
    res = tiltedBox4->explode(testVector,1);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,tiltedBox4));
  }

  {
    testVector = Vector(0.5,0.5,0.5);
    res = unitBox->explode(testVector,1);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,unitBox));
    res = stretchedBox1->explode(testVector,1);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,stretchedBox1));
    res = stretchedBox2->explode(testVector,1);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,stretchedBox2));
    res = stretchedBox3->explode(testVector,1);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,stretchedBox3));
    res = stretchedBox4->explode(testVector,1);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,stretchedBox4));
    res = tiltedBox1->explode(testVector,1);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,tiltedBox1));
    res = tiltedBox2->explode(testVector,1);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,tiltedBox2));
    res = tiltedBox3->explode(testVector,1);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,tiltedBox3));
    res = tiltedBox4->explode(testVector,1);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,tiltedBox4));
  }

  {
    testVector = Vector(0,0,0);
    res = unitBox->explode(testVector,2);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,unitBox));
    res = stretchedBox1->explode(testVector,2);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,stretchedBox1));
    res = stretchedBox2->explode(testVector,2);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,stretchedBox2));
    res = stretchedBox3->explode(testVector,2);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,stretchedBox3));
    res = stretchedBox4->explode(testVector,2);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,stretchedBox4));
    res = tiltedBox1->explode(testVector,2);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,tiltedBox1));
    res = tiltedBox2->explode(testVector,2);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,tiltedBox2));
    res = tiltedBox3->explode(testVector,2);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,tiltedBox3));
    res = tiltedBox4->explode(testVector,2);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,tiltedBox4));
  }

  {
    testVector = Vector(0.5,0.5,0.5);
    res = unitBox->explode(testVector,2);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,unitBox));
    res = stretchedBox1->explode(testVector,2);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,stretchedBox1));
    res = stretchedBox2->explode(testVector,2);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,stretchedBox2));
    res = stretchedBox3->explode(testVector,2);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,stretchedBox3));
    res = stretchedBox4->explode(testVector,2);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,stretchedBox4));
    res = tiltedBox1->explode(testVector,2);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,tiltedBox1));
    res = tiltedBox2->explode(testVector,2);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,tiltedBox2));
    res = tiltedBox3->explode(testVector,2);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,tiltedBox3));
    res = tiltedBox4->explode(testVector,2);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,tiltedBox4));
  }

  unitBox->setCondition(2,Box::Ignore);
  stretchedBox1->setCondition(2,Box::Ignore);
  stretchedBox2->setCondition(2,Box::Ignore);
  stretchedBox3->setCondition(2,Box::Ignore);
  stretchedBox4->setCondition(2,Box::Ignore);
  tiltedBox1->setCondition(2,Box::Ignore);
  tiltedBox2->setCondition(2,Box::Ignore);
  tiltedBox3->setCondition(2,Box::Ignore);
  tiltedBox4->setCondition(2,Box::Ignore);

  {
    testVector = Vector(0,0,0);
    res = unitBox->explode(testVector,1);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,unitBox));
    res = stretchedBox1->explode(testVector,1);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,stretchedBox1));
    res = stretchedBox2->explode(testVector,1);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,stretchedBox2));
    res = stretchedBox3->explode(testVector,1);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,stretchedBox3));
    res = stretchedBox4->explode(testVector,1);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,stretchedBox4));
    res = tiltedBox1->explode(testVector,1);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,tiltedBox1));
    res = tiltedBox2->explode(testVector,1);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,tiltedBox2));
    res = tiltedBox3->explode(testVector,1);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,tiltedBox3));
    res = tiltedBox4->explode(testVector,1);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,tiltedBox4));
  }

  {
    testVector = Vector(0.5,0.5,0.5);
    res = unitBox->explode(testVector,1);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,unitBox));
    res = stretchedBox1->explode(testVector,1);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,stretchedBox1));
    res = stretchedBox2->explode(testVector,1);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,stretchedBox2));
    res = stretchedBox3->explode(testVector,1);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,stretchedBox3));
    res = stretchedBox4->explode(testVector,1);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,stretchedBox4));
    res = tiltedBox1->explode(testVector,1);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,tiltedBox1));
    res = tiltedBox2->explode(testVector,1);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,tiltedBox2));
    res = tiltedBox3->explode(testVector,1);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,tiltedBox3));
    res = tiltedBox4->explode(testVector,1);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,tiltedBox4));
  }

  {
    testVector = Vector(0,0,0);
    res = unitBox->explode(testVector,2);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,unitBox));
    res = stretchedBox1->explode(testVector,2);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,stretchedBox1));
    res = stretchedBox2->explode(testVector,2);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,stretchedBox2));
    res = stretchedBox3->explode(testVector,2);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,stretchedBox3));
    res = stretchedBox4->explode(testVector,2);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,stretchedBox4));
    res = tiltedBox1->explode(testVector,2);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,tiltedBox1));
    res = tiltedBox2->explode(testVector,2);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,tiltedBox2));
    res = tiltedBox3->explode(testVector,2);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,tiltedBox3));
    res = tiltedBox4->explode(testVector,2);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,tiltedBox4));
  }

  {
    testVector = Vector(0.5,0.5,0.5);
    res = unitBox->explode(testVector,2);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,unitBox));
    res = stretchedBox1->explode(testVector,2);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,stretchedBox1));
    res = stretchedBox2->explode(testVector,2);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,stretchedBox2));
    res = stretchedBox3->explode(testVector,2);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,stretchedBox3));
    res = stretchedBox4->explode(testVector,2);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,stretchedBox4));
    res = tiltedBox1->explode(testVector,2);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,tiltedBox1));
    res = tiltedBox2->explode(testVector,2);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,tiltedBox2));
    res = tiltedBox3->explode(testVector,2);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,tiltedBox3));
    res = tiltedBox4->explode(testVector,2);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,tiltedBox4));
  }
}

void BoxUnittest::BoundaryMixedTest(){
  Vector testVector(0,0,0);
  VECTORSET(std::list) res;

  unitBox->setCondition(0,Box::Bounce);
  unitBox->setCondition(1,Box::Ignore);
  unitBox->setCondition(2,Box::Wrap);

  stretchedBox1->setCondition(0,Box::Bounce);
  stretchedBox1->setCondition(1,Box::Ignore);
  stretchedBox1->setCondition(2,Box::Wrap);

  stretchedBox2->setCondition(0,Box::Bounce);
  stretchedBox2->setCondition(1,Box::Ignore);
  stretchedBox2->setCondition(2,Box::Wrap);

  stretchedBox3->setCondition(0,Box::Bounce);
  stretchedBox3->setCondition(1,Box::Ignore);
  stretchedBox3->setCondition(2,Box::Wrap);

  stretchedBox4->setCondition(0,Box::Bounce);
  stretchedBox4->setCondition(1,Box::Ignore);
  stretchedBox4->setCondition(2,Box::Wrap);

  tiltedBox1->setCondition(0,Box::Bounce);
  tiltedBox1->setCondition(1,Box::Ignore);
  tiltedBox1->setCondition(2,Box::Wrap);

  tiltedBox2->setCondition(0,Box::Bounce);
  tiltedBox2->setCondition(1,Box::Ignore);
  tiltedBox2->setCondition(2,Box::Wrap);

  tiltedBox3->setCondition(0,Box::Bounce);
  tiltedBox3->setCondition(1,Box::Ignore);
  tiltedBox3->setCondition(2,Box::Wrap);

  tiltedBox4->setCondition(0,Box::Bounce);
  tiltedBox4->setCondition(1,Box::Ignore);
  tiltedBox4->setCondition(2,Box::Wrap);

  {
    testVector = Vector(0,0,0);
    res = unitBox->explode(testVector,1);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,unitBox));
    res = stretchedBox1->explode(testVector,1);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,stretchedBox1));
    res = stretchedBox2->explode(testVector,1);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,stretchedBox2));
    res = stretchedBox3->explode(testVector,1);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,stretchedBox3));
    res = stretchedBox4->explode(testVector,1);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,stretchedBox4));
    res = tiltedBox1->explode(testVector,1);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,tiltedBox1));
    res = tiltedBox2->explode(testVector,1);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,tiltedBox2));
    res = tiltedBox3->explode(testVector,1);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,tiltedBox3));
    res = tiltedBox4->explode(testVector,1);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,tiltedBox4));
  }

  {
    testVector = Vector(0.5,0.5,0.5);
    res = unitBox->explode(testVector,1);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,unitBox));
    res = stretchedBox1->explode(testVector,1);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,stretchedBox1));
    res = stretchedBox2->explode(testVector,1);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,stretchedBox2));
    res = stretchedBox3->explode(testVector,1);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,stretchedBox3));
    res = stretchedBox4->explode(testVector,1);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,stretchedBox4));
    res = tiltedBox1->explode(testVector,1);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,tiltedBox1));
    res = tiltedBox2->explode(testVector,1);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,tiltedBox2));
    res = tiltedBox3->explode(testVector,1);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,tiltedBox3));
    res = tiltedBox4->explode(testVector,1);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,tiltedBox4));
  }

  {
    testVector = Vector(0,0,0);
    res = unitBox->explode(testVector,2);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,unitBox));
    res = stretchedBox1->explode(testVector,2);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,stretchedBox1));
    res = stretchedBox2->explode(testVector,2);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,stretchedBox2));
    res = stretchedBox3->explode(testVector,2);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,stretchedBox3));
    res = stretchedBox4->explode(testVector,2);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,stretchedBox4));
    res = tiltedBox1->explode(testVector,2);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,tiltedBox1));
    res = tiltedBox2->explode(testVector,2);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,tiltedBox2));
    res = tiltedBox3->explode(testVector,2);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,tiltedBox3));
    res = tiltedBox4->explode(testVector,2);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,tiltedBox4));
  }

  {
    testVector = Vector(0.5,0.5,0.5);
    res = unitBox->explode(testVector,2);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,unitBox));
    res = stretchedBox1->explode(testVector,2);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,stretchedBox1));
    res = stretchedBox2->explode(testVector,2);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,stretchedBox2));
    res = stretchedBox3->explode(testVector,2);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,stretchedBox3));
    res = stretchedBox4->explode(testVector,2);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,stretchedBox4));
    res = tiltedBox1->explode(testVector,2);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,tiltedBox1));
    res = tiltedBox2->explode(testVector,2);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,tiltedBox2));
    res = tiltedBox3->explode(testVector,2);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,tiltedBox3));
    res = tiltedBox4->explode(testVector,2);
    CPPUNIT_ASSERT(testWrapExplode(res,testVector,tiltedBox4));
  }

}
