/*
 * unittest.cpp
 *
 *  Created on: Aug 17, 2009
 *      Author: heber
 */


using namespace std;

#include <cppunit/CompilerOutputter.h>
#include <cppunit/extensions/TestFactoryRegistry.h>
#include <cppunit/ui/text/TestRunner.h>

#include "defs.hpp"
#include "log.hpp"
#include "memoryusageobserver.hpp"
#include "vector.hpp"
#include "vector_ops.hpp"
#include "vectorunittest.hpp"
#include "Plane.hpp"
#include "Exceptions/LinearDependenceException.hpp"
#include "Matrix.hpp"

#ifdef HAVE_TESTRUNNER
#include "UnitTestMain.hpp"
#endif /*HAVE_TESTRUNNER*/

#include <iostream>

using namespace std;

/********************************************** Test classes **************************************/

// Registers the fixture into the 'registry'
CPPUNIT_TEST_SUITE_REGISTRATION( VectorTest );


void VectorTest::setUp()
{
  zero  = Vector(0.,0.,0.);
  unit = Vector(1.,0.,0.);
  otherunit = Vector(0.,1.,0.);
  notunit = Vector(0.,1.,1.);
  two = Vector(2.,1.,0.);
  three = Vector(1,2,3);
};


void VectorTest::tearDown()
{
  logger::purgeInstance();
  errorLogger::purgeInstance();
};

/** UnitTest for Constructors and Vector::IsZero() and Vector::IsOne().
 */
void VectorTest::UnityTest()
{
  // unity and zero tests
  CPPUNIT_ASSERT_EQUAL( true, zero.IsZero() );
  CPPUNIT_ASSERT_EQUAL( false, zero.IsOne() );
  CPPUNIT_ASSERT_EQUAL( false, unit.IsZero() );
  CPPUNIT_ASSERT_EQUAL( true, unit.IsOne() );
  CPPUNIT_ASSERT_EQUAL( false, notunit.IsOne() );
  CPPUNIT_ASSERT_EQUAL( true, otherunit.IsOne() );
  CPPUNIT_ASSERT_EQUAL( false, otherunit.IsZero() );
};

/** UnitTest for Vector::CopyVector(), Vector::AddVector, Vector::SubtractVector() and Vector::Scale()/
 */
void VectorTest::SimpleAlgebraTest()
{
  double factor;
  // copy vector
  fixture = Vector(2.,3.,4.);
  CPPUNIT_ASSERT_EQUAL( Vector(2.,3.,4.), fixture );
  // summation and scaling
  fixture = zero + unit;
  CPPUNIT_ASSERT_EQUAL( true, fixture.IsOne() );
  fixture = zero - unit;
  CPPUNIT_ASSERT_EQUAL( true, fixture.IsOne() );
  CPPUNIT_ASSERT_EQUAL( false, fixture.IsZero() );
  fixture = zero + zero;
  CPPUNIT_ASSERT_EQUAL( true, fixture.IsZero() );
  fixture = notunit - otherunit;
  CPPUNIT_ASSERT_EQUAL( true, fixture.IsOne() );
  fixture = unit - otherunit;
  CPPUNIT_ASSERT_EQUAL( false, fixture.IsOne() );
  fixture = notunit - unit - otherunit;
  CPPUNIT_ASSERT_EQUAL( false, fixture.IsZero() );
  fixture = 0.98 * unit;
  CPPUNIT_ASSERT_EQUAL( false, fixture.IsOne() );
  fixture = 1. * unit;
  CPPUNIT_ASSERT_EQUAL( true, fixture.IsOne() );
  factor = 0.98;
  fixture = factor * unit;
  CPPUNIT_ASSERT_EQUAL( false, fixture.IsOne() );
  factor = 1.;
  fixture = factor * unit;
  CPPUNIT_ASSERT_EQUAL( true, fixture.IsOne() );
};


/** UnitTest for operator versions of Vector::CopyVector(), Vector::AddVector, Vector::SubtractVector() and Vector::Scale().
 */
void VectorTest::OperatorAlgebraTest()
{
  // summation and scaling
  CPPUNIT_ASSERT_EQUAL( true, (zero+unit).IsOne() );
  CPPUNIT_ASSERT_EQUAL( true, (zero+unit).IsOne() );
  CPPUNIT_ASSERT_EQUAL( true, (zero-unit).IsOne() );
  CPPUNIT_ASSERT_EQUAL( false, (zero-unit).IsZero() );
  CPPUNIT_ASSERT_EQUAL( true, (zero+zero).IsZero() );
  CPPUNIT_ASSERT_EQUAL( true, (notunit-otherunit).IsOne() );
  CPPUNIT_ASSERT_EQUAL( false, (unit+otherunit).IsOne() );
  CPPUNIT_ASSERT_EQUAL( false, (notunit-unit-otherunit).IsZero() );
  CPPUNIT_ASSERT_EQUAL( false, (unit*0.98).IsOne() );
  CPPUNIT_ASSERT_EQUAL( true, (unit*1.).IsOne() );

  CPPUNIT_ASSERT_EQUAL( unit, (zero+unit) );
  CPPUNIT_ASSERT_EQUAL( Vector(0.,0.,1.), (notunit-otherunit) );
  CPPUNIT_ASSERT_EQUAL( Vector(-1, 0., 1.), (notunit-unit-otherunit) );
};

/** UnitTest for scalar products.
 */
void VectorTest::EuclidianScalarProductTest()
{
  CPPUNIT_ASSERT_EQUAL( 0., zero.ScalarProduct(zero) );
  CPPUNIT_ASSERT_EQUAL( 0., zero.ScalarProduct(unit) );
  CPPUNIT_ASSERT_EQUAL( 0., zero.ScalarProduct(otherunit) );
  CPPUNIT_ASSERT_EQUAL( 0., zero.ScalarProduct(notunit) );
  CPPUNIT_ASSERT_EQUAL( 1., unit.ScalarProduct(unit) );
  CPPUNIT_ASSERT_EQUAL( 0., otherunit.ScalarProduct(unit) );
  CPPUNIT_ASSERT_EQUAL( 0., otherunit.ScalarProduct(unit) );
  CPPUNIT_ASSERT_EQUAL( 1., otherunit.ScalarProduct(notunit) );
  CPPUNIT_ASSERT_EQUAL( 2., two.ScalarProduct(unit) );
  CPPUNIT_ASSERT_EQUAL( 1., two.ScalarProduct(otherunit) );
  CPPUNIT_ASSERT_EQUAL( 1., two.ScalarProduct(notunit) );
}

/** UnitTest for norms.
 */
void VectorTest::EuclidianNormTest()
{
  CPPUNIT_ASSERT_EQUAL( 0., zero.Norm() );
  CPPUNIT_ASSERT_EQUAL( 0., zero.NormSquared() );
  CPPUNIT_ASSERT_EQUAL( 1., unit.Norm() );
  CPPUNIT_ASSERT_EQUAL( 1., unit.NormSquared() );
  CPPUNIT_ASSERT_EQUAL( 1., otherunit.Norm() );
  CPPUNIT_ASSERT_EQUAL( 1., otherunit.NormSquared() );
  CPPUNIT_ASSERT_EQUAL( 2., notunit.NormSquared() );
  CPPUNIT_ASSERT_EQUAL( sqrt(2.), notunit.Norm() );
}

/** UnitTest for distances.
 */
void VectorTest::EuclidianDistancesTest()
{
  CPPUNIT_ASSERT_EQUAL( 1., zero.distance(unit) );
  CPPUNIT_ASSERT_EQUAL( sqrt(2.), otherunit.distance(unit) );
  CPPUNIT_ASSERT_EQUAL( sqrt(2.), zero.distance(notunit) );
  CPPUNIT_ASSERT_EQUAL( 1., otherunit.distance(notunit) );
  CPPUNIT_ASSERT_EQUAL( sqrt(5.), two.distance(notunit) );
}

/** UnitTest for angles.
 */
void VectorTest::EuclidianAnglesTest()
{
  CPPUNIT_ASSERT_EQUAL( M_PI, zero.Angle(unit) );
  CPPUNIT_ASSERT_EQUAL( 0., unit.Angle(unit) );
  CPPUNIT_ASSERT_EQUAL( true, fabs(M_PI/2. - otherunit.Angle(unit)) < MYEPSILON );
  CPPUNIT_ASSERT_EQUAL( true, fabs(M_PI/2. - unit.Angle(notunit)) < MYEPSILON );
  CPPUNIT_ASSERT_EQUAL( true, fabs(M_PI/4. - otherunit.Angle(notunit)) < MYEPSILON );
};

/** UnitTest for projections.
 */
void VectorTest::ProjectionTest()
{
  CPPUNIT_ASSERT_EQUAL( zero, zero.Projection(unit) );
  CPPUNIT_ASSERT_EQUAL( zero, otherunit.Projection(unit) );
  CPPUNIT_ASSERT_EQUAL( Vector(0.4,0.2,0.),  otherunit.Projection(two) );
  CPPUNIT_ASSERT_EQUAL( Vector(0.,1.,0.),  two.Projection(otherunit) );
};

/**
 * Unittest for operation with normals
 */
void VectorTest::NormalsTest(){
  Vector testVector;
  // the zero Vector should produce an error
  CPPUNIT_ASSERT(!testVector.GetOneNormalVector(zero));

  // first one-component system
  CPPUNIT_ASSERT(testVector.GetOneNormalVector(unit));
  CPPUNIT_ASSERT(testVector.ScalarProduct(unit) < MYEPSILON);

  // second one-component system
  CPPUNIT_ASSERT(testVector.GetOneNormalVector(otherunit));
  CPPUNIT_ASSERT(testVector.ScalarProduct(otherunit) < MYEPSILON);

  // first two-component system
  CPPUNIT_ASSERT(testVector.GetOneNormalVector(notunit));
  CPPUNIT_ASSERT(testVector.ScalarProduct(notunit) < MYEPSILON);

  // second two-component system
  CPPUNIT_ASSERT(testVector.GetOneNormalVector(two));
  CPPUNIT_ASSERT(testVector.ScalarProduct(two) < MYEPSILON);

  // three component system
  CPPUNIT_ASSERT(testVector.GetOneNormalVector(three));
  CPPUNIT_ASSERT(testVector.ScalarProduct(three) < MYEPSILON);
}


/**
 * UnitTest for Vector::IsInParallelepiped().
 */
void VectorTest::IsInParallelepipedTest()
{
  Matrix parallelepiped;
  parallelepiped.at(0,0) = 1;
  parallelepiped.at(1,0) = 0;
  parallelepiped.at(2,0) = 0;
  parallelepiped.at(0,1) = 0;
  parallelepiped.at(1,1) = 1;
  parallelepiped.at(2,1) = 0;
  parallelepiped.at(0,2) = 0;
  parallelepiped.at(1,2) = 0;
  parallelepiped.at(2,2) = 1;

  fixture = zero;
  CPPUNIT_ASSERT_EQUAL( false, fixture.IsInParallelepiped(Vector(2.,2.,2.), parallelepiped) );
  fixture = Vector(2.5,2.5,2.5);
  CPPUNIT_ASSERT_EQUAL( true, fixture.IsInParallelepiped(Vector(2.,2.,2.), parallelepiped) );
  fixture = Vector(1.,1.,1.);
  CPPUNIT_ASSERT_EQUAL( false, fixture.IsInParallelepiped(Vector(2.,2.,2.), parallelepiped) );
  fixture = Vector(3.5,3.5,3.5);
  CPPUNIT_ASSERT_EQUAL( false, fixture.IsInParallelepiped(Vector(2.,2.,2.), parallelepiped) );
  fixture = Vector(2.,2.,2.);
  CPPUNIT_ASSERT_EQUAL( true, fixture.IsInParallelepiped(Vector(2.,2.,2.), parallelepiped) );
  fixture = Vector(2.,3.,2.);
  CPPUNIT_ASSERT_EQUAL( true, fixture.IsInParallelepiped(Vector(2.,2.,2.), parallelepiped) );
  fixture = Vector(-2.,2.,-1.);
  CPPUNIT_ASSERT_EQUAL( false, fixture.IsInParallelepiped(Vector(2.,2.,2.), parallelepiped) );
}

