/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2010 University of Bonn. All rights reserved.
 * Please see the LICENSE file or "Copyright notice" in builder.cpp for details.
 */

/*
 * DipoleAngularCorrelationAction.cpp
 *
 *  Created on: Feb 11, 2011
 *      Author: heber
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "CodePatterns/MemDebug.hpp"

#include "Analysis/analysis_correlation.hpp"
#include "Tesselation/boundary.hpp"
#include "linkedcell.hpp"
#include "CodePatterns/Log.hpp"
#include "Element/element.hpp"
#include "molecule.hpp"
#include "Element/periodentafel.hpp"
#include "LinearAlgebra/Vector.hpp"
#include "World.hpp"
#include "WorldTime.hpp"

#include <iostream>
#include <map>
#include <string>

#include "Actions/AnalysisAction/DipoleAngularCorrelationAction.hpp"

using namespace MoleCuilder;

// and construct the stuff
#include "DipoleAngularCorrelationAction.def"
#include "Action_impl_pre.hpp"

/** =========== define the function ====================== */
Action::state_ptr AnalysisDipoleAngularCorrelationAction::performCall() {
  //int ranges[3] = {1, 1, 1};
  string type;

  // obtain information
  getParametersfromValueStorage();
  ASSERT(!params.periodic, "AnalysisDipoleAngularCorrelationAction() - periodic case not implemented.");

  // get selected atoms
  std::vector<atom*> atoms = World::getInstance().getSelectedAtoms();
  ASSERT(atoms.size() != 0, "AnalysisDipoleAngularCorrelationAction() - not atoms selected.");

  // get current time step
  const unsigned int oldtime = WorldTime::getTime();

  // obtain zero dipole orientation
  World::getInstance().setTime(0);
  std::map<atomId_t, Vector> ZeroVector = CalculateZeroAngularDipole(atoms);

  // go through each step of common trajectory of all atoms in set
  range<size_t> timesteps = getMaximumTrajectoryBounds(atoms);
  for (size_t step = 0; step < timesteps.first; ++step) {
    // calculate dipoles relative to zero orientation
    DipoleAngularCorrelationMap *correlationmap = NULL;
    correlationmap = DipoleAngularCorrelation(atoms, step, ZeroVector, DontResetTime);

    // prepare step string in filename
    std::stringstream stepstream;
    stepstream << std::setw(4) << std::setfill('0') << step;
    const std::string stepname(stepstream.str());

    // output correlation map
    ofstream output;
    std::string filename = params.outputname.string()+"."+stepname+".dat";
    output.open(filename.c_str());
    OutputCorrelationMap<DipoleAngularCorrelationMap>(&output, correlationmap, OutputDipoleAngularCorrelation_Header, OutputDipoleAngularCorrelation_Value);
    output.close();

    // bin map
    BinPairMap *binmap = BinData( correlationmap, params.BinWidth, params.BinStart, params.BinEnd );

    // free correlation map
    delete(correlationmap);

    // output binned map
    ofstream binoutput;
    std::string binfilename = params.binoutputname.string()+"."+stepname+".dat";
    binoutput.open(binfilename.c_str());
    OutputCorrelationMap<BinPairMap> ( &binoutput, binmap, OutputCorrelation_Header, OutputCorrelation_Value );
    binoutput.close();

    // free binned map
    delete(binmap);
  }

  // reset to old time step
  World::getInstance().setTime(oldtime);

  // exit
  return Action::success;
}

Action::state_ptr AnalysisDipoleAngularCorrelationAction::performUndo(Action::state_ptr _state) {
  return Action::success;
}

Action::state_ptr AnalysisDipoleAngularCorrelationAction::performRedo(Action::state_ptr _state){
  return Action::success;
}

bool AnalysisDipoleAngularCorrelationAction::canUndo() {
  return true;
}

bool AnalysisDipoleAngularCorrelationAction::shouldUndo() {
  return true;
}
/** =========== end of function ====================== */
