/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2011 University of Bonn. All rights reserved.
 * Please see the LICENSE file or "Copyright notice" in builder.cpp for details.
 */

/*
 * AtomObserverUnitTest.cpp
 *
 *  Created on: Dec 19, 2011
 *      Author: heber
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "AtomObserverUnitTest.hpp"

#include <cppunit/CompilerOutputter.h>
#include <cppunit/extensions/TestFactoryRegistry.h>
#include <cppunit/ui/text/TestRunner.h>
#include <iostream>

#include "CodePatterns/Log.hpp"

#include "World.hpp"
#include "Atom/atom.hpp"
#include "Atom/AtomObserver.hpp"
#include "Element/element.hpp"
#include "Element/periodentafel.hpp"

#include "stubs/ObserverStub.hpp"

#ifdef HAVE_TESTRUNNER
#include "UnitTestMain.hpp"
#endif /*HAVE_TESTRUNNER*/

/********************************************** Test classes **************************************/
// Registers the fixture into the 'registry'
CPPUNIT_TEST_SUITE_REGISTRATION( AtomObserverTest );

// set up and tear down
void AtomObserverTest::setUp(){
  World::getInstance();
  AtomObserver::getInstance();

  setVerbosity(3);

  // observer checks that we don't receive general updates ...
  observer = new UpdateCountObserver;
  // but only specific notifications in observer1 and observer2
  observer1 = new NotificationObserver(
      AtomObserver::getInstance().getChannel(AtomObservable::PositionChanged)
      );
  observer2 = new NotificationObserver(
      AtomObserver::getInstance().getChannel(AtomObservable::ElementChanged)
      );

  AtomObserver::getInstance().signOn(observer);
  AtomObserver::getInstance().signOn(observer1, AtomObservable::PositionChanged);
  AtomObserver::getInstance().signOn(observer2, AtomObservable::ElementChanged);

  for(int i=0;i<ATOM_COUNT;++i){
    atoms[i]= World::getInstance().createAtom();
    atomIds[i]= atoms[i]->getId();
  }
}

void AtomObserverTest::tearDown()
{
  AtomObserver::getInstance().signOff(observer);
  AtomObserver::getInstance().signOff(observer1, AtomObservable::PositionChanged);
  AtomObserver::getInstance().signOff(observer2, AtomObservable::ElementChanged);

  delete observer;
  delete observer1;
  delete observer2;

  AtomObserver::purgeInstance();
  World::purgeInstance();
}

/** Unit test on whether adding/removing atoms informs relay and onward.
 *
 */
void AtomObserverTest::AtomAddTest()
{
  // check for zero update
  CPPUNIT_ASSERT_EQUAL( (int)0, observer->updates );
  CPPUNIT_ASSERT_EQUAL( false, observer1->wasNotified );
  CPPUNIT_ASSERT_EQUAL( false, observer2->wasNotified );
}

/** Unit test on whether moving atoms informs relay and onward.
 *
 */
void AtomObserverTest::AtomMoveTest()
{
  // move an atom
  atoms[0]->setPosition(Vector(1.,0.,0.));

  // check for update
  CPPUNIT_ASSERT_EQUAL( true, observer1->wasNotified );
  CPPUNIT_ASSERT_EQUAL( false, observer2->wasNotified );
  CPPUNIT_ASSERT_EQUAL( (int)0, observer->updates );
}

/** Unit test on whether changing atom's element informs relay and onward.
 *
 */
void AtomObserverTest::AtomElementTest()
{
  atoms[0]->setType(1);

  // check for update
  CPPUNIT_ASSERT_EQUAL( false, observer1->wasNotified );
  CPPUNIT_ASSERT_EQUAL( true, observer2->wasNotified );
  CPPUNIT_ASSERT_EQUAL( (int)0, observer->updates );
}
