/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2012 University of Bonn. All rights reserved.
 * Copyright (C)  2013 Frederik Heber. All rights reserved.
 * 
 *
 *   This file is part of MoleCuilder.
 *
 *    MoleCuilder is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    MoleCuilder is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with MoleCuilder.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * SamplingGridPropertiesUnitTest.cpp
 *
 *  Created on: Jul 29, 2012
 *      Author: heber
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

using namespace std;

#include <cppunit/CompilerOutputter.h>
#include <cppunit/extensions/TestFactoryRegistry.h>
#include <cppunit/ui/text/TestRunner.h>

// include headers that implement a archive in simple text format
#include <boost/archive/text_oarchive.hpp>
#include <boost/archive/text_iarchive.hpp>

#include "SamplingGridPropertiesUnitTest.hpp"

#include "CodePatterns/Assert.hpp"

#ifdef HAVE_TESTRUNNER
#include "UnitTestMain.hpp"
#endif /*HAVE_TESTRUNNER*/

/********************************************** Test classes **************************************/

// Registers the fixture into the 'registry'
CPPUNIT_TEST_SUITE_REGISTRATION( SamplingGridPropertiesTest );


void SamplingGridPropertiesTest::setUp()
{
  // failing asserts should be thrown
  ASSERT_DO(Assert::Throw);

  // create the grid
  const double begin[3] = { 0., 0., 0. };
  const double end[3] = { 1., 1., 1. };
  props = new SamplingGridProperties(begin, end, 2);
}


void SamplingGridPropertiesTest::tearDown()
{
  delete props;
}

/** UnitTest for isCompatible()
 */
void SamplingGridPropertiesTest::isCompatible_Test()
{
  const double begin[3] = { 0., 0., 0. };
  const double otherbegin[3] = { 1., 0.1, -0.5 };
  const double end[3] = { 1., 1., 1. };
  const double otherend[3] = { 2., 2., 2. };

  // create other props
  SamplingGridProperties sameprops(begin, end, 2);
  SamplingGridProperties otherprops(otherbegin, end, 2);
  SamplingGridProperties anotherprops(begin, otherend, 2);
  SamplingGridProperties moreotherprops(begin, end, 4);

  // only checks for same level
  CPPUNIT_ASSERT( props->isCompatible(*props) );
  CPPUNIT_ASSERT( props->isCompatible(sameprops) );
  CPPUNIT_ASSERT( sameprops.isCompatible(*props) );
  CPPUNIT_ASSERT( props->isCompatible(otherprops) );
  CPPUNIT_ASSERT( otherprops.isCompatible(*props) );
  CPPUNIT_ASSERT( props->isCompatible(anotherprops) );
  CPPUNIT_ASSERT( anotherprops.isCompatible(*props) );
  CPPUNIT_ASSERT( !props->isCompatible(moreotherprops) );
  CPPUNIT_ASSERT( !moreotherprops.isCompatible(*props) );
}

/** UnitTest for operator==()
 */
void SamplingGridPropertiesTest::equality_Test()
{
  const double begin[3] = { 0., 0., 0. };
  const double otherbegin[3] = { 1., 0.1, -0.5 };
  const double end[3] = { 1., 1., 1. };
  const double otherend[3] = { 2., 2., 2. };

  // create other props
  SamplingGridProperties sameprops(begin, end, 2);
  SamplingGridProperties otherprops(otherbegin, end, 2);
  SamplingGridProperties anotherprops(begin, otherend, 2);
  SamplingGridProperties moreotherprops(begin, end, 4);

  // check for same level, begin, and end
  CPPUNIT_ASSERT( *props == *props );
  CPPUNIT_ASSERT( *props == sameprops );
  CPPUNIT_ASSERT( sameprops == *props);
  CPPUNIT_ASSERT( *props != otherprops);
  CPPUNIT_ASSERT( otherprops != *props);
  CPPUNIT_ASSERT( *props != anotherprops );
  CPPUNIT_ASSERT( anotherprops != *props );
  CPPUNIT_ASSERT( *props != moreotherprops );
  CPPUNIT_ASSERT( moreotherprops != *props );

  // check against ZeroInstance
  CPPUNIT_ASSERT( *props != ZeroInstance<SamplingGridProperties>() );
  CPPUNIT_ASSERT( sameprops != ZeroInstance<SamplingGridProperties>() );
  CPPUNIT_ASSERT( otherprops != ZeroInstance<SamplingGridProperties>() );
  CPPUNIT_ASSERT( anotherprops != ZeroInstance<SamplingGridProperties>() );
  CPPUNIT_ASSERT( moreotherprops != ZeroInstance<SamplingGridProperties>() );
}


/** UnitTest for serialization
 */
void SamplingGridPropertiesTest::serializeTest()
{
  // serialize
  std::stringstream outputstream;
  boost::archive::text_oarchive oa(outputstream);
  oa << props;

  // deserialize
  SamplingGridProperties *sameprops = NULL;
  std::stringstream returnstream(outputstream.str());
  boost::archive::text_iarchive ia(returnstream);
  ia >> sameprops;

  CPPUNIT_ASSERT( sameprops != NULL );
  CPPUNIT_ASSERT( *props == *sameprops );

  delete sameprops;
}
