/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2012 University of Bonn. All rights reserved.
 * Copyright (C)  2013 Frederik Heber. All rights reserved.
 * 
 *
 *   This file is part of MoleCuilder.
 *
 *    MoleCuilder is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    MoleCuilder is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with MoleCuilder.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * IndexedVectorsUnitTest.cpp
 *
 *  Created on: Jul 29, 2012
 *      Author: heber
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

using namespace std;

#include <cppunit/CompilerOutputter.h>
#include <cppunit/extensions/TestFactoryRegistry.h>
#include <cppunit/ui/text/TestRunner.h>

#include "IndexedVectorsUnitTest.hpp"

#include <cmath>
#include <limits>

#include <boost/assign.hpp>


#include "CodePatterns/Assert.hpp"

#ifdef HAVE_TESTRUNNER
#include "UnitTestMain.hpp"
#endif /*HAVE_TESTRUNNER*/

using namespace boost::assign;

/********************************************** Test classes **************************************/

// Registers the fixture into the 'registry'
CPPUNIT_TEST_SUITE_REGISTRATION( IndexedVectorsTest );


void IndexedVectorsTest::setUp()
{
  // failing asserts should be thrown
  ASSERT_DO(Assert::Throw);

  // create two vector_t
  ones += 1.,1.,1.;
  twos += 2.,2.,2.;
  threes += 3.,3.,3.;
  CPPUNIT_ASSERT_EQUAL( IndexedVectors::FixedSize, ones.size() );
  CPPUNIT_ASSERT_EQUAL( IndexedVectors::FixedSize, twos.size() );
  CPPUNIT_ASSERT_EQUAL( IndexedVectors::FixedSize, threes.size() );

  // create vectors

  IndexedVectors::vectors_t vectors;
  vectors.push_back(ones);
  vectors.push_back(twos);
  IndexedVectors::vectors_t othervectors;
  othervectors.push_back(threes);
  othervectors.push_back(threes);

  // create two indices
  IndexedVectors::indices_t indices;
  indices += 1,2;
  IndexedVectors::indices_t otherindices;
  otherindices += 1,3;

  // create indexed vectors
  ivectors = new IndexedVectors(indices, vectors);
  otherivectors = new IndexedVectors(otherindices, othervectors);
}


void IndexedVectorsTest::tearDown()
{
  delete ivectors;
  delete otherivectors;
}


/** UnitTest for cstor's
 */
void IndexedVectorsTest::Constructor_Test()
{
  // check whether -1 is dropped
  IndexedVectors::indices_t indices;
  indices += 1,-1,3;
  IndexedVectors::vectors_t vectors;
  vectors.push_back(ones);
  vectors.push_back(twos);
  vectors.push_back(threes);
  IndexedVectors testivectors(indices, vectors);

  CPPUNIT_ASSERT_EQUAL( (size_t)2, testivectors.vectors.size() );
  CPPUNIT_ASSERT( ones == testivectors.vectors[1] );
  CPPUNIT_ASSERT( threes == testivectors.vectors[3] );
  CPPUNIT_ASSERT( testivectors.vectors.find(-1) == testivectors.vectors.end() );
}

/** UnitTest for operator+=()
 */
void IndexedVectorsTest::operatorPlusEqual_Test()
{
  // safeguard initial sizes
  CPPUNIT_ASSERT_EQUAL( (size_t)2, ivectors->vectors.size() );
  CPPUNIT_ASSERT_EQUAL( (size_t)2, otherivectors->vectors.size() );

  // perform operation
  *ivectors += *otherivectors;

  // check new and ole sizes
  CPPUNIT_ASSERT_EQUAL( (size_t)3, ivectors->vectors.size() );
  CPPUNIT_ASSERT_EQUAL( (size_t)2, otherivectors->vectors.size() );

  // then check result
  IndexedVectors::vector_t result( IndexedVectors::nullvector );
  CPPUNIT_ASSERT_EQUAL( IndexedVectors::FixedSize, result.size() );
  for (size_t i=0; i<IndexedVectors::FixedSize; ++i)
    result[i] = ones[i] + threes[i];
  for (IndexedVectors::indexedvectors_t::const_iterator iter = ivectors->vectors.begin();
      iter != ivectors->vectors.end(); ++iter) {
    CPPUNIT_ASSERT_EQUAL( IndexedVectors::FixedSize, iter->second.size() );
  }
  CPPUNIT_ASSERT( result == ivectors->vectors[1] );
  CPPUNIT_ASSERT( twos == ivectors->vectors[2] );
  CPPUNIT_ASSERT( threes == ivectors->vectors[3] );
}

/** UnitTest for operator-=()
 */
void IndexedVectorsTest::operatorMinusEqual_Test()
{
  // safeguard initial sizes
  CPPUNIT_ASSERT_EQUAL( (size_t)2, ivectors->vectors.size() );
  CPPUNIT_ASSERT_EQUAL( (size_t)2, otherivectors->vectors.size() );

  // perform operation
  *ivectors -= *otherivectors;

  // check new and ole sizes
  CPPUNIT_ASSERT_EQUAL( (size_t)3, ivectors->vectors.size() );
  CPPUNIT_ASSERT_EQUAL( (size_t)2, otherivectors->vectors.size() );

  // then check result
  IndexedVectors::vector_t result( IndexedVectors::nullvector );
  IndexedVectors::vector_t thirdresult( IndexedVectors::nullvector );
  CPPUNIT_ASSERT_EQUAL( IndexedVectors::FixedSize, result.size() );
  for (size_t i=0; i<IndexedVectors::FixedSize; ++i) {
    result[i] = ones[i] - threes[i];
    thirdresult[i] = -threes[i];
  }
  for (IndexedVectors::indexedvectors_t::const_iterator iter = ivectors->vectors.begin();
      iter != ivectors->vectors.end(); ++iter) {
    CPPUNIT_ASSERT_EQUAL( IndexedVectors::FixedSize, iter->second.size() );
  }
  CPPUNIT_ASSERT( result == ivectors->vectors[1] );
  CPPUNIT_ASSERT( twos == ivectors->vectors[2] );
  CPPUNIT_ASSERT( thirdresult == ivectors->vectors[3] );
}


/** UnitTest for operator==()
 */
void IndexedVectorsTest::equality_Test()
{
  CPPUNIT_ASSERT( !(*ivectors == *otherivectors) );
  CPPUNIT_ASSERT( *ivectors != *otherivectors );

  // test against empty ivectors
  IndexedVectors emptyivectors;
  CPPUNIT_ASSERT( !(*ivectors == emptyivectors) );
  CPPUNIT_ASSERT( *ivectors != emptyivectors );

  // tests against themselves
  CPPUNIT_ASSERT( *ivectors == *ivectors );
  CPPUNIT_ASSERT( *otherivectors == *otherivectors );
  CPPUNIT_ASSERT( emptyivectors == emptyivectors );

  // check against ZeroInstance
  CPPUNIT_ASSERT( *ivectors != ZeroInstance<IndexedVectors>() );
  CPPUNIT_ASSERT( *otherivectors != ZeroInstance<IndexedVectors>() );
}
