/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2012 University of Bonn. All rights reserved.
 * Copyright (C)  2013 Frederik Heber. All rights reserved.
 * 
 *
 *   This file is part of MoleCuilder.
 *
 *    MoleCuilder is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    MoleCuilder is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with MoleCuilder.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * HistogramUnitTest.cpp
 *
 *  Created on: Jul 26, 2012
 *      Author: heber
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

using namespace std;

#include <cppunit/CompilerOutputter.h>
#include <cppunit/extensions/TestFactoryRegistry.h>
#include <cppunit/ui/text/TestRunner.h>

#include "Fragmentation/Summation/SetValues/Histogram.hpp"

#include "HistogramUnitTest.hpp"

#include <cmath>
#include <limits>

#include <boost/assign.hpp>

#include "CodePatterns/Assert.hpp"

#ifdef HAVE_TESTRUNNER
#include "UnitTestMain.hpp"
#endif /*HAVE_TESTRUNNER*/

using namespace boost::assign;

/********************************************** Test classes **************************************/

// Registers the fixture into the 'registry'
CPPUNIT_TEST_SUITE_REGISTRATION( HistogramTest );


void HistogramTest::setUp()
{
  // failing asserts should be thrown
  ASSERT_DO(Assert::Throw);

  histogram = NULL;
}


void HistogramTest::tearDown()
{
  delete histogram;
}

/** UnitTest for checking internal state after given samples
 */
void HistogramTest::internalState_Test()
{
  // generate non-empty histogramgram
  Histogram::samples_t sampled_values;
  sampled_values += 1.,2.,3.,4.;
  CPPUNIT_ASSERT_EQUAL( (size_t)4, sampled_values.size() );
  histogram = new Histogram(sampled_values, 1., 1.);
  CPPUNIT_ASSERT_EQUAL( (size_t)4+1, histogram->bins.size() );
  CPPUNIT_ASSERT( histogram->bins.end() != histogram->bins.begin() );
  CPPUNIT_ASSERT_EQUAL( 1., histogram->bins.begin()->second );
  CPPUNIT_ASSERT_EQUAL( 0., (--histogram->bins.end())->second );
}

/** UnitTest for getLowerEndBin()
 */
void HistogramTest::getLowerEndBin_Test()
{
  // generate non-empty histogramgram
  Histogram::samples_t sampled_values;
  sampled_values += 1.,2.,3.,4.;
  CPPUNIT_ASSERT_EQUAL( (size_t)4, sampled_values.size() );
  histogram = new Histogram(sampled_values, 1., 1.);

  // check values inside
  CPPUNIT_ASSERT( histogram->bins.end() != histogram->getLowerEndBin(1.) );
  CPPUNIT_ASSERT_EQUAL( 1., histogram->getLowerEndBin(1.)->first );
  CPPUNIT_ASSERT( histogram->bins.end() != histogram->getLowerEndBin(2.) );
  CPPUNIT_ASSERT_EQUAL( 2., histogram->getLowerEndBin(2.)->first );
  CPPUNIT_ASSERT( histogram->bins.end() != histogram->getLowerEndBin(3.) );
  CPPUNIT_ASSERT_EQUAL( 3., histogram->getLowerEndBin(3.)->first );
  CPPUNIT_ASSERT( histogram->bins.end() != histogram->getLowerEndBin(4.) );
  CPPUNIT_ASSERT_EQUAL( 4., histogram->getLowerEndBin(4.)->first );
  CPPUNIT_ASSERT( histogram->bins.end() != histogram->getLowerEndBin(1.5) );
  CPPUNIT_ASSERT_EQUAL( 1., histogram->getLowerEndBin(1.5)->first );
  CPPUNIT_ASSERT( histogram->bins.end() != histogram->getLowerEndBin(2.5) );
  CPPUNIT_ASSERT_EQUAL( 2., histogram->getLowerEndBin(2.5)->first );
  CPPUNIT_ASSERT( histogram->bins.end() != histogram->getLowerEndBin(3.5) );
  CPPUNIT_ASSERT_EQUAL( 3., histogram->getLowerEndBin(3.5)->first );
  CPPUNIT_ASSERT( histogram->bins.end() != histogram->getLowerEndBin(4.5) );
  CPPUNIT_ASSERT_EQUAL( 4., histogram->getLowerEndBin(4.5)->first );

  // check values outside
  CPPUNIT_ASSERT( histogram->bins.end() == histogram->getLowerEndBin(.9) );
  CPPUNIT_ASSERT( histogram->bins.end() == histogram->getLowerEndBin(5.01) );
  CPPUNIT_ASSERT( histogram->bins.end() == histogram->getLowerEndBin(5.) );
}

/** UnitTest for getHigherEndBin()
 */
void HistogramTest::getHigherEndBin_Test()
{
  // generate non-empty histogramgram
  Histogram::samples_t sampled_values;
  sampled_values += 1.,2.,3.,4.;
  CPPUNIT_ASSERT_EQUAL( (size_t)4, sampled_values.size() );
  histogram = new Histogram(sampled_values, 1., 1.);

  // check values inside
  CPPUNIT_ASSERT( histogram->bins.end() != histogram->getHigherEndBin(.5) );
  CPPUNIT_ASSERT_EQUAL( 1., histogram->getHigherEndBin(.5)->first );
  CPPUNIT_ASSERT( histogram->bins.end() != histogram->getHigherEndBin(1.) );
  CPPUNIT_ASSERT_EQUAL( 2., histogram->getHigherEndBin(1.)->first );
  CPPUNIT_ASSERT( histogram->bins.end() != histogram->getHigherEndBin(2.) );
  CPPUNIT_ASSERT_EQUAL( 3., histogram->getHigherEndBin(2.)->first );
  CPPUNIT_ASSERT( histogram->bins.end() != histogram->getHigherEndBin(3.) );
  CPPUNIT_ASSERT_EQUAL( 4., histogram->getHigherEndBin(3.)->first );
  CPPUNIT_ASSERT( histogram->bins.end() != histogram->getHigherEndBin(4.) );
  CPPUNIT_ASSERT_EQUAL( 5., histogram->getHigherEndBin(4.)->first );
  CPPUNIT_ASSERT( histogram->bins.end() == histogram->getHigherEndBin(5.) );
  CPPUNIT_ASSERT( histogram->bins.end() != histogram->getHigherEndBin(1.5) );
  CPPUNIT_ASSERT_EQUAL( 2., histogram->getHigherEndBin(1.5)->first );
  CPPUNIT_ASSERT( histogram->bins.end() != histogram->getHigherEndBin(2.5) );
  CPPUNIT_ASSERT_EQUAL( 3., histogram->getHigherEndBin(2.5)->first );
  CPPUNIT_ASSERT( histogram->bins.end() != histogram->getHigherEndBin(3.5) );
  CPPUNIT_ASSERT_EQUAL( 4., histogram->getHigherEndBin(3.5)->first );
  CPPUNIT_ASSERT( histogram->bins.end() != histogram->getHigherEndBin(4.5) );
  CPPUNIT_ASSERT_EQUAL( 5., histogram->getHigherEndBin(4.5)->first );
  CPPUNIT_ASSERT( histogram->bins.end() == histogram->getHigherEndBin(5.5) );

  // check values outside
  CPPUNIT_ASSERT( histogram->bins.end() != histogram->getHigherEndBin(-.1) );
  CPPUNIT_ASSERT_EQUAL( 1., histogram->getHigherEndBin(-.1)->first );
  CPPUNIT_ASSERT( histogram->bins.end() == histogram->getHigherEndBin(5.01) );
  CPPUNIT_ASSERT( histogram->bins.end() == histogram->getHigherEndBin(5.) );
}


/** UnitTest for getLowerEnd()
 */
void HistogramTest::getLowerEnd_Test()
{
  // create non-empty histogram and test
  Histogram::samples_t sampled_values;
  sampled_values += 1.,2.,3.,4.;
  histogram = new Histogram(sampled_values, 1., 1.);

  // go through each bin and check against getLowerEnd()
  for (Histogram::Bins_t::const_iterator iter = histogram->bins.begin();
      iter != histogram->bins.end(); ++iter)
    CPPUNIT_ASSERT_EQUAL( iter->first, histogram->getLowerEnd(iter->first) );

  CPPUNIT_ASSERT_EQUAL( 7., histogram->getLowerEnd(7.2) );
  CPPUNIT_ASSERT_EQUAL( -5., histogram->getLowerEnd(-4.5) );
  CPPUNIT_ASSERT_EQUAL( 4000., histogram->getLowerEnd(4000.1) );
  CPPUNIT_ASSERT_EQUAL( -4001., histogram->getLowerEnd(-4000.9) );
}

/** UnitTest for isEmpty()
 */
void HistogramTest::isEmpty_Test()
{
  histogram = new Histogram(1., 1.);
  // test on empty histogramgram
  CPPUNIT_ASSERT( histogram->isEmpty() );

  // create non-empty histogram and test
  Histogram::samples_t sampled_values;
  sampled_values += 1.,2.,3.,4.;
  delete histogram;
  histogram = new Histogram(sampled_values, 1., 1.);
  CPPUNIT_ASSERT( !histogram->isEmpty() );

}

/** UnitTest for area()
 */
void HistogramTest::areaTest()
{
  histogram = new Histogram(1., 1.);
  // test on empty histogramgram
  CPPUNIT_ASSERT_EQUAL( 0., histogram->area() );

  // create non-empty histogram and sum up
  Histogram::samples_t sampled_values;
  sampled_values += 1.,2.,3.,4., 4.;
  delete histogram;
  histogram = new Histogram(sampled_values, 1., 1.);
  CPPUNIT_ASSERT_EQUAL( 5., histogram->area() );
  Histogram::samples_t more_values;
  more_values += 1.75,2.5,3.;
  Histogram otherhistogram(more_values, 1.75, .625);
  CPPUNIT_ASSERT_EQUAL( 3., otherhistogram.area() );

}

/** UnitTest for superposeOtherHistogram()
 */
void HistogramTest::superposeOtherHistogram_Test()
{
  // create two histograms, one is larger
  Histogram::samples_t sampled_values;
  sampled_values += 1.,2.,3.,4.;
  histogram = new Histogram(sampled_values, 1., 1.);
  Histogram::samples_t more_values;
  more_values += 1.75,2.5,3.;
  Histogram otherhistogram(more_values, 1.75, .625);

  // check that size increases
  CPPUNIT_ASSERT_EQUAL( (size_t)3+1, otherhistogram.bins.size() );
  otherhistogram += *histogram;
  const size_t number = (size_t)((
      otherhistogram.getLowerEnd(4.+histogram->binwidth)
      + otherhistogram.binwidth
      - otherhistogram.getLowerEnd(1.))/otherhistogram.binwidth)+1;
  CPPUNIT_ASSERT_EQUAL( number, otherhistogram.bins.size() );
}

/** UnitTest for operator+=()
 */
void HistogramTest::operatorPlusEqual_Test()
{
  // create non-empty histograms and sum them
  Histogram::samples_t sampled_values;
  sampled_values += 1.,2.,3.,4.;
  histogram = new Histogram(sampled_values, 1., 1.);

  // add upon larger
  {
    Histogram::samples_t more_values;
    more_values += 0.5, 1.25, 2.5,3., 4.2;
    Histogram otherhistogram(more_values, 0.5, .625);
    const double areas = histogram->area() + otherhistogram.area();

    // check that sum is now sum of both
    otherhistogram += *histogram;
//    CPPUNIT_ASSERT_EQUAL( areas, otherhistogram.area() );
    CPPUNIT_ASSERT( fabs(areas - otherhistogram.area()) < std::numeric_limits<double>::epsilon()*1e+1 );
  }

  // add upon smaller
  {
    Histogram::samples_t more_values;
    more_values += 1.75,2.5,3.;
    Histogram otherhistogram(more_values, 1.75, .625);
    const double areas = histogram->area() + otherhistogram.area();

    // check that sum is now sum of both
    otherhistogram += *histogram;
//    CPPUNIT_ASSERT_EQUAL( areas, otherhistogram.area() );
    CPPUNIT_ASSERT( fabs(areas - otherhistogram.area()) < std::numeric_limits<double>::epsilon()*1e+1 );
  }
}

/** UnitTest for operator-=()
 */
void HistogramTest::operatorMinusEqual_Test()
{
  // create non-empty histograms and sum them
  Histogram::samples_t sampled_values;
  sampled_values += 1.,2.,3.,4.;
  histogram = new Histogram(sampled_values, 1., 1.);

  // subtract smaller
  {
    Histogram::samples_t more_values;
    more_values += 0.5, 1.25, 2.5,3., 4.2;
    Histogram otherhistogram(more_values, 0.5, .625);
    const double difference = otherhistogram.area() - histogram->area();

    // check that sum is now difference of both
    otherhistogram -= *histogram;
//    CPPUNIT_ASSERT_EQUAL( difference, otherhistogram.area() );
    CPPUNIT_ASSERT( fabs(difference - otherhistogram.area()) < std::numeric_limits<double>::epsilon()*1e+1 );
  }

  // subtract larger
  {
    Histogram::samples_t more_values;
    more_values += 1.75,2.5,3.;
    Histogram otherhistogram(more_values, 1.75, .625);
    const double difference = otherhistogram.area() - histogram->area();

    // check that sum is now difference of both
    otherhistogram -= *histogram;
//    CPPUNIT_ASSERT_EQUAL( difference, otherhistogram.area() );
    CPPUNIT_ASSERT( fabs(difference - otherhistogram.area()) < std::numeric_limits<double>::epsilon()*1e+1 );
  }
}

/** UnitTest for operator==()
 */
void HistogramTest::equality_Test()
{
  Histogram::samples_t sampled_values;
  sampled_values += 1.,2.,3.,4.;
  histogram = new Histogram(sampled_values, 1., 1.);

  // create different histogram
  Histogram::samples_t more_values;
  more_values += 0.5, 1.25, 2.5,3., 4.2;
  Histogram otherhistogram(more_values, 0.5, .625);

  CPPUNIT_ASSERT( !(*histogram == otherhistogram) );
  CPPUNIT_ASSERT( *histogram != otherhistogram );

  // test against empty histogram
  Histogram emptyhistogram;
  CPPUNIT_ASSERT( !(*histogram == emptyhistogram) );
  CPPUNIT_ASSERT( *histogram != emptyhistogram );

  // tests against themselves
  CPPUNIT_ASSERT( *histogram == *histogram );
  CPPUNIT_ASSERT( otherhistogram == otherhistogram );
  CPPUNIT_ASSERT( emptyhistogram == emptyhistogram );

  // check against ZeroInstance
  CPPUNIT_ASSERT( *histogram != ZeroInstance<Histogram>() );
  CPPUNIT_ASSERT( otherhistogram != ZeroInstance<Histogram>() );
}
