/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2013 Frederik Heber. All rights reserved.
 * 
 *
 *   This file is part of MoleCuilder.
 *
 *    MoleCuilder is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    MoleCuilder is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with MoleCuilder.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * EigenvaluesUnitTest.cpp
 *
 *  Created on: Sep 27, 2013
 *      Author: heber
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

using namespace std;

#include <cppunit/CompilerOutputter.h>
#include <cppunit/extensions/TestFactoryRegistry.h>
#include <cppunit/ui/text/TestRunner.h>

// include headers that implement a archive in simple text format
#include <boost/archive/text_oarchive.hpp>
#include <boost/archive/text_iarchive.hpp>

#include "EigenvaluesUnitTest.hpp"

#include <algorithm>
#include <limits>

#include <boost/assign.hpp>
#include <boost/foreach.hpp>

#include <sstream>

#include "CodePatterns/Assert.hpp"

#ifdef HAVE_TESTRUNNER
#include "UnitTestMain.hpp"
#endif /*HAVE_TESTRUNNER*/

using namespace boost::assign;

/********************************************** Test classes **************************************/

// Registers the fixture into the 'registry'
CPPUNIT_TEST_SUITE_REGISTRATION( EigenvaluesTest );


void EigenvaluesTest::setUp()
{
  // failing asserts should be thrown
  ASSERT_DO(Assert::Throw);

  samples += 1., 1.5, 2., 3., 4.;
  CPPUNIT_ASSERT_EQUAL( (size_t)5, samples.size() );

  ev = new Eigenvalues(samples);
}

void EigenvaluesTest::tearDown()
{
  delete ev;
}

/** UnitTest for operator==()
 */
void EigenvaluesTest::equality_Test()
{
  // create different ev
  Eigenvalues::samples_t othersamples;
  othersamples += 1., 2., 3.;
  Eigenvalues otherev(othersamples);

  CPPUNIT_ASSERT( !(*ev == otherev) );
  CPPUNIT_ASSERT( *ev != otherev );

  // test against empty ev
  Eigenvalues emptyev;
  CPPUNIT_ASSERT( !(*ev == emptyev) );
  CPPUNIT_ASSERT( *ev != emptyev );

  // tests against themselves
  CPPUNIT_ASSERT( *ev == *ev );
  CPPUNIT_ASSERT( otherev == otherev );
  CPPUNIT_ASSERT( emptyev == emptyev );

  // check against ZeroInstance
  CPPUNIT_ASSERT( *ev != ZeroInstance<Eigenvalues>() );
  CPPUNIT_ASSERT( otherev != ZeroInstance<Eigenvalues>() );
}

/** UnitTest for operator+=()
 */
void EigenvaluesTest::operatorPlusEqual_NonOverlapping_Test()
{
  {
    // create non-overlapping set
    Eigenvalues::samples_t othersamples;
    othersamples += 5., 6., 7.;
    Eigenvalues otherev(othersamples);
    const size_t otheraddedsize = otherev.addedsamples.size();
    const size_t othersubtractedsize = otherev.subtractedsamples.size();
    const size_t addedsize = ev->addedsamples.size();
    const size_t subtractedsize = ev->subtractedsamples.size();
    *ev += otherev;
    // otherev did not change
    CPPUNIT_ASSERT_EQUAL( otheraddedsize, otherev.addedsamples.size() );
    CPPUNIT_ASSERT_EQUAL( othersubtractedsize, otherev.subtractedsamples.size() );
    // ev grew
    CPPUNIT_ASSERT_EQUAL( addedsize+otheraddedsize, ev->addedsamples.size() );
    CPPUNIT_ASSERT_EQUAL( subtractedsize+othersubtractedsize, ev->subtractedsamples.size() );
  }
}

/** UnitTest for operator+=()
 */
void EigenvaluesTest::operatorPlusEqual_Test()
{
  {
    // create non-overlapping set
    Eigenvalues::samples_t othersamples;
    othersamples += 1., 3., 5.;
    Eigenvalues otherev(othersamples);
    const size_t otheraddedsize = otherev.addedsamples.size();
    const size_t othersubtractedsize = otherev.subtractedsamples.size();
    const size_t addedsize = ev->addedsamples.size();
    const size_t subtractedsize = ev->subtractedsamples.size();
    *ev += otherev;
    // otherev did not change
    CPPUNIT_ASSERT_EQUAL( otheraddedsize, otherev.addedsamples.size() );
    CPPUNIT_ASSERT_EQUAL( othersubtractedsize, otherev.subtractedsamples.size() );
    // ev grew
    CPPUNIT_ASSERT( addedsize+otheraddedsize != ev->addedsamples.size() );
    CPPUNIT_ASSERT( subtractedsize+othersubtractedsize == ev->subtractedsamples.size() );
    CPPUNIT_ASSERT_EQUAL( (size_t)6, ev->addedsamples.size() );
    CPPUNIT_ASSERT_EQUAL( (size_t)0, ev->subtractedsamples.size() );
  }
}

/** UnitTest for operator-=()
 */
void EigenvaluesTest::operatorMinusEqual_NonOverlapping_Test()
{
  {
    // create non-overlapping set
    Eigenvalues::samples_t othersamples;
    othersamples += 5., 6., 7.;
    Eigenvalues otherev(othersamples);
    const size_t otheraddedsize = otherev.addedsamples.size();
    const size_t othersubtractedsize = otherev.subtractedsamples.size();
    const size_t addedsize = ev->addedsamples.size();
    const size_t subtractedsize = ev->subtractedsamples.size();
    *ev -= otherev;
    // otherev did not change
    CPPUNIT_ASSERT_EQUAL( otheraddedsize, otherev.addedsamples.size() );
    CPPUNIT_ASSERT_EQUAL( othersubtractedsize, otherev.subtractedsamples.size() );
    // ev grew only in subtracted
    CPPUNIT_ASSERT_EQUAL( addedsize, ev->addedsamples.size() );
    CPPUNIT_ASSERT_EQUAL( otheraddedsize, ev->subtractedsamples.size() );
  }
}

/** UnitTest for operator-=()
 */
void EigenvaluesTest::operatorMinusEqual_Test()
{
  {
    // create non-overlapping set
    Eigenvalues::samples_t othersamples;
    othersamples += 1., 3., 5.;
    Eigenvalues otherev(othersamples);
    const size_t otheraddedsize = otherev.addedsamples.size();
    const size_t othersubtractedsize = otherev.subtractedsamples.size();
    const size_t addedsize = ev->addedsamples.size();
    const size_t subtractedsize = ev->subtractedsamples.size();
    *ev -= otherev;
    // otherev did not change
    CPPUNIT_ASSERT_EQUAL( otheraddedsize, otherev.addedsamples.size() );
    CPPUNIT_ASSERT_EQUAL( othersubtractedsize, otherev.subtractedsamples.size() );
    // ev grew only in subtracted
    CPPUNIT_ASSERT( addedsize == ev->addedsamples.size() );
    CPPUNIT_ASSERT( otheraddedsize == ev->subtractedsamples.size() );
  }
}
