/*
 * SamplingGridProperties.hpp
 *
 *  Created on: 25.07.2012
 *      Author: heber
 */

#ifndef SAMPLINGGRIDPROPERTIES_HPP_
#define SAMPLINGGRIDPROPERTIES_HPP_

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "boost/serialization/export.hpp"
#include "boost/serialization/array.hpp"

#include "Fragmentation/Summation/ZeroInstance.hpp"

/** This class stores a sample function on a three-dimensional grid.
 *
 */
class SamplingGridProperties {
public:
  /** Constructor for class SamplingGridProperties.
   *
   * \param _begin offset of grid
   * \param _end edge length per axis
   * \param _level number of gridpoints as \f$2^{\mathrm{level}}\f$ per unit(!) length
   */
  SamplingGridProperties(
      const double _begin[3],
      const double _end[3],
      const int _level);

  /** Copy constructor for class SamplingGridProperties.
   *
   * \param _props instance to copy from
   */
  SamplingGridProperties(const SamplingGridProperties &_props);

  /** Default constructor.
   */
  SamplingGridProperties();

  virtual ~SamplingGridProperties();

  /** Checks whether another instance is compatible with this one.
   *
   *  \note Compatibility implies only that both grids have same grid spacing.
   *
   * \param _props other properties to check against
   * \return true - are compatible, false - else
   */
  bool isCompatible(const SamplingGridProperties &_props) const
  {
    return level == _props.level;
  }

  /** Assignment operator.
   *
   * \param other other instance to assign ourselves to
   */
  SamplingGridProperties& operator=(const SamplingGridProperties& other);

  /** Equality operator for class SamplingGridProperties.
   *
   * \param _props other object to compare to
   */
  bool operator==(const SamplingGridProperties &_props) const;

  /** Inequality operator for class SamplingGridProperties.
   *
   * \param _props other object to compare to
   */
  bool operator!=(const SamplingGridProperties &_props) const
  {
    return (!(*this == _props));
  }

  /** Returns the volume of the domain for this sampled function.
   *
   * @return volume
   */
  const double getVolume() const;

  /** Returns the total number of gridpoints of the discrete mesh covering the volume.
   *
   * @return number of gridpoints sampled_values should have
   */
  const size_t getTotalGridPoints() const;

  /** Returns the number of grid points per axis.
   *
   * @return number of grid points per unit length
   */
  const size_t getGridPointsPerAxis() const;

  /** Returns the length of the domain for the given \a axis.
   *
   * \param axis axis for which to get step length
   * \return domain length for the given axis, i.e. end - begin
   */
  const double getTotalLengthPerAxis(const size_t axis) const;

  /** Returns the real step length from one discrete grid point to the next.
   *
   * \param axis axis for which to get step length
   * \return step length for the given axis, as domain length over getGridPointsPerAxis()
   */
  const double getDeltaPerAxis(const size_t axis) const;


public:
  //!> offset of grid
  double begin[3];
  //!> size of grid, i.e. edge length per axis of domain
  double end[3];
  //!> level, i.e. \f$2^{\mathrm{level}}\f$ grid points per axis per unit(!) length
  int level;

private:

  friend class boost::serialization::access;
  // serialization
  template <typename Archive>
  void serialize(Archive& ar, const unsigned int version)
  {
    int i;
    for (i=0; i<3; ++i)
      ar & begin[i];
    for (i=0; i<3; ++i)
      ar & end[i];
    ar & level;
  }

};

template<> SamplingGridProperties ZeroInstance<SamplingGridProperties>();

// we need to give this class a unique key for serialization
// its is only serialized through its base class FragmentJob
BOOST_CLASS_EXPORT_KEY(SamplingGridProperties)

// define inline functions
#include "SamplingGridProperties_inline.hpp"

#endif /* SAMPLINGGRIDPROPERTIES_HPP_ */
