/*
 * IndexedVectors.hpp
 *
 *  Created on: 29.07.2012
 *      Author: heber
 */

#ifndef INDEXEDVECTORS_HPP_
#define INDEXEDVECTORS_HPP_

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <iosfwd>
#include <map>
#include <vector>

class IndexedVectorsTest;

/** IndexedVectors represents a class that contains a a set of vectors,
 * each associated to a specific index. When adding or subtracting only
 * the ones are combined that have matching indices.
 *
 * This is needed for summing up force vectors per nuclei obtained from
 * fragment calculations.
 *
 */
class IndexedVectors
{
  //!> grant unit test access to private parts
  friend class IndexedVectorsTest;
public:
  //!> typedef for a single vector
  typedef std::vector<double> vector_t;
  //!> typedef for the index type
  typedef size_t index_t;
  //!> typedef for the indices matching the bunch of vectors
  typedef std::vector<vector_t> vectors_t;
  //!> typedef for the ordered indices matching the bunch of vectors
  typedef std::vector<index_t> indices_t;
  //!> typedef for a bunch of indexed vectors
  typedef std::map<index_t, vector_t> indexedvectors_t;

  enum SpecificIndices_t {
    DropIndex = -1
  };

  /** Default constructor for class IndexedVectors.
   *
   */
  IndexedVectors() {}

  /** Constructor for class IndexedVectors.
   *
   * We construct the internal map from \a _indices and \a _vectors. For
   * every index -1 contained in \a _indices the respective vector in
   * \a _vectors is \b not added but silently dropped.
   *
   * \param _indices index to each vector
   * \param _vectors vectors
   */
  IndexedVectors(const indices_t &_indices, const vectors_t &_vectors);

  /** Assignment operator.
   *
   * \note This is required to place IndexedVectors in STL containers.
   *
   * \param other other instance to assign this one to
   * \return ref to this instance
   */
  IndexedVectors& operator=(const IndexedVectors &other);

  /** Addition operator with another IndexedVector instance \a other.
   *
   * \param other other instance to sum onto this one.
   * \return ref to this instance
   */
  IndexedVectors& operator+=(const IndexedVectors &other)
  {
    superposeOtherIndexedVectors(other, +1.);
    return *this;
  }

  /** Subtraction operator with another IndexedVector instance \a other.
   *
   * \param other other instance to subtract from this one.
   * \return ref to this instance
   */
  IndexedVectors& operator-=(const IndexedVectors &other)
  {
    superposeOtherIndexedVectors(other, -1.);
    return *this;
  }

  /** Const getter to index vectors.
   *
   * \return const reference to indexed vectors
   */
  const indexedvectors_t& getVectors() const
  { return vectors; }

  /** Equality operator.
   *
   * @param other other instance to check against
   * @return true - both are equal, false - some IndexedVectors differ
   */
  bool operator==(const IndexedVectors& other) const;

  bool operator!=(const IndexedVectors& other) const
  {
    return (!(*this == other));
  }

private:
  /** Helper function that contains all the logic of how to superpose two
   * indexed vectors.
   *
   * Is called by IndexedVectors::operator+=() and IndexedVectors::operator-=()
   *
   * @param other other histogram
   * @param prefactor +1. is then addition, -1. is subtraction.
   */
  void superposeOtherIndexedVectors(const IndexedVectors &other, const double prefactor);

private:
  //!> internal map with all indexed vectors
  indexedvectors_t vectors;
  //!> fixed size of all vector_t
  static const size_t FixedSize;
  //!> static instance representing a null vector
  static const vector_t nullvector;

  //!> grant access to output operator
  friend std::ostream & operator<<(std::ostream &ost, const IndexedVectors &other);
};

/** Output operator for IndexedVector.
 *
 *  Prints a space-separated list of all members as "(index, vector)".
 *
 * \param ost output stream to print to
 * \param other instance to print
 * \return ref to ost for concatenation
 */
std::ostream & operator<<(std::ostream &ost, const IndexedVectors &other);

template<typename T> T ZeroInstance();
template<> IndexedVectors ZeroInstance<IndexedVectors>();


#endif /* INDEXEDVECTORS_HPP_ */
