/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2010-2012 University of Bonn. All rights reserved.
 * 
 *
 *   This file is part of MoleCuilder.
 *
 *    MoleCuilder is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    MoleCuilder is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with MoleCuilder.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * VectorSetUnitTest.cpp
 *
 *  Created on: Jan 06, 2012
 *      Author: heber
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

using namespace std;

#include <cppunit/CompilerOutputter.h>
#include <cppunit/extensions/TestFactoryRegistry.h>
#include <cppunit/ui/text/TestRunner.h>

#include "VectorSetUnitTest.hpp"

#include "RealSpaceMatrix.hpp"
#include "VectorSet.hpp"

#ifdef HAVE_TESTRUNNER
#include "UnitTestMain.hpp"
#endif /*HAVE_TESTRUNNER*/

#include <iostream>

/********************************************** Test classes **************************************/

// Registers the fixture into the 'registry'
CPPUNIT_TEST_SUITE_REGISTRATION( VectorSetTest );


void VectorSetTest::setUp()
{
  list.push_back(Vector(0.,0.,0.));
  list.push_back(Vector(1.,0.,0.));
  list.push_back(Vector(0.,1.,0.));
  list.push_back(Vector(0.,1.,1.));
  list.push_back(Vector(2.,1.,0.));
  list.push_back(Vector(1.,2.,3.));

  CPPUNIT_ASSERT_EQUAL( (size_t)6, list.size() );
}


void VectorSetTest::tearDown()
{
  list.clear();
}

/** UnitTest for Cstor(Set &)
 */
void VectorSetTest::copyConstructorTest()
{
  VECTORSET(std::vector) worklist(list);
  CPPUNIT_ASSERT_EQUAL( list.size(), worklist.size() );
  std::vector<Vector>::const_iterator iter = list.begin();
  VECTORSET(std::vector)::const_iterator workiter = worklist.begin();
  for (;(iter != list.end()) && (workiter != worklist.end()); ++iter, ++workiter)
    CPPUNIT_ASSERT( ((*iter)) == (*workiter) );
}


/** UnitTest for translate()
 */
void VectorSetTest::translateTest()
{
  VECTORSET(std::vector) worklist(list);
  Vector translater(1.,0.,0.);
  worklist.translate(translater);
  VECTORSET(std::vector)::const_iterator iter = list.begin();
  VECTORSET(std::vector)::const_iterator workiter = worklist.begin();
  for (;(iter != list.end()) && (workiter != worklist.end()); ++iter, ++workiter)
    CPPUNIT_ASSERT( ((*iter)+(translater)) == (*workiter) );
}

/** UnitTest for transform()
 */
void VectorSetTest::transformTest()
{
  VECTORSET(std::vector) worklist(list);
  RealSpaceMatrix M;
  M.setIdentity();
  M *= 2.;
  worklist.transform(M);
  VECTORSET(std::vector)::const_iterator iter = list.begin();
  VECTORSET(std::vector)::const_iterator workiter = worklist.begin();
  for (;(iter != list.end()) && (workiter != worklist.end()); ++iter, ++workiter)
    CPPUNIT_ASSERT( (2.*(*iter)) == (*workiter) );
}

/** UnitTest for minDistance()
 */
void VectorSetTest::minDistanceTest()
{
  VECTORSET(std::vector) worklist(list);
  {
    VECTORSET(std::vector) somelist;
    for (double x=1.;x<6.;++x)
      somelist.push_back(Vector(x,0.,0.));
    Vector center(0.,0.,0.);
    Vector mindist = somelist.minDistance(center);
    CPPUNIT_ASSERT_EQUAL (Vector(1., 0., 0.), mindist);
  }
  {
    Vector center(5.,0.,0.);
    Vector mindist = worklist.minDistance(center);
    CPPUNIT_ASSERT_EQUAL (Vector(2., 1., 0.)-center, mindist);
  }
  {
    for (VECTORSET(std::vector)::const_iterator iter = worklist.begin();
        iter != worklist.end(); ++iter) {
      Vector mindist = worklist.minDistance(*iter);
      CPPUNIT_ASSERT_EQUAL( Vector(0.,0.,0.), mindist );
    }
  }
}

/** UnitTest for minDistance()
 */
void VectorSetTest::minDistSquaredTest()
{
  VECTORSET(std::vector) worklist(list);
  {
    Vector center(5.,0.,0.);
    double mindist = worklist.minDistSquared(center);
    double check = Vector(2., 1., 0.).DistanceSquared(center);
    CPPUNIT_ASSERT_EQUAL( check, mindist );
  }
  {
    for (VECTORSET(std::vector)::const_iterator iter = worklist.begin();
        iter != worklist.end(); ++iter) {
      double mindist = worklist.minDistSquared(*iter);
      CPPUNIT_ASSERT_EQUAL( 0., mindist );
    }
  }
}
