/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2010 University of Bonn. All rights reserved.
 * Please see the LICENSE file or "Copyright notice" in builder.cpp for details.
 */

/*
 * cleanUp.cpp
 *
 *  Created on: Oct 28, 2011
 *      Author: heber
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <iostream>
#include <iomanip>

#include "CodePatterns/MemDebug.hpp"

#include "cleanUp.hpp"

#include "CodePatterns/Chronos.hpp"
#include "CodePatterns/errorlogger.hpp"
#include "CodePatterns/logger.hpp"

#include "Actions/ActionHistory.hpp"
#include "Actions/ActionRegistry.hpp"
#include "Actions/OptionRegistry.hpp"
#include "Actions/ValueStorage.hpp"

#include "RandomNumbers/RandomNumberDistributionFactory.hpp"
#include "RandomNumbers/RandomNumberEngineFactory.hpp"
#include "RandomNumbers/RandomNumberGeneratorFactory.hpp"

#include "Parser/ChangeTracker.hpp"
#include "Parser/FormatParserStorage.hpp"

#include "UIElements/CommandLineUI/CommandLineParser.hpp"
#include "UIElements/Menu/MenuDescription.hpp"
#include "UIElements/UIFactory.hpp"
#include "World.hpp"

/** In this function all dynamicly allocated member variables to static/global
 * variables are added to the ignore list of Memory/MemDebug.
 *
 * Use this to prevent their listing in the Memory::getState() at the end of the
 * program. Check with valgrind that truely no memory leak occurs!
 */
void AddStaticEntitiestoIgnoreList()
{
  // zeroVec and unitVec are global variables (on the stack) but vectorContent
  // within is situated on the heap and has to be ignored
  Memory::ignore(zeroVec.get());
  Memory::ignore(unitVec[0].get());
  Memory::ignore(unitVec[1].get());
  Memory::ignore(unitVec[2].get());
}

/** Cleans all singleton instances in an orderly fashion.
 * C++ does not guarantee any specific sequence of removal of single instances
 * which have static/global variables. Some singletons depend on others hence we
 * acertain a specific ordering here, which is is used via the atexit() hook.
 */
void cleanUp()
{
  printTimings();
  Chronos::purgeInstance();

  RandomNumberDistributionFactory::purgeInstance();
  RandomNumberEngineFactory::purgeInstance();
  RandomNumberGeneratorFactory::purgeInstance();
  FormatParserStorage::purgeInstance();
  ChangeTracker::purgeInstance();
  World::purgeInstance();
  MenuDescription::purgeInstance();
  UIFactory::purgeInstance();
  ValueStorage::purgeInstance();
  CommandLineParser::purgeInstance();
  MoleCuilder::ActionRegistry::purgeInstance();
  MoleCuilder::OptionRegistry::purgeInstance();
  MoleCuilder::ActionHistory::purgeInstance();
  // we have to remove these two static as otherwise their boost::shared_ptrs are still present
  MoleCuilder::Action::removeStaticStateEntities();
  // put some static variables' dynamic contents on the Memory::ignore map to avoid their
  // admonishing lateron
  AddStaticEntitiestoIgnoreList();
  logger::purgeInstance();
  errorLogger::purgeInstance();
#ifdef LOG_OBSERVER
  cout << observerLog().getLog();
#endif
  Memory::getState();
}

/** We give a list of all times per action and a total time.
 *
 */
void printTimings()
{
  const MoleCuilder::ActionRegistry &AR = MoleCuilder::ActionRegistry::getInstance();
  const Chronos &Chron = Chronos::getInstance();
  std::cout << "(Non-zero) Times used per Action [seconds]:" << std::endl;
  for (MoleCuilder::ActionRegistry::const_iterator iter = AR.getBeginIter(); iter != AR.getEndIter(); ++iter)
    if (Chron.getTime(iter->first) != 0.) { // dont give if action has not been used
      std::cout << "  " << setiosflags(ios::left) << setw(24) << setfill('.') << iter->first;
      std::cout << setiosflags(ios::left) << setprecision(6) << fixed << Chron.getTime(iter->first) << std::endl;
    }
  std::cout << "Total Time: " << Chron.SumUpTotalTime() << " seconds" << std::endl;
  std::cout << "Total Actions called: " << Chron.SumUpTotalFunctions() << std::endl;
}
